#include "checks_splitter.h"

#include <yandex/maps/wiki/validator/categories.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <maps/libs/log8/include/log8.h>


namespace maps {
namespace wiki {
namespace validation {

namespace {

const CheckIds BASE_CHECKS = {
    "base_only",
    "base_with_cardinality",
};

const std::string GROUP_BASE_CHECKS = "base_checks";
const std::string GROUP_OTHER_CHECKS = "other_checks";

const CategoryIds BIG_CATEGORIES = {
    validator::categories::ADDR::id(),
    validator::categories::ADDR_NM::id(),
    validator::categories::BLD::id(),
    validator::categories::HYDRO_FC_EL::id(),
    //validator::categories::HYDRO_LN_EL::id(),
    validator::categories::RD::id(),
    validator::categories::RD_NM::id(),
    validator::categories::RD_EL::id(),
    validator::categories::RD_JC::id(),
    validator::categories::VEGETATION::id(),
    validator::categories::VEGETATION_EL::id(),
};

const std::map<CategoryId,double> FIX_VALUES = {
    {validator::categories::RD_JC::id(), 5},
    {validator::categories::VEGETATION_EL::id(), 0.5},
    {validator::categories::ADDR::id(), 0.3},
    {validator::categories::ADDR_NM::id(), 0.3},
};

struct HeavyGroup
{
    std::string id;
    CategoryId category;
};

std::vector<HeavyGroup> HEAVY_GROUPS =
{
    {
        "rd_jc_checks",
        validator::categories::RD_JC::id()
    },
    {
        "addr_nm_checks",
        validator::categories::ADDR_NM::id()
    },
};

} // anonymous namespace

ChecksSplitter::ChecksSplitter(const validator::Validator& validator)
{
    CheckIds checks;
    for (const auto& module : validator.modules()) {
        for (const auto& checkId : module.checkIds()) {
            checks.insert(checkId);
            const auto& categories = module.categoriesByCheckId(checkId);
            checkToCategories_[checkId].insert(categories.begin(), categories.end());
        }
    }
    categoriesCount_ = categories(checks).size();
}

std::list<CheckIds> ChecksSplitter::split(CheckIds checks) const
{
    if (checks.empty()) {
        return {};
    }

    CheckIds baseChecks;
    for (const auto& checkId : BASE_CHECKS) {
        if (checks.count(checkId)) {
            baseChecks.emplace(checkId);
            checks.erase(checkId);
        }
    }

    std::map<std::string, CheckIds> groups;
    auto toGroup = [&](const std::string& grp, CheckIds grpChecks)
    {
        if (grp.empty() || grpChecks.empty()) {
            return;
        }

        groups[grp].insert(grpChecks.begin(), grpChecks.end());
        for (const auto& checkId : grpChecks) {
            checks.erase(checkId);
        }
    };

    for (const auto& heavyGroup : HEAVY_GROUPS) {
        CheckIds groupChecks;
        for (const auto& checkId : checks) {
            auto checkCategories = categories(checkId);
            if (checkCategories.count(heavyGroup.category)) {
                groupChecks.emplace(checkId);
            }
        }
        toGroup(heavyGroup.id, std::move(groupChecks));
    }

    while (!checks.empty()) {
        auto maxCheckId = findMaxCheckId(checks);
        auto bigGroupCategories = bigCategories(maxCheckId);
        if (bigGroupCategories.empty()) {
            break;
        }

        CheckIds grpChecks;
        for (const auto& checkId : checks) {
            auto checkCategories = categories(checkId);
            if (std::any_of(checkCategories.begin(), checkCategories.end(),
                    [&](const CategoryId& catId)
                        { return bigGroupCategories.count(catId) > 0; })) {
                    grpChecks.insert(checkId);
            }
        }
        auto grp = maxCheckId + ":" + common::join(bigGroupCategories, '.');
        toGroup(grp, std::move(grpChecks));
    }

    auto checksCopy = checks;
    for (const auto& checkId : checksCopy) {
        std::string grp;
        size_t grpMinValue = 0;

        auto checkCategories = categories(checkId);
        for (const auto& grp2checks : groups) {
            const auto& grpChecks = grp2checks.second;
            auto grpCategories = categories(grpChecks);
            if (!std::all_of(checkCategories.begin(), checkCategories.end(),
                    [&](const CategoryId& catId)
                        { return grpCategories.count(catId) > 0; })) {
                continue;
            }

            auto grpValue = checksValue(grpChecks);
            if (grp.empty() || (grpValue < grpMinValue)) {
                grp = grp2checks.first;
                grpMinValue = grpValue;
            }
        }
        toGroup(grp, {checkId});
    }

    if (!checks.empty()) {
        auto checksCategories = categories(checks);
        std::string grp;
        size_t grpMinValue = 0;
        for (const auto& grp2checks : groups) {
            const auto& grpChecks = grp2checks.second;
            auto grpCategories = categories(grpChecks);
            if (!std::any_of(grpCategories.begin(), grpCategories.end(),
                    [&](const CategoryId& catId)
                        { return checksCategories.count(catId) > 0; })) {
                continue;
            }
            auto grpValue = checksValue(grpChecks);
            if (grp.empty() || (grpValue < grpMinValue)) {
                grp = grp2checks.first;
                grpMinValue = grpValue;
            }
        }
        if (grp.empty()) {
            grp = GROUP_OTHER_CHECKS;
        }
        toGroup(grp, checks);
    }
    toGroup(GROUP_BASE_CHECKS, std::move(baseChecks));

    std::multimap<size_t, std::string, std::greater<size_t>> orderedGroups;
    for (const auto& group2checks : groups) {
        const auto& checks = group2checks.second;
        orderedGroups.emplace(checksValue(checks), group2checks.first);
    }

    std::list<CheckIds> result;
    for (const auto& pair : orderedGroups) {
        const auto& group = pair.second;
        const auto& checks = groups.at(group);
        INFO() << "=== GROUP: " << group << " " << pair.first << " ===";
        INFO() << " CATEGORIES: " << common::join(categories(checks), ',');
        INFO() << " CHECKS: " << common::join(checks, ',');
        result.push_back(checks);
    }
    return result;
}

size_t ChecksSplitter::checksValue(const CheckIds& checks) const
{
    size_t value = 0;
    for (const auto& category : categories(checks)) {
        double m = BIG_CATEGORIES.count(category) ? 1 : 0;
        auto it = FIX_VALUES.find(category);
        if (it != FIX_VALUES.end()) {
            m = it->second;
        }
        value += 1 + std::round(categoriesCount_ * m);
    }
    return value;
}

CheckId ChecksSplitter::findMaxCheckId(const CheckIds& checks) const
{
    CheckId maxCheckId;
    size_t maxValue = 0;
    for (const auto& checkId : checks) {
        auto value = checksValue({checkId});
        if (value > maxValue) {
            maxValue = value;
            maxCheckId = checkId;
        }
    }
    return maxCheckId;
}

CategoryIds ChecksSplitter::bigCategories(const CheckId& checkId) const
{
    CategoryIds result;
    for (const auto& category : categories(checkId)) {
        if (BIG_CATEGORIES.count(category)) {
            result.insert(category);
        }
    }
    return result;
}

const CategoryIds& ChecksSplitter::categories(const CheckId& checkId) const
{
    auto it = checkToCategories_.find(checkId);
    ASSERT(it != checkToCategories_.end());
    return it->second;
}

CategoryIds ChecksSplitter::categories(const CheckIds& checks) const
{
    CategoryIds result;
    for (const auto& checkId : checks) {
        const auto& cats = categories(checkId);
        result.insert(cats.begin(), cats.end());
    }
    return result;
}

} // namespace validation
} // namespace wiki
} // namespace maps
