#include "module.h"
#include "../transport-thread-common/common.h"
#include "../utils/misc.h"

#include <yandex/maps/wiki/validator/check.h>
#include <yandex/maps/wiki/validator/categories.h>

#include <maps/libs/geolib/include/convex_hull.h>

#include <boost/none.hpp>


namespace maps {
namespace wiki {
namespace validator {
namespace checks {
using categories::TRANSPORT_OPERATOR;
using categories::TRANSPORT_METRO_EXIT;
using categories::TRANSPORT_METRO_STATION;
using categories::TRANSPORT_METRO_THREAD;
using categories::TRANSPORT_METRO_LINE;
using categories::TRANSPORT_THREAD_STOP;
using categories::TRANSPORT_TRANSITION;
using categories::TRANSPORT_PASSAGEWAY;
using categories::FREQ_DT;
namespace {
const size_t METRO_MIN_FREQUENCY = 2;
const size_t METRO_MAX_FREQUENCY = 11;
const TSeconds METRO_MIN_TRANSITION_TIME = 30;
const TSeconds METRO_MAX_TRANSITION_TIME = 300;
const TSeconds METRO_MIN_PASSAGEWAY_TIME = 0;
const TSeconds METRO_MAX_PASSAGEWAY_TIME = 250;
const TSeconds METRO_MIN_TRAVEL_TIME = 60;
const TSeconds METRO_MAX_TRAVEL_TIME = 240;
const TSeconds METRO_MIN_WAIT_TIME = 10;
const TSeconds METRO_MAX_WAIT_TIME = 40;

const TransportMetroExit::TGeom&
exitGeomById(CheckContext* context, TId id)
{
    return context->objects<TRANSPORT_METRO_EXIT>().byId(id)->geom();
}

} // anonymous namespace

VALIDATOR_CHECK_PART(transport_metro_thread_time, metro_thread_stop,
    TRANSPORT_METRO_STATION, TRANSPORT_THREAD_STOP,
    TRANSPORT_METRO_THREAD, TRANSPORT_METRO_LINE,
    TRANSPORT_OPERATOR)
{
    context->objects<TRANSPORT_THREAD_STOP>().visit(
        [&](const TransportThreadStop* threadStop)
        {
            checkThreadStopsTiming<TRANSPORT_METRO_STATION,
                TRANSPORT_METRO_LINE,
                TRANSPORT_METRO_THREAD>(
                    context,
                    TravelTimeIsMandatory::Yes,  WaitTimeIsMandatory::Yes,
                    threadStop,
                    METRO_MIN_TRAVEL_TIME, METRO_MAX_TRAVEL_TIME,
                    METRO_MIN_WAIT_TIME, METRO_MAX_WAIT_TIME,
                    Severity::Error);
        });
}

VALIDATOR_CHECK_PART(transport_metro_thread_time, transition,
    TRANSPORT_METRO_STATION, TRANSPORT_TRANSITION, TRANSPORT_METRO_LINE,
    TRANSPORT_OPERATOR)
{
    context->objects<TRANSPORT_TRANSITION>().visit(
        [&](const TransportTransition* transition)
        {
            if (!isBoundToOperatorToValidate(context, transition)) {
                return;
            }
            if (!isDurationValid(transition->transitionTime(),
                                 METRO_MIN_TRANSITION_TIME,
                                 METRO_MAX_TRANSITION_TIME))
                context->report(
                    Severity::Error,
                    "invalid-transition-time",
                    bufferedConvexHull(
                        {
                            stationGeomById<TRANSPORT_METRO_STATION>(context, transition->stationA()),
                            stationGeomById<TRANSPORT_METRO_STATION>(context, transition->stationB())
                        },
                        utils::BUFFER_DISTANCE),
                    {transition->id()});
        });
}

VALIDATOR_CHECK_PART(transport_metro_thread_time, passageway,
    TRANSPORT_METRO_EXIT, TRANSPORT_METRO_STATION, TRANSPORT_PASSAGEWAY,
    TRANSPORT_METRO_LINE, TRANSPORT_OPERATOR)
{
    context->objects<TRANSPORT_PASSAGEWAY>().visit(
        [&](const TransportPassageway* passageway)
        {
            if (!isBoundToOperatorToValidate(context, passageway)) {
                return;
            }
            if (!isDurationValid(passageway->transitionTime(),
                                 METRO_MIN_PASSAGEWAY_TIME,
                                 METRO_MAX_PASSAGEWAY_TIME))
                context->report(
                    Severity::Error,
                    "invalid-transition-time",
                    bufferedConvexHull(
                        {
                          stationGeomById<TRANSPORT_METRO_STATION>(context, passageway->station()),
                          exitGeomById(context, passageway->exit())
                        },
                        utils::BUFFER_DISTANCE),
                    {passageway->id()});
        });
}

VALIDATOR_CHECK_PART(transport_metro_thread_time, thread_schedules, TRANSPORT_METRO_THREAD,
    TRANSPORT_METRO_LINE, TRANSPORT_OPERATOR, FREQ_DT)
{
    performSchedulesCheck<TRANSPORT_METRO_THREAD, TRANSPORT_METRO_LINE>(context, METRO_MIN_FREQUENCY, METRO_MAX_FREQUENCY);
}

} // namespace checks
} // namespace validator
} // namespace wiki
} // namespace maps
