#include "module.h"
#include "../transport-thread-common/common.h"
#include "../utils/misc.h"

#include <yandex/maps/wiki/validator/check.h>
#include <yandex/maps/wiki/validator/categories.h>

namespace maps {
namespace wiki {
namespace validator {
namespace checks {

using categories::TRANSPORT_OPERATOR;
using categories::TRANSPORT_METRO_THREAD;
using categories::TRANSPORT_METRO_STATION;
using categories::TRANSPORT_THREAD_STOP;
using categories::TRANSPORT_METRO_EL;
using categories::TRANSPORT_METRO_LINE;
using categories::RD_EL;

VALIDATOR_CHECK_PART( transport_metro_thread_relations, metro_thread_element_relations,
        TRANSPORT_METRO_THREAD, TRANSPORT_METRO_EL,
        TRANSPORT_METRO_LINE, TRANSPORT_OPERATOR)
{
    std::map<TId, std::set<TId>> lineToThreads;
    context->objects<TRANSPORT_METRO_THREAD>().visit([&](const TransportThread* thread) {
        lineToThreads[thread->line()].insert(thread->id());
        if (!isBoundToOperatorToValidate<TRANSPORT_METRO_LINE>(context, thread)) {
            return;
        }
        if (thread->elements().empty()) {
            context->critical(
                "transport-thread-without-elements",
                boost::none,
                { thread->id() });
        }
        if (thread->stops().size() <= 1) {
            context->critical(
                "transport-thread-too-few-stops",
                boost::none,
                { thread->id() });
        }
    });
    context->objects<TRANSPORT_METRO_EL>().visit(
        [&](const TransportElement* metroEl) {
        if (!isBoundToOperatorToValidate(context, metroEl)) {
            return;
        }
        for (auto lineId : metroEl->lines()) {
            if (!context->objects<TRANSPORT_METRO_LINE>()
                    .byId(lineId)->transportOperator()) {
                continue;
            }
            auto it = lineToThreads.find(lineId);
            bool foundThread = false;
            if (lineToThreads.end() != it) {
                for (auto threadId : metroEl->threads()) {
                    if (it->second.count(threadId)) {
                        foundThread = true;
                        break;
                    }
                }
            }
            if (!foundThread) {
                context->error(
                    "metro-element-not-assigned-to-threads-of-line",
                     utils::geomForReport(metroEl),
                    {metroEl->id(), lineId});
            }
        }
    });

    context->objects<TRANSPORT_METRO_THREAD>().visit(
        [&](const TransportThread* thread) {
        if (!isBoundToOperatorToValidate<TRANSPORT_METRO_LINE>(context, thread)) {
            return;
        }
        for (auto elementId : thread->elements()) {
            const auto& elementLines = context->objects<TRANSPORT_METRO_EL>().byId(elementId)->lines();
            if (std::find(elementLines.begin(), elementLines.end(), thread->line()) ==
                elementLines.end()) {
                context->error(
                    "metro-thread-element-not-assigned-to-the-same-line",
                     boost::none,
                    {elementId, thread->id()});
            }
        }
    });
}

VALIDATOR_CHECK_PART( transport_metro_thread_relations, metro_station_relations,
        TRANSPORT_METRO_THREAD, TRANSPORT_THREAD_STOP, TRANSPORT_METRO_STATION, TRANSPORT_METRO_LINE, TRANSPORT_OPERATOR)
{
    context->objects<TRANSPORT_METRO_THREAD>().visit([&](const TransportThread* thread) {
        if (!isBoundToOperatorToValidate<TRANSPORT_METRO_LINE>(context, thread)) {
            return;
        }
        std::set<TId> metStations;
        bool foundRepeats = false;
        std::vector<const TransportThreadStop*> stops;
        for (auto stopId : thread->stops()) {
            stops.push_back(context->objects<TRANSPORT_THREAD_STOP>().byId(stopId));
        }
        if (checkThreadStopSequenceHasFatalErrors(context, stops, thread->id())) {
            return;
        }
        for (const auto& stop : stops) {
            auto station = context->objects<TRANSPORT_METRO_STATION>().byId(stop->station());
            if (!metStations.insert(stop->station()).second) {
                if (foundRepeats || !thread->isCircular()) {
                    context->error(
                    "metro-thread-station-repeats",
                    station->geom(),
                    {thread->id(), station->id()});
                }
                foundRepeats = true;
            }
            if (station->line() != thread->line()) {
                context->error(
                    "metro-thread-and-station-not-same-line",
                    station->geom(),
                    {thread->id(), station->id()});
            }
        }
    });
}


} // namespace checks
} // namespace validator
} // namespace wiki
} // namespace maps
