#include "module.h"

#include <yandex/maps/wiki/validator/check.h>
#include <yandex/maps/wiki/validator/categories.h>
#include "../utils/misc.h"

#include <maps/libs/geolib/include/convex_hull.h>
#include <maps/libs/geolib/include/polygon.h>

#include <set>
#include <tuple>

namespace maps::wiki::validator::checks {

using categories::TRANSPORT_METRO_STATION;
using categories::TRANSPORT_PASSAGEWAY;
using categories::TRANSPORT_TRANSITION;
using categories::TRANSPORT_PASSAGEWAY_BOARDING;
using categories::TRANSPORT_TRANSITION_BOARDING;

namespace {

bool areStationsOnTheSameLine(CheckContext* context, TId stationId1, TId stationId2)
{
    auto viewStations = context->objects<TRANSPORT_METRO_STATION>();
    if (!viewStations.loaded(stationId1)) {
        return false;
    }
    if (!viewStations.loaded(stationId2)) {
        return false;
    }
    auto station1 = viewStations.byId(stationId1);
    auto station2 = viewStations.byId(stationId2);
    return station1->line() == station2->line();
}

geolib3::Polygon2 geomForReport(CheckContext* context, TId stationId1, TId stationId2)
{
    return geolib3::bufferedConvexHull({
        context->objects<TRANSPORT_METRO_STATION>().byId(stationId1)->geom(),
        context->objects<TRANSPORT_METRO_STATION>().byId(stationId2)->geom()
    }, utils::BUFFER_DISTANCE);
}

} // namespace

VALIDATOR_CHECK_PART(transport_metro_boarding, transition,
    TRANSPORT_TRANSITION, TRANSPORT_TRANSITION_BOARDING, TRANSPORT_METRO_STATION)
{
    std::set<std::tuple<TId, TId, TId>> transitionSequences;

    context->objects<TRANSPORT_TRANSITION_BOARDING>().visit(
        [&](const TransportTransitionBoarding* boarding) {
            auto viewTransitions = context->objects<TRANSPORT_TRANSITION>();
            auto transitionId = boarding->transition();
            if (!viewTransitions.loaded(transitionId)) {
                return;
            }
            auto transition = viewTransitions.byId(transitionId);

            auto startStationId =
                boarding->oneway() == TransportTransitionBoarding::Oneway::From
                ? transition->stationA()
                : transition->stationB();
            auto endStationId =
                boarding->oneway() == TransportTransitionBoarding::Oneway::From
                ? transition->stationB()
                : transition->stationA();

            if (!areStationsOnTheSameLine(context, boarding->previousStation(), startStationId)) {
                context->critical(
                    "wrong-boarding-prev-station",
                    geomForReport(context, boarding->previousStation(), startStationId),
                    { boarding->id() });
            }

            if (boarding->nextStation() && !areStationsOnTheSameLine(context, boarding->nextStation(), endStationId)) {
                context->critical(
                    "wrong-boarding-next-station",
                    geomForReport(context, boarding->nextStation(), endStationId),
                    { boarding->id() });
            }

            auto transitionSequence = std::make_tuple(
                boarding->previousStation(),
                transitionId,
                boarding->nextStation());
            if (transitionSequences.contains(transitionSequence)) {
                context->critical(
                    "duplicate-transition-boardings",
                    geomForReport(context, startStationId, endStationId),
                    { transitionId });
            }
            transitionSequences.insert(transitionSequence);
        });
}

VALIDATOR_CHECK_PART(transport_metro_boarding, passageway,
    TRANSPORT_PASSAGEWAY, TRANSPORT_PASSAGEWAY_BOARDING, TRANSPORT_METRO_STATION)
{
    std::set<std::pair<TId, TId>> passagewaySequences;

    context->objects<TRANSPORT_PASSAGEWAY_BOARDING>().visit(
        [&](const TransportPassagewayBoarding* boarding) {
            auto viewPassageways = context->objects<TRANSPORT_PASSAGEWAY>();
            auto passagewayId = boarding->passageway();
            if (!viewPassageways.loaded(passagewayId)) {
                return;
            }
            auto passageway = viewPassageways.byId(passagewayId);

            auto stationId = passageway->station();

            if (!common::isIn(passageway->oneway(),
                    {TransportPassageway::Oneway::To, TransportPassageway::Oneway::Both}))
            {
                context->critical(
                    "wrong-passageway-boarding-oneway",
                    geomForReport(context, boarding->previousStation(), stationId),
                    { boarding->id() });
            }

            if (!areStationsOnTheSameLine(context, boarding->previousStation(), stationId)) {
                context->critical(
                    "wrong-boarding-prev-station",
                    geomForReport(context, boarding->previousStation(), stationId),
                    { boarding->id() });
            }

            auto passagewaySequence = std::make_pair(boarding->previousStation(), passagewayId);
            if (passagewaySequences.contains(passagewaySequence)) {
                context->critical(
                    "duplicate-passageway-boardings",
                    geomForReport(context, boarding->previousStation(), stationId),
                    { passagewayId });
            }
            passagewaySequences.insert(passagewaySequence);
        });
}

} // maps::wiki::validator::checks
