#include "disjoint_sets.h"

#include <maps/libs/common/include/exception.h>

#include <numeric>
#include <unordered_map>

namespace maps {
namespace wiki {
namespace validator {
namespace utils {

DisjointSets::DisjointSets(size_t size)
    : numClusters_(size),
      ranks_(size, 1),
      parents_(size)
{ std::iota(parents_.begin(), parents_.end(), 0); }

size_t DisjointSets::find(size_t x) const
{
    REQUIRE(x < parents_.size(), "Index out of bounds");
    if (parents_[x] == x) {
        return x;
    }

    parents_[x] = find(parents_[x]);
    return parents_[x];
}

void DisjointSets::unionize(size_t x, size_t y)
{
    size_t xRoot = find(x);
    size_t yRoot = find(y);

    if (xRoot != yRoot) {
        numClusters_ -= 1;

        if (ranks_[xRoot] < ranks_[yRoot]) {
            parents_[xRoot] = yRoot;
        } else if (ranks_[xRoot] > ranks_[yRoot]) {
            parents_[yRoot] = xRoot;
        } else {
            parents_[yRoot] = xRoot;
            ranks_[xRoot] += 1;
        }
    }
}

std::vector<std::vector<size_t>> DisjointSets::clusters() const
{
    std::unordered_map<size_t, std::vector<size_t>> clustersMap;
    for (size_t i = 0; i < parents_.size(); ++i) {
        clustersMap[find(i)].push_back(i);
    }

    std::vector<std::vector<size_t>> result;
    result.reserve(clustersMap.size());

    for (auto& [_, cluster] : clustersMap) {
        result.emplace_back(std::move(cluster));
    }
    return result;
}

} // namespace utils
} // namespace validator
} // namespace wiki
} // namespace maps
