#include "face_builder.h"
#include <yandex/maps/wiki/validator/categories.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/polyline.h>

#include <unordered_map>
#include <unordered_set>
#include <stack>
#include <algorithm>
#include <iterator>

namespace maps {
namespace wiki {
namespace validator {
namespace utils {

namespace {

struct Neighbor {
    TId id;
    TId edgeId;
};

struct GraphNode {
    std::vector<Neighbor> neighbors;
    geolib3::Point2 geom;
    bool discovered;
    bool visited;
};

std::unordered_map<TId, GraphNode> buildGraph(
        const std::vector<TId>& edgeIds,
        const GeomObjectsView<Edge>& edgesView)
{
    std::unordered_map<TId, GraphNode> nodesById;
    for (TId edgeId : edgeIds) {
        const Edge* edge = edgesView.byId(edgeId);

        GraphNode& startNode = nodesById[edge->startJunction()];
        startNode.discovered = false;
        startNode.visited = false;
        startNode.geom = edge->geom().points().front();
        startNode.neighbors.push_back(Neighbor{edge->endJunction(), edgeId});

        GraphNode& endNode = nodesById[edge->endJunction()];
        endNode.discovered = false;
        endNode.visited = false;
        endNode.geom = edge->geom().points().back();
        endNode.neighbors.push_back(Neighbor{edge->startJunction(), edgeId});
    }

    return nodesById;
}

typedef std::function<void(TId, const GraphNode&)> NodeVisitor;

void depthFirstSearch(
        std::unordered_map<TId, GraphNode>& graph,
        NodeVisitor componentsVisitor,
        NodeVisitor nodeVisitor)
{
    for (const auto& startNode : graph) {
        if (startNode.second.visited) {
            continue;
        }

        componentsVisitor(startNode.first, startNode.second);

        TId curNodeId = startNode.first;
        std::stack<TId> dfsStack;
        dfsStack.push(curNodeId);

        while (!dfsStack.empty()) {
            curNodeId = dfsStack.top();
            GraphNode& curNode = graph[curNodeId];
            dfsStack.pop();
            curNode.discovered = true;

            nodeVisitor(curNodeId, curNode);

            for (const Neighbor& neighbor : curNode.neighbors) {
                GraphNode& otherNode = graph[neighbor.id];
                if (!otherNode.discovered) {
                    otherNode.discovered = true;
                    dfsStack.push(neighbor.id);
                }
            }

            curNode.visited = true;
        }
    }
}

} // namespace

FaceBuilder::FaceBuilder(
        const Face* face,
        const GeomObjectsView<Edge>& edgesView)
    : face_(face)
    , edgesView_(edgesView)
{
    if (face->edges().empty()) {
        return;
    }

    std::unordered_map<TId, GraphNode> nodesById =
        buildGraph(face->edges(), edgesView_);

    NodeVisitor componentsVisitor = [&](TId id, const GraphNode& node) {
        componentRepresentatives_.push_back(FaceNode{id, node.geom});
    };

    NodeVisitor nodesVisitor = [&](TId id, const GraphNode& node) {
            if (node.neighbors.size() == 1) {
                openBounds_.push_back(FaceNode{id, node.geom});
            } else if (node.neighbors.size() > 2) {
                selfIntersections_.push_back(FaceNode{id, node.geom});
            }
            segments_.push_back(Segment{FaceNode{id, node.geom}, 0});
    };

    depthFirstSearch(nodesById, componentsVisitor, nodesVisitor);

    if (!valid()) {
        return;
    }

    bool first = true;
    TId prevEdgeId;

    for (Segment& segment : segments_) {
        const std::vector<Neighbor>& neighbors =
            nodesById[segment.startNode.id].neighbors;
        if (first) {
            first = false;
            segment.edgeId = neighbors[0].id != segments_.back().startNode.id ?
                neighbors[0].edgeId : neighbors[1].edgeId;
        } else {
            segment.edgeId = neighbors[0].edgeId != prevEdgeId ?
                neighbors[0].edgeId : neighbors[1].edgeId;
        }

        prevEdgeId = segment.edgeId;
    }
}

bool FaceBuilder::valid() const
{
    return componentRepresentatives_.size() == 1
        && openBounds_.empty()
        && selfIntersections_.empty();
};

void FaceBuilder::checkValid() const
{
    REQUIRE(componentRepresentatives_.size() == 1, "Multiple components");
    REQUIRE(openBounds_.empty() && selfIntersections_.empty(),
        "Not forming ring");
}

const std::vector<FaceBuilder::Segment>& FaceBuilder::segments() const
{
    checkValid();
    return segments_;
}

geolib3::PointsVector FaceBuilder::Segment::points(
        const GeomObjectsView<Edge>& edgesView) const
{
    geolib3::PointsVector points = edgesView.byId(edgeId)->geom().points();

    if (points.front() != startNode.geom) {
        std::reverse(points.begin(), points.end());
    }

    return points;
}

geolib3::PointsVector FaceBuilder::points() const
{
    checkValid();

    geolib3::PointsVector points;
    for (const auto& segment : segments_) {
        geolib3::PointsVector segmentPoints = segment.points(edgesView_);
        std::copy(
                segmentPoints.begin(), std::prev(segmentPoints.end()),
                std::back_inserter(points));
    }
    points.push_back(points[0]);
    return points;
}

} // namespace utils
} // namespace validator
} // namespace wiki
} // namespace maps
