#pragma once

#ifndef VALIDATOR_CHECKS_UTILS_FACE_CHECKS_INL
#error "direct inclusion of face_checks-inl.h is not allowed, " \
    "please include face_checks.h instead"
#endif

#include "category_traits.h"
#include "face_builder.h"
#include "geom.h"
#include "misc.h"
#include "object_elements_within_aoi.h"

#include <yandex/maps/wiki/validator/check_context.h>
#include <yandex/maps/wiki/validator/objects/edge.h>
#include <yandex/maps/wiki/validator/objects/face.h>

#include <maps/libs/common/include/exception.h>

namespace maps {
namespace wiki {
namespace validator {
namespace utils {

namespace detail {

struct NodeCompare
{
    bool operator()(const FaceNode& node, const FaceNode& other) const
    { return node.id < other.id; }
};

typedef std::set<FaceNode, NodeCompare> NodesSet;

struct CompoundGeomPart
{
    const Face* face;
    geolib3::Polygon2 polygon;
    NodesSet nodes;
};

void checkFaceIntersections(
        CheckContext* context,
        const std::vector<CompoundGeomPart>& parts);

void checkFaceNesting(
        CheckContext* context,
        const std::vector<CompoundGeomPart>& parts,
        bool allPartsValid,
        Severity interiorFaceSeverity);

template<typename TObject>
Severity getInteriorFaceSeverity(const TObject*)
{
    return Severity::Error;
}

template<>
inline Severity getInteriorFaceSeverity<AdmUnit>(const AdmUnit* object)
{
    return object->levelKind() == AdmUnit::LevelKind::Country ? Severity::Fatal : Severity::Error;
}

} // namespace detail

template<typename FaceCategory>
void runFaceValidityCheck(CheckContext* context)
{
    typedef typename ElementCategory<FaceCategory>::type ElementCategory;
    context->checkLoaded<FaceCategory>();
    context->checkLoaded<ElementCategory>();

    auto viewElement = context->objects<ElementCategory>();

    context->objects<FaceCategory>().batchVisit([&](const Face* face) {
        if (!objectElementsWithinAoi<FaceCategory>(context, face)) {
            return;
        }

        FaceBuilder faceBuilder(face, viewElement);

        for (const auto& node : faceBuilder.openBounds()) {
            context->fatal(
                    "face-open-bound",
                    node.geom,
                    {face->id(), node.id});
        }

        for (const auto& node : faceBuilder.selfIntersections()) {
            context->fatal("face-self-intersection",
                    node.geom,
                    {face->id(), node.id});
        }

        if (faceBuilder.componentRepresentatives().size() > 1) {
            std::vector<TId> errorIds;
            errorIds.push_back(face->id());
            for (const auto& node : faceBuilder.componentRepresentatives()) {
                errorIds.push_back(node.id);
            }
            context->fatal("face-multiple-components", boost::none, errorIds);
        }
    }, detail::VISIT_BATCH_SIZE);
}

template<class FaceCategory>
void runFaceDimensionsCheck(
        CheckContext* context,
        Severity severity,
        double minArea,
        double maxPerimeterToAreaRatio)
{
    typedef typename ElementCategory<FaceCategory>::type ElementCategory;
    context->checkLoaded<ElementCategory>();

    auto viewElement = context->objects<ElementCategory>();

    context->objects<FaceCategory>().batchVisit([&](const Face* face) {
        if (!objectElementsWithinAoi<FaceCategory>(context, face)) {
            return;
        }

        FaceBuilder faceBuilder(face, viewElement);

        auto geom = facePolygon(faceBuilder);
        if (geom) {
            double mercatorLengthRatio = mercatorDistanceRatio(*geom);

            if (geom->area() * mercatorLengthRatio * mercatorLengthRatio
                    < minArea) {
                context->report(
                        severity,
                        "face-too-small",
                        geom->boundingBox().center(),
                        {face->id()});
            }

            double perimeterAreaRatio =
                geom->perimeter() / (geom->area() * mercatorLengthRatio);
            if (perimeterAreaRatio > maxPerimeterToAreaRatio) {
                context->report(
                        severity,
                        "face-perimeter-area-ratio-too-big",
                        geom->boundingBox().center(),
                        {face->id()});
            }
        }
    }, detail::VISIT_BATCH_SIZE);
}

template<class CompoundObjectCategory>
void runFaceIntersectionsNestingCheck(CheckContext* context, size_t batchSize)
{
    typedef typename FaceCategory<CompoundObjectCategory>::type FaceCategory;
    typedef typename ElementCategory<FaceCategory>::type ElementCategory;
    context->checkLoaded<CompoundObjectCategory>();
    context->checkLoaded<FaceCategory>();
    context->checkLoaded<ElementCategory>();

    auto viewFace = context->objects<FaceCategory>();
    auto viewElement = context->objects<ElementCategory>();

    context->objects<CompoundObjectCategory>().batchVisit(
            [&](const typename CompoundObjectCategory::TObject* object) {
        if (!objectFacesWithinAoi<CompoundObjectCategory>(context, object)) {
            return;
        }

        std::vector<detail::CompoundGeomPart> parts;
        bool allPartsValid = true;

        for (TId faceId : object->faces()) {
            const Face* face = viewFace.byId(faceId);
            FaceBuilder faceBuilder(face, viewElement);

            auto geom = facePolygon(faceBuilder);
            if (geom) {
                detail::NodesSet nodes;
                for (const auto& segment : faceBuilder.segments()) {
                    nodes.insert(segment.startNode);
                }

                parts.push_back(detail::CompoundGeomPart{face, *geom, nodes});
            } else {
                allPartsValid = false;
            }
        }

        detail::checkFaceIntersections(context, parts);
        detail::checkFaceNesting(context, parts, allPartsValid,
            detail::getInteriorFaceSeverity(object));
    }, batchSize);
}

inline boost::optional<geolib3::Polygon2> facePolygon(
        const FaceBuilder& faceBuilder)
{
    if (faceBuilder.valid()) {
        auto points = faceBuilder.points();
        if (points.size() >= 4) {
            try {
                return geolib3::Polygon2(points);
            } catch (const maps::Exception&) {
                // validation failed
            }
        }
    }

    return boost::none;
}

} // namespace utils
} // namespace validator
} // namespace wiki
} // namespace maps
