#include "face_checks.h"
#include <yandex/maps/wiki/validator/check_context.h>

#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/spatial_relation.h>

#include <algorithm>
#include <set>

namespace maps {
namespace wiki {
namespace validator {
namespace utils {
namespace detail {

namespace {

namespace gl = maps::geolib3;

void checkFaceIntersections(
        const CompoundGeomPart& part,
        const CompoundGeomPart& other,
        CheckContext* context)
{
    std::vector<FaceNode> intersections;
    std::set_intersection(
            part.nodes.begin(), part.nodes.end(),
            other.nodes.begin(), other.nodes.end(),
            std::back_inserter(intersections), NodeCompare());

    if (intersections.size() == 1 &&
            part.face->edges() != other.face->edges()) {
        std::vector<TId> idsForReport{
            part.face->parent(),
            part.face->id(), other.face->id(),
            intersections.front().id };
        context->warning(
            "touching-faces",
            intersections.front().geom, idsForReport);
    } else {
        for (const FaceNode& intersection : intersections) {
            std::vector<TId> idsForReport{
                part.face->parent(),
                part.face->id(), other.face->id(),
                intersection.id };
            context->fatal(
                "intersecting-faces",
                intersection.geom, idsForReport);
        }
    }
}

} // namespace

void checkFaceIntersections(
        CheckContext* context,
        const std::vector<CompoundGeomPart>& parts)
{
    for (size_t i = 0; i < parts.size(); ++i) {
        for (size_t j = i + 1; j < parts.size(); ++j) {
            checkFaceIntersections(parts[i], parts[j], context);
        }
    }
}

void checkFaceNesting(
        CheckContext* context,
        const std::vector<CompoundGeomPart>& parts,
        bool allPartsValid,
        Severity interiorFaceSeverity)
{
    if (parts.empty()) {
        return;
    }

    struct PartInfo
    {
        std::vector<const Face*> nestedFaces;
        bool withinOtherFaces = false;
    };
    std::vector<PartInfo> partInfos(parts.size());

    for (size_t i = 0; i < parts.size(); ++i) {
        for (size_t j = 0; j < parts.size(); ++j) {
            if (i == j) {
                continue;
            }

            const auto& part = parts[i];
            const auto& other = parts[j];

            if (gl::spatialRelation(
                    other.polygon, part.polygon, gl::Within)) {
                partInfos[i].nestedFaces.push_back(other.face);
                partInfos[j].withinOtherFaces = true;
            }
        }
    }

    for (size_t i = 0; i < parts.size(); ++i) {
        const auto& part = parts[i];
        const auto& info = partInfos[i];

        gl::Point2 geomForReport = part.polygon.boundingBox().center();

        if (allPartsValid && !info.withinOtherFaces && part.face->isInterior()) {
            context->fatal(
                    "interior-face-not-within-face",
                    geomForReport, {part.face->parent(), part.face->id()});
        }

        for (const Face* otherFace : info.nestedFaces) {
            if (part.face->isInterior()) {
                context->report(
                        interiorFaceSeverity,
                        "face-within-interior-face",
                        geomForReport,
                        {part.face->parent(), part.face->id(), otherFace->id()});
            }

            if (!part.face->isInterior() && !otherFace->isInterior()) {
                context->fatal(
                        "exterior-face-within-face",
                        geomForReport,
                        {part.face->parent(), part.face->id(), otherFace->id()});
            }
        }
    }
}

} // namespace detail
} // namespace utils
} // namespace validator
} // namespace wiki
} // namespace maps
