#include "find_gaps_representatives.h"
#include "disjoint_sets.h"
#include "geom.h"

#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/distance.h>

#include <limits>
#include <set>

namespace maps {
namespace wiki {
namespace validator {
namespace utils {

template<class TGeom>
std::vector<std::pair<size_t, size_t>> findGapsRepresentatives(
       const std::vector<TGeom>& geoms,
       double minGapDistance)
{
    std::vector<std::pair<size_t, size_t>> result;

    if (geoms.empty()) {
        return result;
    }

    // Compute distances and clustering by distance
    std::vector<std::vector<double>> distances(
        geoms.size(),
        std::vector<double>(geoms.size()));
    utils::DisjointSets clustering(geoms.size());
    for (size_t i = 0; i < geoms.size(); ++i) {
        for (size_t j = i + 1; j < geoms.size(); ++j) {
            double mercatorDistance = geolib3::distance(geoms[i], geoms[j]);
            double mercatorDistanceRatio =
                utils::mercatorDistanceRatio(geolib3::expand(
                    geoms[i].boundingBox(),
                    geoms[j].boundingBox()));

            distances[i][j] = mercatorDistance * mercatorDistanceRatio;
            distances[j][i] = distances[i][j];

            if (distances[i][j] < minGapDistance) {
                clustering.unionize(i, j);
            }
        }
    }

    // Find closest cluster pairs and report them
    auto clusters = clustering.clusters();
    std::set<std::pair<size_t, size_t>> reported;
    for (size_t i = 0; i < clusters.size(); ++i) {
        double minDistance = std::numeric_limits<double>::infinity();
        std::pair<size_t, size_t> minDistanceElements;
        size_t minDistanceJ = i;
        for (size_t j = 0; j < clusters.size(); ++j) {
            if (i == j) {
                continue;
            }

            for (size_t iElement : clusters[i]) {
                for (size_t jElement : clusters[j]) {
                    double distance = distances[iElement][jElement];
                    if (minDistance > distance) {
                        minDistance = distance;
                        minDistanceElements.first = iElement;
                        minDistanceElements.second = jElement;
                        minDistanceJ = j;
                    }
                }
            }
        }

        if (minDistance < std::numeric_limits<double>::infinity() &&
                reported.insert(std::make_pair(
                    std::min(i, minDistanceJ),
                    std::max(i, minDistanceJ))).second) {
            result.push_back(minDistanceElements);
        }
    }

    return result;
}


template std::vector<std::pair<size_t, size_t>>
findGapsRepresentatives<geolib3::Polyline2>(
    const std::vector<geolib3::Polyline2>& geoms,
    double minGapDistance);

template std::vector<std::pair<size_t, size_t>>
findGapsRepresentatives<geolib3::Polygon2>(
    const std::vector<geolib3::Polygon2>& geoms,
    double minGapDistance);

} // namespace utils
} // namespace validator
} // namespace wiki
} // namespace maps
