#include "geom.h"

#include <maps/libs/geolib/include/direction.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/conversion_geos.h>
#include <maps/libs/common/include/exception.h>

#include <geos/geom/MultiLineString.h>
#include <geos/geom/Polygon.h>

#include <memory>

namespace maps {
namespace wiki {
namespace validator {
namespace utils {

namespace gl = maps::geolib3;

gl::BoundingBox boundingBox(const gl::PointsVector& points)
{
    REQUIRE(!points.empty(), "Cannot find bounding box of an empty PointsVector");

    double maxX = points[0].x();
    double maxY = points[0].y();
    double minX = maxX;
    double minY = maxY;

    for (const auto& point : points) {
        maxX = std::max(point.x(), maxX);
        minX = std::min(point.x(), minX);
        maxY = std::max(point.y(), maxY);
        minY = std::min(point.y(), minY);
    }
    return gl::BoundingBox(gl::Point2(minX, minY), gl::Point2(maxX, maxY));
}

// Taken from libs/geolib/spatial_relation.cpp, algorithm taken from
// http://alienryderflex.com/polygon/
bool isPointInsideRing(const gl::Point2& point, const gl::PointsVector& ring)
{
    if (ring.empty()) {
        return false;
    }

    REQUIRE(ring.front() == ring.back(), "Not a ring");

    bool result = false;
    gl::Point2 prev = ring[0];
    for (size_t i = 1; i < ring.size(); ++i) {
        const gl::Point2& next = ring[i];
        if ((gl::sign(next.y() - point.y()) == -1
                && gl::sign(point.y() - prev.y()) <= 0)
                || (gl::sign(prev.y() - point.y()) == -1
                    && gl::sign(point.y() - next.y()) <= 0)) {
            double intersection_x = (point.y() - next.y()) / (prev.y() - next.y())
                * (prev.x() - next.x()) + next.x();
            if (gl::sign(intersection_x - point.x()) == -1) {
                result = !result;
            }
        }
        prev = next;
    }
    return result;
}

double ccwAngle(const gl::Direction2& from, const gl::Direction2& to)
{
    double result = gl::signedAngle(from, to);
    if (result < 0.0) {
        result += 2.0 * M_PI;
    }
    return result;
}

double intersectionLength(
        const geolib3::PolylinesVector& polylines,
        const geolib3::Polygon2& polygon)
{
    auto polylinesGeosGeom = geolib3::internal::geolib2geosGeometry(polylines);
    auto polygonGeosGeom = geolib3::internal::geolib2geosGeometry(polygon);
    std::unique_ptr<geos::geom::Geometry> intersection(
        polylinesGeosGeom->intersection(polygonGeosGeom.get()));
    return intersection->getLength();
}

} // namespace utils
} // namespace validator
} // namespace wiki
} // namespace maps
