#pragma once

#ifndef VALIDATOR_CHECKS_UTILS_NAMES_INL
#error "Direct inclusion of names-inl.h is not allowed, " \
    "please include names.h instead"
#endif

#include "name_data.h"

#include <yandex/maps/wiki/common/misc.h>
#include <maps/libs/common/include/exception.h>

#include <algorithm>
#include <cstdint>
#include <map>
#include <set>

namespace maps {
namespace wiki {
namespace validator {
namespace utils {

namespace {

template<typename Type>
struct GeomForReport {
    static boost::none_t geom(const Type*) {
        return boost::none;
    }
};

}

struct Regexps {
    static const icu::RegexPattern BEGINS_WITH_SPACE_REGEX;
    static const icu::RegexPattern DOUBLE_SPACE_REGEX;
    static const icu::RegexPattern NONCYRILLIC_LETTER_REGEX;
};

template<class NameCategory, class Object>
void runBasicNamesCheck(
        const Object* object, CheckContext* context, Severity officialNameSeverity)
{
    struct NameCounts
    {
        NameCounts()
            : localCount(0)
            , nonLocalCount(0)
        { }

        size_t localCount;
        size_t nonLocalCount;
    };

    auto geomForReport = GeomForReport<Object>::geom(object);

    std::set<std::pair<NameRelation::Type, std::string>> nameTypeLangs;
    std::map<std::string, NameCounts> nameCountsByLang;
    for (const NameRelation& rel : object->names()) {
        const Name* name = context->objects<NameCategory>().byId(rel.id);
        if (isNameEmpty(name->name())) {
            context->error(
                "empty-name", geomForReport, { object->id() });
            continue;
        }
        if (matchesPattern(name->name(), Regexps::BEGINS_WITH_SPACE_REGEX)) {
            context->error(
                "name-begins-with-space",
                geomForReport, { object->id() });
        }
        if (matchesPattern(name->name(), Regexps::DOUBLE_SPACE_REGEX)) {
            context->error(
                "double-space-in-name",
                geomForReport, { object->id() });
        }

        if (!isValidLang(name->lang())) {
            context->fatal(
                "unknown-name-lang",
                geomForReport, { object->id() });
        }
        if (name->lang() == LANG_RU
                && matchesPattern(
                    name->name(), Regexps::NONCYRILLIC_LETTER_REGEX)) {
            context->error(
                "illegal-letter-for-name-lang",
                geomForReport, { object->id() });
        }
        if (!common::isIn(
                rel.type,
                { NameRelation::Type::Synonym,
                  NameRelation::Type::Old })
                && !nameTypeLangs.insert({
                    rel.type, name->lang()}).second) {
            context->critical(
                "multiple-non-synonym-names",
                geomForReport,
                { object->id() });
        }

        if (name->isLocal()) {
            ++nameCountsByLang[name->lang()].localCount;
        } else {
            ++nameCountsByLang[name->lang()].nonLocalCount;
        }
    }

    for (const auto& langCountsPair : nameCountsByLang) {
        const auto& lang = langCountsPair.first;
        const auto& nameCounts = langCountsPair.second;

        if (!nameTypeLangs.count({ NameRelation::Type::Official, lang })) {
            context->report(
                officialNameSeverity,
                "official-name-missing",
                geomForReport,
                { object->id() });
        }

        if (nameCounts.localCount && nameCounts.nonLocalCount) {
            for (size_t i = 0; i < nameCounts.nonLocalCount; ++i) {
                context->error(
                    "non-local-name-for-local-lang",
                    geomForReport,
                    { object->id() });
            }
        }
    }
}

template<class NameCategory, class Object>
NameData namesByType(
        const Object* object,
        CheckContext* context,
        NameRelation::Type type)
{
    NameData result;
    for (const NameRelation& nameRel : object->names()) {
        if (nameRel.type == type
                && context->objects<NameCategory>().loaded(nameRel.id)) {
            const Name* name = context->objects<NameCategory>().byId(nameRel.id);
            result.push_back(NameDatum{name->name(), name->lang(), name->isLocal()});
        }
    }
    return result;
}

template<class NameCategory>
std::set<std::string> langsFromNames(
        const std::vector<NameRelation>& names,
        CheckContext* context)
{
    std::set<std::string> result;
    for (const auto& nameRel : names) {
        if (context->objects<NameCategory>().loaded(nameRel.id)) {
            const auto* name = context->objects<NameCategory>().byId(nameRel.id);
            result.insert(name->lang());
        }
    }
    return result;
}

template<class NameCategory, class Object>
NameData officialNames(const Object* object, CheckContext* context)
{
    return namesByType<NameCategory, Object>(
            object, context, NameRelation::Type::Official);
}

template<class NameCategory, class Object>
bool isNamed(const Object* object, CheckContext* context)
{
    NameData officialNamesData =
        officialNames<NameCategory, Object>(object, context);
    return std::any_of(
        std::begin(officialNamesData),
        std::end(officialNamesData),
        [](const NameDatum& nameDatum)
        { return !isNameEmpty(nameDatum.name); });
}

template<class NameCategory, class Object, class NameVisitor>
void visitRenderedNames(
        const Object* object,
        CheckContext* context,
        NameVisitor visitor)
{
    NameData renderLabelsData = namesByType<NameCategory, Object>(
            object, context, NameRelation::Type::RenderLabel);
    std::set<std::string> langsWithRenderLabel;

    for (const auto& labelDatum : renderLabelsData) {
        langsWithRenderLabel.insert(labelDatum.lang);
        visitor(object, context, labelDatum.name);
    }

    NameData officialNamesData =
            officialNames<NameCategory, Object>(object, context);
    for (const auto& nameDatum : officialNamesData) {
        if (!langsWithRenderLabel.count(nameDatum.lang)) {
            visitor(object, context, nameDatum.name);
        }
    }
}

template<class NameCategory, class Object>
void runCapsNameCheck(const Object* object, CheckContext* context)
{
    constexpr size_t MAX_CAPS_NAME_LEN = 5;

    for (const NameRelation& nameRel : object->names()) {
        if (common::isIn(nameRel.type, {NameRelation::Type::AddressLabel,
                                       NameRelation::Type::RenderLabel,
                                       NameRelation::Type::Official})
                && context->objects<NameCategory>().loaded(nameRel.id)) {
            auto name = context->objects<NameCategory>().byId(nameRel.id)->name();

            auto words = splitIntoWords(name);

            if (std::any_of(
                    words.begin(),
                    words.end(),
                    [](const std::string& word) {
                        return countCodePoints(word) > MAX_CAPS_NAME_LEN
                                && isUpperCase(word);
                    })) {
                context->warning(
                    "name-with-all-capital-letters",
                    GeomForReport<Object>::geom(object),
                    {object->id()});
            }
        }
    }
}

} // namespace utils
} // namespace validator
} // namespace wiki
} // namespace maps
