#pragma once

#ifndef VALIDATOR_CHECKS_UTILS_TOPOLOGY_CHECKS_INL
#error "direct inclusion of topology_checks-inl.h is not allowed, " \
    "please include topology_checks.h instead"
#endif

#include "misc.h"
#include "geom.h"

#include <yandex/maps/wiki/validator/categories.h>
#include <yandex/maps/wiki/validator/check_context.h>

#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/spatial_relation.h>

#include <cmath>
#include <algorithm>
#include <iterator>
#include <vector>
#include <unordered_set>
#include <unordered_map>

namespace maps {
namespace wiki {
namespace validator {
namespace utils {

namespace {

const double SEGMENT_MIN_LENGTH = 1e-6; // geo meters

const double JC_BBOX_WITHOUT_UNRELATED_ELEMENTS_BIG = 2 * 2.0;   //geo meters
const double JC_BBOX_WITHOUT_UNRELATED_ELEMENTS_SMALL = 2 * 0.3; //geo meters

const size_t JC_MAX_DEGREE_FOR_BIG_BBOX = 1;

const size_t VISIT_BATCH_SIZE = 10000;
const size_t INTERSECTIONS_VISIT_BATCH_SIZE = 1000;

const double INTERSECTIONS_MAX_BBOX_SIZE = 1e3; // mercator meters

} // namespace

namespace detail {

enum class IntersectionsCheckCyclingMode
{
    Disallow,
    Allow
};

void checkSelfIntersections(
    CheckContext* context, IntersectionsCheckCyclingMode cyclingMode,
    TId id, const geolib3::Polyline2& geom,
    Severity severity);

void checkIntersections(
    CheckContext* context,
    TId firstId, const geolib3::Polyline2& firstGeom,
    TId secondId, const geolib3::Polyline2& secondGeom,
    Severity severity, const std::set<TId>& commonAncestors);

template<class JunctionCategory>
void checkEndPoint(
        CheckContext* context,
        const std::string& name,
        const geolib3::Point2& geom,
        TId elementId,
        TId expectedJcId,
        Severity severity)
{
    auto junctions = context->objects<JunctionCategory>().byBbox(
            geolib3::BoundingBox(geom, EPS, EPS));
    if (junctions.empty()
            && !context->aoi().empty()
            && !context->aoi().intersects(geom)) {
        return;
    }

    bool foundExpected = false;
    for (auto junction : junctions) {
        if (junction->id() != expectedJcId) {
            context->report(
                    severity, "unexpected-junction-at-" + name,
                    geom, {elementId, junction->id()});
        } else {
            foundExpected = true;
        }
    }

    if (!foundExpected) {
        context->report(
                severity, name + "-junction-misplaced",
                geom, {elementId, expectedJcId});
    }
}

template<class ElementCategory>
std::vector<const typename ElementCategory::TObject*> elemsByBbox(
        CheckContext* context,
        const geolib3::Point2& center,
        double boundingBoxSize)
{
    double cosLatitude = cos(geolib3::mercator2GeoPoint(center).y() * M_PI / 180);
    boundingBoxSize /= cosLatitude;
    return context->objects<ElementCategory>().byBbox(
            geolib3::BoundingBox(center, boundingBoxSize, boundingBoxSize));
}

std::set<TId> commonIds(const std::set<TId>& one, const std::set<TId>& other);

template<class ElementCategory>
void runLinearElementsIntersectionsCheckImpl(
    CheckContext* context,
    detail::IntersectionsCheckCyclingMode checkCyclingMode,
    Severity severity)
{
    context->checkLoaded<ElementCategory>();

    auto viewElement = context->objects<ElementCategory>();

    viewElement.batchVisit(
            [&](const typename ElementCategory::TObject* element)
    {
        const geolib3::Polyline2& geom = element->geom();
        detail::checkSelfIntersections(
                context, checkCyclingMode,
                element->id(), geom,
                severity);

        geolib3::BoundingBox bbox(geom.pointAt(0), EPS, EPS);
        std::unordered_set<TId> otherIds;
        for (size_t i = 1; i < geom.points().size(); ++i) {
            bbox = geolib3::expand(bbox, geom.pointAt(i));
            if (bbox.width() > INTERSECTIONS_MAX_BBOX_SIZE
                  || bbox.height() > INTERSECTIONS_MAX_BBOX_SIZE
                  || i + 1 == geom.points().size()) {
                const auto& others = viewElement.byBbox(
                        geolib3::resizeByValue(bbox, EPS));
                for (auto other : others) {
                    if (other->id() < element->id()
                          && otherIds.insert(other->id()).second) {
                        detail::checkIntersections(
                                context,
                                element->id(), geom,
                                other->id(), other->geom(),
                                severity, {});
                    }
                }
                bbox = geolib3::BoundingBox(geom.pointAt(i), EPS, EPS);
            }
        }
    }, INTERSECTIONS_VISIT_BATCH_SIZE);
}

} // namespace detail

template<class ElementCategory>
void runLinearElementsIntersectionsCheck(
    CheckContext* context,
    Severity severity)
{
    detail::runLinearElementsIntersectionsCheckImpl<ElementCategory>(
        context,
        detail::IntersectionsCheckCyclingMode::Disallow,
        severity);
}

template<class ElementCategory, class FaceCategory>
void runFaceElementsIntersectionsCheck(CheckContext* context)
{
    context->checkLoaded<FaceCategory>();
    context->checkLoaded<ElementCategory>();

    auto viewFace = context->objects<FaceCategory>();
    auto viewElement = context->objects<ElementCategory>();

    size_t elementSize = 0;
    viewElement.visit([&](const typename ElementCategory::TObject*) {
        ++elementSize;
    });

    std::unordered_map<TId, std::set<TId>> domainObjectsByElementId;
    domainObjectsByElementId.reserve(elementSize);

    viewElement.visit(
            [&](const typename ElementCategory::TObject* element)
    {
        auto& domainObjects = domainObjectsByElementId[element->id()];
        for (TId faceId : element->parents()) {
            if (viewFace.loaded(faceId)) {
                auto face = viewFace.byId(faceId);
                domainObjects.insert(face->parent());
            }
        }
    });

    viewElement.batchVisit(
            [&](const typename ElementCategory::TObject* element)
    {
        const geolib3::Polyline2& geom = element->geom();
        detail::checkSelfIntersections(
                context, detail::IntersectionsCheckCyclingMode::Allow,
                element->id(), geom,
                Severity::Fatal);

        geolib3::BoundingBox bbox(geom.pointAt(0), EPS, EPS);
        std::unordered_set<TId> otherIds;
        for (size_t i = 1; i < geom.points().size(); ++i) {
            bbox = geolib3::expand(bbox, geom.pointAt(i));
            if (bbox.width() > INTERSECTIONS_MAX_BBOX_SIZE
                  || bbox.height() > INTERSECTIONS_MAX_BBOX_SIZE
                  || i + 1 == geom.points().size()) {
                const auto& others = viewElement.byBbox(
                        geolib3::resizeByValue(bbox, EPS));
                for (auto other : others) {
                    if (other->id() < element->id()
                          && otherIds.insert(other->id()).second) {
                        std::set<TId> commonDomainObjects = detail::commonIds(
                                domainObjectsByElementId.at(element->id()),
                                domainObjectsByElementId.at(other->id()));

                        detail::checkIntersections(
                                context,
                                element->id(), geom,
                                other->id(), other->geom(),
                                commonDomainObjects.empty()
                                ? Severity::Warning
                                : Severity::Fatal,
                                commonDomainObjects);
                    }
                }
                bbox = geolib3::BoundingBox(geom.pointAt(i), EPS, EPS);
           }
        }
    }, INTERSECTIONS_VISIT_BATCH_SIZE);
}


template<class ElementCategory, class JunctionCategory>
void runJcElCoverageCheck(
        CheckContext* context,
        Severity severity,
        std::function<Severity(const typename ElementCategory::TObject*)> unexpectedElementSeverityFunc,
        CheckElementsAtJunction checkElementsAtJunction)
{
    context->checkLoaded<ElementCategory>();
    context->checkLoaded<JunctionCategory>();

    context->objects<JunctionCategory>().batchVisit(
            [&](const typename JunctionCategory::TObject* junction)
    {
        std::unordered_set<TId> relatedElemIds;
        relatedElemIds.insert(
                junction->inElements().cbegin(),
                junction->inElements().cend());
        relatedElemIds.insert(
                junction->outElements().cbegin(),
                junction->outElements().cend());

        size_t relatedElemsCount =
            junction->inElements().size() + junction->outElements().size();
        double bBoxSize = relatedElemsCount <= JC_MAX_DEGREE_FOR_BIG_BBOX ?
                          JC_BBOX_WITHOUT_UNRELATED_ELEMENTS_BIG :
                          JC_BBOX_WITHOUT_UNRELATED_ELEMENTS_SMALL;
        auto elems = detail::elemsByBbox<ElementCategory>(
                context, junction->geom(), bBoxSize);

        if (elems.empty() || relatedElemsCount == 0) {
            context->report(
                    severity, "stray-junction", junction->geom(), {junction->id()});
            return;
        }

        if (checkElementsAtJunction == CheckElementsAtJunction::Yes) {
            for (const auto& elem : elems) {
                if (relatedElemIds.count(elem->id()) == 0) {
                    context->report(
                            unexpectedElementSeverityFunc(elem), "unexpected-element-at-junction",
                            junction->geom(), {junction->id(), elem->id()});
                }
            }
        }
    }, VISIT_BATCH_SIZE);

    context->objects<ElementCategory>().batchVisit(
            [&](const typename ElementCategory::TObject* element)
    {
        detail::checkEndPoint<JunctionCategory>(
                context, "startpoint", element->geom().points().front(),
                element->id(), element->startJunction(),
                severity);

        detail::checkEndPoint<JunctionCategory>(
                context, "endpoint", element->geom().points().back(),
                element->id(), element->endJunction(),
                severity);
    }, VISIT_BATCH_SIZE);
}

template<class ElementCategory, class JunctionCategory>
void runJcElCoverageCheck(
        CheckContext* context,
        Severity severity,
        CheckElementsAtJunction checkElementsAtJunction)
{
    runJcElCoverageCheck<ElementCategory, JunctionCategory>(context, severity,
        [&](const typename ElementCategory::TObject*) {
            return severity;
        },
        checkElementsAtJunction);
}


template<class JunctionCategory>
void runOpenBoundsCheck(CheckContext* context)
{
    context->checkLoaded<JunctionCategory>();

    std::unordered_set<TId> reportedIds;
    context->objects<JunctionCategory>().visit(
            [&](const Junction* junction)
    {
        if (junction->inElements().size()
                + junction->outElements().size() == 1) {
            TId idToReport = junction->inElements().empty()
                ? junction->outElements().front()
                : junction->inElements().front();

            if (reportedIds.insert(idToReport).second) {
                context->fatal(
                    "open-bound", junction->geom(), { idToReport });
            }
        }
    });
}

inline bool isSegmentTooShort(const geolib3::Segment2& segment)
{
    geolib3::Point2 start = mercator2GeoPoint(segment.start());
    geolib3::Point2 end = mercator2GeoPoint(segment.end());
    double length = geolib3::fastGeoDistance(start, end);
    return (length < SEGMENT_MIN_LENGTH);
}

template<class ElementCategory>
void runSegmentsLengthCheck(CheckContext* context)
{
    context->checkLoaded<ElementCategory>();

    context->objects<ElementCategory>().visit(
            [&](const typename ElementCategory::TObject* element)
    {
        for (const auto& segment: element->geom().segments()) {
            if (isSegmentTooShort(segment)) {
                context->warning(
                        "segment-too-short",
                        segment.midpoint(),
                        {element->id()});
            }
        }
    });
}


} // namespace utils
} // namespace validator
} // namespace wiki
} // namespace maps
