#include "topology_checks.h"
#include "misc.h"

#include <yandex/maps/wiki/validator/check_context.h>

#include <maps/libs/geolib/include/line.h>
#include <maps/libs/geolib/include/direction.h>
#include <maps/libs/geolib/include/intersection.h>
#include <maps/libs/geolib/include/projection.h>
#include <maps/libs/geolib/include/distance.h>

namespace maps {
namespace wiki {
namespace validator {
namespace utils {
namespace detail {

namespace gl = maps::geolib3;

namespace {

//distance 8.6cm between roads with length l.9m and same start point
const double MAX_ANGLE_BETWEEN_OVERLAPPING_SEGMENTS = 0.045;

const std::string SAME_OBJECT_PREFIX = "same-object-";
const std::string EMPTY_PREFIX = "";

bool isCycle(const gl::Polyline2& polyline)
{ return polyline.points().front() == polyline.points().back(); }

std::vector<gl::Point2> selfIntersections(const gl::Polyline2& polyline)
{
    const gl::PointsVector& points = polyline.points();
    if (points.empty()) {
        return {};
    }

    std::vector<gl::Segment2> segments;
    for (const gl::Segment2& segment: polyline.segments()) {
        if (!segment.isDegenerate()) {
            segments.push_back(segment);
        }
    }

    if (segments.size() < 2) {
        return {};
    }

    std::vector<gl::Point2> result;
    for (size_t iSegment = 1; iSegment < segments.size(); ++iSegment) {
        const gl::Segment2& currentSegment = segments[iSegment];

        for (size_t iPrev = 0; iPrev < iSegment; ++iPrev) {
            const gl::Segment2& prevSegment = segments[iPrev];
            const bool mustHaveCommonPoint = (iPrev + 1 == iSegment)
                || (iPrev == 0 && iSegment + 1 == segments.size()
                        && isCycle(polyline));
            auto intersections = gl::intersection(currentSegment, prevSegment);
            if ((!mustHaveCommonPoint && intersections.size() > 0) ||
                (mustHaveCommonPoint && intersections.size() == 2)) {
                std::copy(
                        intersections.begin(), intersections.end(),
                        std::back_inserter(result));
            }
        }
    }

    return result;
}

double angle(gl::Line2 line, gl::Line2 other)
{
    const double dirAngle = angle(line.direction(), other.direction());
    return dirAngle < M_PI / 2 ? dirAngle : M_PI - dirAngle;
}

boost::optional<gl::Segment2> segmentsOverlap(
        const gl::Segment2& segment, const gl::Segment2& other)
{
    // Avoid fail on line construction
    if (gl::squaredLength(segment) <= gl::EPS * gl::EPS
        || gl::squaredLength(other) <= gl::EPS * gl::EPS) {
        return boost::none;
    }

    double length = gl::length(segment);

    gl::Point2 otherStart = other.start();
    double startProjLength = gl::projectionFactor(segment, otherStart) * length;

    gl::Point2 otherEnd = other.end();
    double endProjLength = gl::projectionFactor(segment, otherEnd) * length;

    if (startProjLength > endProjLength) {
        std::swap(startProjLength, endProjLength);
        std::swap(otherStart, otherEnd);
    }

    if (startProjLength > length - EPS || endProjLength < EPS) {
        return boost::none;
    }

    gl::Line2 line = segment.line();

    if (angle(line, other.line()) > MAX_ANGLE_BETWEEN_OVERLAPPING_SEGMENTS) {
        return boost::none;
    }

    // transform everything into the coordinate system with x axis
    // along the first segment.
    gl::BoundingBox transformedBbox(gl::Point2(0, -EPS), gl::Point2(length, EPS));

    auto transformedYCoord = [&](const gl::Point2& pt)
    {
        return gl::innerProduct(pt - line.anyPoint(), line.normal())
            / gl::length(line.normal());
    };
    gl::Segment2 transformedOther(
            gl::Point2(startProjLength, transformedYCoord(otherStart)),
            gl::Point2(endProjLength, transformedYCoord(otherEnd)));
    if (gl::spatialRelation(transformedBbox, transformedOther, gl::Disjoint)) {
        return boost::none;
    }

    return gl::Segment2(
            startProjLength > 0 ? gl::projection(line, otherStart) : segment.start(),
            endProjLength < length ? gl::projection(line, otherEnd) : segment.end());
}

std::vector<gl::Segment2> selfOverlaps(const gl::Polyline2& polyline) {
    std::vector<gl::Segment2> ret;
    for (size_t iSegment = 1; iSegment < polyline.segmentsNumber(); ++iSegment) {
        for (size_t iPrev = 0; iPrev < iSegment; ++iPrev) {
            auto overlap = segmentsOverlap(
                    polyline.segmentAt(iPrev), polyline.segmentAt(iSegment));
            if (overlap) {
                ret.push_back(*overlap);
            }
        }
    }
    return ret;
}

std::vector<gl::Segment2> overlaps(
        const gl::Polyline2& polyline, const gl::Polyline2& other)
{
    std::vector<gl::Segment2> ret;
    for (size_t iSegment = 0; iSegment < polyline.segmentsNumber(); ++iSegment) {
        for (size_t iOtherSegment = 0; iOtherSegment < other.segmentsNumber();
             ++iOtherSegment) {
            auto overlap = segmentsOverlap(
                    polyline.segmentAt(iSegment), other.segmentAt(iOtherSegment));
            if (overlap) {
                ret.push_back(*overlap);
            }
        }
    }
    return ret;
}

bool isValidIntersection(
        const gl::Polyline2& polyline, const gl::Polyline2& other,
        const gl::Point2& point)
{
    if ((polyline.points().front() == other.points().front() ||
         polyline.points().front() == other.points().back()) &&
        point == polyline.points().front()) {
        return true;
    }

    if ((polyline.points().back() == other.points().front() ||
         polyline.points().back() == other.points().back()) &&
        point == polyline.points().back()) {
        return true;
    }

    return false;
}

} // namespace

void checkSelfIntersections(
        CheckContext* context, IntersectionsCheckCyclingMode cyclingMode,
        TId id, const gl::Polyline2& geom,
        Severity severity)
{
    for (const gl::Point2& intersection : selfIntersections(geom)) {
        context->report(
                severity,
                "element-self-intersection", intersection, {id});
    }

    if (cyclingMode == IntersectionsCheckCyclingMode::Disallow
            && isCycle(geom)) {
        context->report(
                severity,
                "element-self-intersection", geom.points().front(), {id});
    }

    for (const gl::Segment2& overlap : selfOverlaps(geom)) {
        context->report(
                severity,
                "element-self-overlap", overlap.midpoint(), {id});
    }
}

void checkIntersections(
        CheckContext* context,
        TId firstId, const gl::Polyline2& firstGeom,
        TId secondId, const gl::Polyline2& secondGeom,
        Severity severity, const std::set<TId>& commonDomainObjects)
{
    std::string messagePrefix =
            (commonDomainObjects.empty() ? EMPTY_PREFIX : SAME_OBJECT_PREFIX);
    std::vector<TId> reportedIds = {firstId, secondId};
    reportedIds.insert(
        reportedIds.end(),
        commonDomainObjects.begin(), commonDomainObjects.end());

    for (const gl::Polyline2& intersection :
             gl::intersection(firstGeom, secondGeom)) {
        if (intersection.pointsNumber() == 1 &&
            !isValidIntersection(
                    firstGeom, secondGeom,
                    intersection.pointAt(0))) {
            context->report(
                    severity,
                    messagePrefix + "element-intersection",
                    intersection.pointAt(0),
                    reportedIds);
        }
    }

    for (const gl::Segment2& overlap :
             overlaps(firstGeom, secondGeom)) {
        context->report(
                severity,
                messagePrefix + "element-overlap",
                overlap.midpoint(),
                reportedIds);
    }
}

std::set<TId> commonIds(const std::set<TId>& one, const std::set<TId>& other)
{
    std::set<TId> commonIds;
    std::set_intersection(
            one.begin(), one.end(),
            other.begin(), other.end(),
            std::inserter(commonIds, commonIds.begin()));
    return commonIds;
}

} // namespace detail
} // namespace utils
} // namespace validator
} // namespace wiki
} // namespace maps
