#include "module.h"
#include <yandex/maps/wiki/validator/check.h>
#include <yandex/maps/wiki/validator/categories.h>

#include "../utils/face_builder.h"
#include "../utils/face_checks.h"
#include "../utils/object_elements_within_aoi.h"

using maps::wiki::validator::categories::VEGETATION;
using maps::wiki::validator::categories::VEGETATION_FC;
using maps::wiki::validator::categories::VEGETATION_EL;
using maps::wiki::validator::categories::VEGETATION_CNT;

namespace maps {
namespace wiki {
namespace validator {
namespace checks {

namespace {
const size_t VISIT_BATCH_SIZE = 1000;
}

VALIDATOR_SIMPLE_CHECK( vegetation_cnt,
        VEGETATION, VEGETATION_FC, VEGETATION_EL, VEGETATION_CNT )
{
    auto viewVegetation = context->objects<VEGETATION>();
    auto viewVegetationFc = context->objects<VEGETATION_FC>();
    auto viewVegetationEl = context->objects<VEGETATION_EL>();

    context->objects<VEGETATION_CNT>().batchVisit([&](const Center* center) {
        auto featureId = center->parent();
        if (!context->objects<VEGETATION>().loaded(featureId)) {
            context->report(
                Severity::Error,
                "vegetation-outside-aoi",
                center->geom(),
                {center->id(), center->parent()});
            return;
        }
        const auto& feature = viewVegetation.byId(featureId);
        if (!utils::objectFacesWithinAoi<VEGETATION>(context, feature)) {
            return;
        }

        bool isCntInside = false;

        for (auto faceId : feature->faces()) {
            const auto& face = viewVegetationFc.byId(faceId);
            utils::FaceBuilder faceBuilder(face, viewVegetationEl);
            auto geom = utils::facePolygon(faceBuilder);

            if (geom && geolib3::spatialRelation(
                    *geom, center->geom(), geolib3::SpatialRelation::Contains)) {
                if (face->isInterior()) {
                    isCntInside = false;
                    break;
                } else {
                    isCntInside = true;
                }
            }
        }

        if (!isCntInside) {
            context->report(
                Severity::Error,
                "vegetation-cnt-outside-feature",
                center->geom(),
                {center->id(), center->parent()});
        }
    }, VISIT_BATCH_SIZE);
}

} // namespace checks
} // namespace validator
} // namespace wiki
} // namespace maps
