#include "export_service.h"
#include "gdal_helpers.h"
#include "helpers.h"
#include "json2ymapsdf.h"
#include "measurable_task.h"
#include "pg_helpers.h"
#include "revision2json.h"

#include <yandex/maps/wiki/revisionapi/revisionapi.h>
#include <maps/libs/log8/include/log8.h>

#include <filesystem>
#include <fstream>
#include <future>

namespace maps::wiki::exporter {

namespace revapi = maps::wiki::revisionapi;
namespace fs = std::filesystem;

namespace {

/// Export json for service categories
void writeJsonService(
    const ExportConfig& exportCfg,
    const ExportFiles& exportFiles,
    const CategoriesMap& categoriesMap)
{
    JsonPass pass;
    pass.name = "service";
    pass.commitId = exportCfg.commitId();
    revapi::Strings categories;
    revapi::Strings allowedRelatives;
    for (const auto& [categoryId, attrInfos]: categoriesMap) {
        INFO () << "Putting service category " << categoryId;
        categories.push_back("cat:" + categoryId);
        for (const auto& [_, attrInfo] : attrInfos) {
            if (attrInfo.relativeIdSource) {
                allowedRelatives.push_back(attrInfo.relativeIdSource->relativeCategoryId);
            }
        }
    }

    auto& pool = exportCfg.mainPool();
    auto branch = loadBranchByString(pool, exportCfg.branch());
    REQUIRE(exportCfg.commitId(), "export from empty database");
    revapi::ExportParams params{branch, exportCfg.commitId(), categories, allowedRelatives};

    fs::path outputPath = exportFiles(TmpFile::SERVICE_JSON_FILE);
    std::ofstream file;
    file.open(outputPath.string());
    REQUIRE(!file.fail(), "Can't open file: " << outputPath.string());
    auto callback = revapi::SingleStreamWrapper(file);

    runMeasurable([&exportCfg, &params, callback]{
        exportJsons(exportCfg, params, callback, {});
    }, exportCfg.logger(), "generating json");
}

Datasets exportCategories(
    const ExportConfig& exportCfg,
    const ExportFiles& exportFiles,
    const CategoriesMap& categories,
    const CategoriesFiltersMap& filters)
{
    exportCfg.checkCanceled();
    writeJsonService(exportCfg, exportFiles, categories);
    exportCfg.checkCanceled();
    runMeasurable(
        [&]{ json2Shape(
                exportFiles(TmpFile::SERVICE_JSON_FILE),
                exportFiles(TmpFile::SERVICE_SHAPE_DIR),
                categories,
                filters); },
        exportCfg.logger(), "converting json to shape");

    exportCfg.checkCanceled();
    runMeasurable(
        [&]{ convertFromShape(
                exportFiles(TmpFile::SERVICE_SHAPE_DIR),
                exportFiles(TmpFile::SERVICE_GEOJSON_DIR)); },
        exportCfg.logger(), "converting shape to geojson");

    // Create archives
    exportCfg.checkCanceled();
    runMeasurable(
        [&]{ tarGz(
                exportFiles(TmpFile::SERVICE_SHAPE_DIR),
                exportFiles(ResultFile::SERVICE_SHAPE_ARCHIVE, NO_REGION)); },
        exportCfg.logger(), "archiving shape");

    exportCfg.checkCanceled();
    runMeasurable(
        [&]{ tarGz(
                exportFiles(TmpFile::SERVICE_GEOJSON_DIR),
                exportFiles(ResultFile::SERVICE_GEOJSON_ARCHIVE, NO_REGION)); },
        exportCfg.logger(), "archiving geojson");

    exportCfg.checkCanceled();
    std::optional<Dataset> dataset = makeDataset(exportCfg, exportFiles);
    return dataset ? Datasets{std::move(*dataset)} : Datasets{};
}

} // namespace

Datasets exportServiceSubset(
    const ExportConfig& exportCfg,
    const ExportFiles& exportFiles)
{
    exportCfg.checkCanceled();
    writeJsonService(exportCfg, exportFiles, exportCfg.serviceCategories());

    exportCfg.checkCanceled();
    runMeasurable(
        [&]{ json2Shape(
                exportFiles(TmpFile::SERVICE_JSON_FILE),
                exportFiles(TmpFile::SERVICE_SHAPE_DIR),
                exportCfg.serviceCategories()); },
        exportCfg.logger(), "converting json to shape");

    // Create archive
    exportCfg.checkCanceled();
    runMeasurable(
        [&]{ tarGz(
                exportFiles(TmpFile::SERVICE_SHAPE_DIR),
                exportFiles(ResultFile::SERVICE_SHAPE_ARCHIVE, NO_REGION)); },
        exportCfg.logger(), "archiving shape");

    exportCfg.checkCanceled();
    std::optional<Dataset> dataset = makeDataset(exportCfg, exportFiles);
    return dataset ? Datasets{std::move(*dataset)} : Datasets{};
}

Datasets exportMrcSubset(
    const ExportConfig& exportCfg,
    const ExportFiles& exportFiles)
{
    return exportCategories(
        exportCfg,
        exportFiles,
        exportCfg.mrcCategories(),
        exportCfg.mrcFilters());
}

Datasets exportMrcPedestrianSubset(
    const ExportConfig& exportCfg,
    const ExportFiles& exportFiles)
{

    return exportCategories(
        exportCfg,
        exportFiles,
        exportCfg.mrcPedestrianCategories(),
        {});
}
} // namespace maps::wiki::exporter
