#include "run.h"
#include "helpers.h"
#include "pg_helpers.h"
#include "export_service.h"
#include "export_domain.h"
#include "export_masstransit.h"
#include "measurable_task.h"

#include <maps/libs/concurrent/include/scoped_guard.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/masstransit/geobase.h>

#include <boost/regex.hpp>
#include <boost/algorithm/string/predicate.hpp>

namespace maps::wiki::exporter {

namespace {

std::optional<std::chrono::system_clock::time_point>
extractSchemaTs(const ExportConfig& cfg, const std::string& schema)
{
    static const boost::regex PATTERN(cfg.schemaPrefix() + TASK_TAG_RE);

    boost::smatch matchResult;
    boost::regex_match(schema, matchResult, PATTERN);

    if (matchResult.size() == 2) {
        std::string date = matchResult[1];
        std::tm time;
        memset(&time, 0, sizeof(time));
        strptime(date.c_str(), DATE_FORMAT.c_str(), &time);
        return std::chrono::system_clock::from_time_t(timegm(&time));
    }
    return std::nullopt;
}


bool published(const std::string& schema)
{
    return boost::algorithm::ends_with(schema, PUBLISHED_SUFFIX);
}


/// @brief: Drops schemas that match the TASK_TAG_RE pattern
/// and are older than TMP_SCHEMATA_TO_KEEP_DAYS
void cleanupTmpSchemata(const ExportConfig& cfg)
{
    using Days = std::chrono::duration<int,std::ratio<60*60*24>>;
    using Clock = std::chrono::system_clock;

    auto thresholdTs = Clock::now() - Days(TMP_SCHEMATA_TO_KEEP_DAYS);
    INFO() << "Threshold ts is " << thresholdTs.time_since_epoch().count();

    try {
        auto& pool = cfg.ymapsdfPool();
        auto schemas = getSchemasByPattern(pool, cfg.schemaPrefix() + "%");
        for (const auto& schema: schemas) {
            const auto schemaTs = extractSchemaTs(cfg, schema);
            const auto oldSchema = schemaTs && thresholdTs > *schemaTs;
            if (published(schema) || oldSchema) {
                dropSchema(pool, schema);
            }
        }
    } catch (const std::exception& e) {
        WARN() << "Failed to drop tmp schemata: " << e.what();
    }
}


void renameTmpSchemata(const ExportConfig& cfg)
{
    cfg.logger().logInfo() << "Renaming tmp db schemata";
    try {
        auto& pool = cfg.ymapsdfPool();

        // always rename common schema without region suffix
        const auto schema = cfg.schema(NO_REGION);
        renameSchema(pool, schema, schema + PUBLISHED_SUFFIX);

        auto regions = cfg.regions();
        for (const auto& region: regions) {
            if (region == NO_REGION) {
                continue; // renamed already
            }
            const auto schema = cfg.schema(region);
            renameSchema(pool, schema, schema + PUBLISHED_SUFFIX);
        }
    } catch (const std::exception& e) {
        WARN() << "Failed to rename schemata: " << e.what();
    }
}

} // namespace


void setGeoData(const std::string& path)
{
    if (!path.empty()) {
        INFO() << "Using geodata: " << path;
        masstransit::setGeodataBinPath(path);
    }
}

Datasets runExport(const ExportConfig& exportCfg, const std::string& rootDir)
{
    auto& logger = exportCfg.logger();

    logger.logInfo()
        << "Arguments: branch: " << exportCfg.branch()
        << ", commit id: " << exportCfg.commitId()
        << ", subset: " << exportCfg.subset();

    const auto regions = exportCfg.regions();

    logger.logInfo() << "Removing old directories from filesystem";
    ExportFiles::cleanupOldDirs(rootDir, exportCfg.subset());
    logger.logInfo() << "Old directories have been cleaned up";

    logger.logInfo() << "Removing old db schemata";
    cleanupTmpSchemata(exportCfg);
    logger.logInfo() << "Old db schemata have been dropped";

    ExportFiles exportFiles(rootDir, exportCfg);

    Datasets datasets;
    switch (exportCfg.subset()) {
        case Subset::Ymapsdf:
            datasets = exportDomainSubset(exportCfg, exportFiles);
            break;
        case Subset::Masstransit:
            datasets = exportMasstransitSubset(exportCfg, exportFiles);
            break;
        case Subset::Service:
            datasets = exportServiceSubset(exportCfg, exportFiles);
            break;
        case Subset::Mrc:
            datasets = exportMrcSubset(exportCfg, exportFiles);
            break;
        case Subset::MrcPedestrian:
            datasets = exportMrcPedestrianSubset(exportCfg, exportFiles);
            break;
        default:
            throw maps::LogicError() << "Invalid export subset value: "
                << static_cast<int>(exportCfg.subset());
    }

    renameTmpSchemata(exportCfg);

    logger.logInfo() << "done";
    INFO() << "Export done";

    return datasets;
}

} // namespace maps::wiki::exporter
