#include "../lib/config.h"
#include "../lib/dir_lock.h"
#include "../lib/export_files.h"
#include "../lib/helpers.h"
#include "../lib/pg_helpers.h"
#include "../lib/run.h"
#include "../lib/task_params.h"

#include <maps/libs/concurrent/include/scoped_guard.h>
#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/shell_cmd.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>
#include <yandex/maps/wiki/mds_dataset/dataset_gateway.h>
#include <yandex/maps/wiki/mds_dataset/export_metadata.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/tasks/export.h>
#include <yandex/maps/wiki/unittest/unittest.h>

#define BOOST_TEST_DYN_LINK
#define BOOST_TEST_MAIN
#include <boost/algorithm/string.hpp>
#include <boost/format.hpp>
#include <boost/regex.hpp>
#include <boost/test/unit_test.hpp>
#include <gdal/gdal.h>
#include <gdal/ogrsf_frmts.h>

#include <chrono>
#include <filesystem>
#include <fstream>
#include <iomanip>

namespace fs = std::filesystem;
namespace rev = maps::wiki::revision;
namespace ds = maps::wiki::mds_dataset;

using ExportDatasetWriter = ds::DatasetWriter<ds::ExportMetadata>;

namespace maps {
namespace wiki {
namespace exporter {

extern std::string makeTaskTag(uint64_t taskId, const std::string& branch, revision::DBID commitId);


namespace test {

namespace {

const std::string LOCK = "lock";
const std::string SERVICE_CONFIG = "./tests/services.local.export-tests.xml";
const std::string JSON_OBJECTS_PATH = "./tests/json/1.json";
const std::string MIGRATION_PATH = "/usr/lib/migrations/mapspro/versions";
const std::string MIGRATIONS_UPGRADE_PATTERN = ".*_upgrade.sql";

const std::string JSON_FILE = "json";
const std::string JSON_TAR_GZ = "json.tar.gz";
const std::string DUMP_TAR_GZ = "ymapsdf2.dump.tar.gz";
const std::string SHAPE_TAR_GZ = "shape.tar.gz";
const std::string ERROR_SHAPE_FILE = "error.shp";
const std::string MD5_SUMS = "md5sums";

const std::string CAT_AOI = "aoi";
const std::string CAT_BLD = "bld";
const std::string CAT_ERROR = "error";
const std::string CAT_RD_EL = "rd_el";

const int USER_ID = 84277110; // miplot
constexpr int MAX_TASK_ID = 1000000;
constexpr rev::DBID TRUNK_BRANCH = 0;
const std::string TAG = "grinder-export-worker";

const std::string MDS_DATASET_PATH_DEFAULT = "export";


std::string getCfgParam(
    const common::ExtendedXmlDoc& cfg,
    const std::string& path,
    const std::string& attrName)
{
    return cfg.getAttr<std::string>(tasks::getConfigExportXpath() + path, attrName);
}


std::string getCfgParam(
    const common::ExtendedXmlDoc& cfg,
    const std::string& path,
    const std::string& attrName,
    const std::string& defaultValue)
{
    return cfg.getAttr<std::string>(tasks::getConfigExportXpath() + path, attrName, defaultValue);
}


std::vector<std::string> readLines(const std::string& fname)
{
    std::ifstream in(fname);
    std::string line;
    std::vector<std::string> result;

    while (std::getline(in, line)) {
        if (!line.empty()) {
            result.push_back(std::move(line));
        }
    }
    return result;
}


void createDirs(const std::vector<fs::path>& paths)
{
    for (const auto& path : paths) {
        fs::create_directories(path);
        fs::permissions(path, fs::add_perms
                            | fs::owner_write
                            | fs::group_write
                            | fs::others_write);
    }
}


void removeDirs(const std::vector<fs::path>& paths)
{
    for (const auto& path : paths) {
        fs::remove_all(path);
    }
}


rev::DBID getHeadCommitId(maps::pgpool3::Pool& pool)
{
    auto txn = pool.masterReadOnlyTransaction();
    rev::RevisionsGateway gtw(*txn);
    return gtw.headCommitId();
}


void importObjects(const std::string& jsonObjectsPath)
{
    std::ostringstream cmd;
    cmd << "revisionapi"
        << " --ignore-json-ids"
        << " --cfg=" << SERVICE_CONFIG
        << " --cmd=import"
        << " --branch=0"
        << " --path=" << jsonObjectsPath
        << " --user-id=" << USER_ID;

    auto res = shell::runCmd(cmd.str());
    REQUIRE(res.exitCode == 0, "Can't import test objects to db: "
            << res.stdErr);
}


void setGeometry(pgpool3::Pool& pgPool, rev::DBID objId, const std::string& wkb)
{
    auto txn = pgPool.masterWriteableTransaction();
    std::ostringstream query;
    query << "UPDATE revision.geometry as g\n"
          << "SET contents = '" << wkb << "'\n"
          << "FROM revision.object_revision o\n"
          << "WHERE g.id = o.geometry_id and o.object_id = " << objId;
    txn->exec(query.str());
    txn->commit();
}


boost::test_tools::predicate_result
matches(const std::string& url, const std::string& pattern)
{
    boost::test_tools::predicate_result result =
        boost::regex_match(url, boost::regex{pattern});
    if (!result) {
        result.message() << "URL: '" << url << "' does not match to the pattern: /" << pattern << "/.";
    }
    return result;
}

} // namespace


class ExportFixture : public unittest::DatabaseFixture {
public:
    ExportFixture()
        : unittest::DatabaseFixture(SERVICE_CONFIG, TAG)
        , mdsClient(makeMdsConfig(configXml()))
        , taskParams(
            std::chrono::system_clock::now().time_since_epoch().count() % MAX_TASK_ID,
            0,
            std::to_string(TRUNK_BRANCH),
            Subset::Ymapsdf,
            IsTested::Yes
        )
    {
        clearDb();
        importObjects(JSON_OBJECTS_PATH);
        headCommitId = getHeadCommitId(pool());
        taskParams.commitId = headCommitId;
        readConfigParams();

        exportPgPoolPtr = createPgPool(CONN_PARAMS);
        createDirs({TMP_BASE_DIR,
                    FILESYSTEM_BASE_DIR,
                    BACKUP_BASE_DIR,
                    RESULTS_BASE_DIR});

    }

    virtual ~ExportFixture()
    {
        try {
            removeDirs({
                TMP_BASE_DIR,
                FILESYSTEM_BASE_DIR,
                BACKUP_BASE_DIR,
                RESULTS_BASE_DIR});
        } catch (const std::exception& e) {
            ERROR() << "Failed to remove tmp dirs: " << e.what();
        }
    }


    rev::DBID headCommitId;
    std::shared_ptr<pgpool3::Pool> exportPgPoolPtr;
    mds::Mds mdsClient;

    std::string TMP_BASE_DIR;
    std::string FILESYSTEM_BASE_DIR;
    std::string BACKUP_BASE_DIR;
    std::string RESULTS_BASE_DIR;
    std::string RESULT_BASE_URI;
    std::string MDS_READ_URL_PATTERN_PREFIX;
    std::string MDS_DATASET_PATH;
    PgPool3ConnParams CONN_PARAMS;

    TaskParams taskParams;


private:
    void clearDb()
    {
        auto txn = pool().masterWriteableTransaction();
        auto revGateway = rev::RevisionsGateway(
            *txn, revision::BranchManager(*txn).load(TRUNK_BRANCH));
        revGateway.truncateAll();
        revGateway.createDefaultBranches();
        txn->commit();
    }

    void readConfigParams()
    {
        TMP_BASE_DIR = getCfgParam(configXml(), "", "tmp-base-dir");
        BACKUP_BASE_DIR = getCfgParam(configXml(), "", "json-backup-dir");
        FILESYSTEM_BASE_DIR = getCfgParam(configXml(), "/result", "filesystem-base");
        auto resultSubdir = getCfgParam(configXml(), "/result", "tmp-path");

        RESULTS_BASE_DIR = FILESYSTEM_BASE_DIR + "/" + resultSubdir;

        MDS_READ_URL_PATTERN_PREFIX = (
            boost::format("http://%1%:%2%/get-%3%/\\d+/")
                % getCfgParam(configXml(), "/mds", "host")
                % getCfgParam(configXml(), "/mds", "read-port")
                % getCfgParam(configXml(), "/mds", "namespace-name")).str();

        MDS_DATASET_PATH = getCfgParam(configXml(), "/mds", "dataset-path", MDS_DATASET_PATH_DEFAULT);

        auto pgPool2ConnStr = getCfgParam(configXml(), "", "conn-str");
        CONN_PARAMS = makePgPool3ConnParams(pgPool2ConnStr);
    }
};


BOOST_AUTO_TEST_CASE(dir_lock_unlock)
{
    auto dir = fs::temp_directory_path() /
               fs::unique_path("wiki-export-test-lock-%%%%%%%%");
    concurrent::ScopedGuard tmpDirGuard([&]{ fs::remove_all(dir); });
    fs::create_directories(dir);


    DirLock dirLock(dir.string(), LOCK);
    fs::path lockPath(dir.string() + "/" + LOCK);

    BOOST_CHECK(dirLock.tryLock());
    BOOST_CHECK(fs::exists(lockPath));
    dirLock.unlock();
    BOOST_CHECK(!fs::exists(lockPath));
}


BOOST_AUTO_TEST_CASE(dir_double_lock_unlock)
{
    auto dir = fs::temp_directory_path() /
               fs::unique_path("wiki-export-test-lock-%%%%%%%%");
    concurrent::ScopedGuard tmpDirGuard([&]{ fs::remove_all(dir); });
    fs::create_directories(dir);

    DirLock dirLock(dir.string(), LOCK);
    fs::path lockPath(dir.string() + "/" + LOCK);

    dirLock.tryLock();
    BOOST_CHECK(fs::exists(lockPath));

    DirLock dirLock2(dir.string(), LOCK);
    BOOST_CHECK(!dirLock2.tryLock());

    dirLock.unlock();
    BOOST_CHECK(!fs::exists(lockPath));
    BOOST_CHECK(dirLock2.tryLock());
    BOOST_CHECK(fs::exists(lockPath));
    dirLock2.unlock();
    BOOST_CHECK(!fs::exists(lockPath));

    // Repeated unlock does not cause an error
    dirLock2.unlock();

    // Unlocking a removed lock does not cause an error
    BOOST_CHECK(dirLock.tryLock());
    BOOST_CHECK(fs::exists(lockPath));
    fs::remove(lockPath);
    BOOST_CHECK(!fs::exists(lockPath));
    dirLock.unlock();
}


BOOST_AUTO_TEST_CASE(dir_does_not_exist)
{
    const std::string badDir = "./non/existing/dir";
    DirLock dirLock(badDir, LOCK);
    fs::path lockPath(badDir + "/" + LOCK);
    BOOST_CHECK(!fs::exists(lockPath));
    BOOST_REQUIRE_THROW(dirLock.tryLock(), DirNotFoundError);
}


BOOST_AUTO_TEST_CASE(make_task_tag)
{
    auto t = ::time(nullptr);
    struct tm* locTime = ::localtime(&t);

    std::ostringstream os;
    os << 1900 + locTime->tm_year
       << std::setfill('0') << std::setw(2) << 1 + locTime->tm_mon
       << std::setfill('0') << std::setw(2) << locTime->tm_mday;
    std::string date = os.str();


    BOOST_CHECK_EQUAL(
            makeTaskTag(1234, "trunk", 5),
            date + "_001234_trunk_5");

    BOOST_CHECK_EQUAL(
            makeTaskTag(1234, "0", 77),
            date + "_001234_0_77");

    BOOST_REQUIRE_THROW(
        makeTaskTag(12345,
            "a_loooong_string_longer_than_seventy_characters_"
            "in_place_of_the_branch_name", 5),
        maps::RuntimeError);
}


BOOST_AUTO_TEST_CASE(cleanup_old_dirs)
{
    auto baseDirPath = fs::temp_directory_path() /
                       fs::unique_path("wiki-export-test-cleanup-%%%%%%%%");
    fs::create_directories(baseDirPath);
    concurrent::ScopedGuard guard( [&]{fs::remove_all(baseDirPath);} );

    size_t numDirs = 6;
    size_t keepDirs = 2;
    std::vector<std::string> dirs;
    for (size_t i = 0; i < numDirs; ++i) {
        std::string dir = "dir_" + std::to_string(i);
        fs::create_directory(baseDirPath / dir);
        dirs.push_back(std::move(dir));
    }

    cleanupOldDirs(baseDirPath.string(), "dir_\\d", keepDirs);

    for (size_t i = 0; i < numDirs - keepDirs; ++i) {
        BOOST_CHECK(!fs::exists(baseDirPath / dirs[i]));
    }
    for (size_t i = numDirs - keepDirs; i < numDirs; ++i) {
        BOOST_CHECK(fs::exists(baseDirPath / dirs[i]));
    }
}


BOOST_FIXTURE_TEST_SUITE(main, ExportFixture)

BOOST_AUTO_TEST_CASE(should_get_export_exportFiles)
{
    const ExportConfig exportCfg(taskParams, configXml());
    const ExportFiles exportFiles(configXml(), exportCfg.tag(), exportCfg.subset());

    {
        const std::string root{"tests/tmp/wiki-export-unit-test-tmp"};
        const std::string work{root + "/export-task-" + exportCfg.tag()};
        BOOST_CHECK_EQUAL(exportFiles(TmpFile::ARCHIVES_DIR), work + "/archives");
        BOOST_CHECK_EQUAL(exportFiles(TmpFile::DUMP_TAR_GZ), work + "/archives/ymapsdf2.dump.tar.gz");
        BOOST_CHECK_EQUAL(exportFiles(TmpFile::DUMP_TAR_GZ_CHECKSUM), work + "/archives/ymapsdf2.dump.tar.gz.md5");
        BOOST_CHECK_EQUAL(exportFiles(TmpFile::DUMP_DIR), work + "/ymapsdf2.dump");
        BOOST_CHECK_EQUAL(exportFiles(TmpFile::JSON2YMAPSDF_LOG), work + "/json2ymapsdf_errors.txt");
        BOOST_CHECK_EQUAL(exportFiles(TmpFile::JSON_ARCHIVE), work + "/archives/json.tar.gz");
        BOOST_CHECK_EQUAL(exportFiles(TmpFile::JSON_DIR), work + "/json");
        BOOST_CHECK_EQUAL(exportFiles(TmpFile::PRINTED_CFG), work + "/printed-config.json");
        BOOST_CHECK_EQUAL(exportFiles(TmpFile::ROOT_DIR), root);
        BOOST_CHECK_EQUAL(exportFiles(TmpFile::SERVICE_JSON_FILE), work + "/json");
        BOOST_CHECK_EQUAL(exportFiles(TmpFile::SERVICE_SHAPE_DIR), work + "/shape");
        BOOST_CHECK_EQUAL(exportFiles(TmpFile::WORK_DIR), work);
    }

    {
        const std::string root{"tests/tmp/wiki-export-unit-test/tmp-results"};
        const std::string work{root + "/" + exportCfg.tag()};
        BOOST_CHECK_EQUAL(exportFiles(ResultFile::CHECKSUMS_FILE, NO_REGION), work + "/md5sums");
        BOOST_CHECK_EQUAL(exportFiles(ResultFile::DUMP_TAR_GZ, NO_REGION), work + "/ymapsdf2.dump.tar.gz");
        BOOST_CHECK_EQUAL(exportFiles(ResultFile::DUMP_GZ_TAR, NO_REGION), work + "/ymapsdf2.dump.gz.tar");
        BOOST_CHECK_EQUAL(exportFiles(ResultFile::ROOT_DIR, NO_REGION), root);
        BOOST_CHECK_EQUAL(exportFiles(ResultFile::SERVICE_JSON_ARCHIVE, NO_REGION), work + "/json.tar.gz");
        BOOST_CHECK_EQUAL(exportFiles(ResultFile::SERVICE_SHAPE_ARCHIVE, NO_REGION), work + "/shape.tar.gz");
        BOOST_CHECK_EQUAL(exportFiles(ResultFile::WORK_DIR, NO_REGION), work);
    }

    {
        const Region region{"russia"};
        const std::string root{"tests/tmp/wiki-export-unit-test/tmp-results"};
        const std::string work{root + "/" + region + "-" + exportCfg.tag()};
        BOOST_CHECK_EQUAL(exportFiles(ResultFile::CHECKSUMS_FILE, region), work + "/md5sums");
        BOOST_CHECK_EQUAL(exportFiles(ResultFile::DUMP_TAR_GZ, region), work + "/ymapsdf2.dump.tar.gz");
        BOOST_CHECK_EQUAL(exportFiles(ResultFile::DUMP_GZ_TAR, region), work + "/ymapsdf2.dump.gz.tar");
        BOOST_CHECK_EQUAL(exportFiles(ResultFile::ROOT_DIR, region), root);
        BOOST_CHECK_EQUAL(exportFiles(ResultFile::SERVICE_JSON_ARCHIVE, region), work + "/json.tar.gz");
        BOOST_CHECK_EQUAL(exportFiles(ResultFile::SERVICE_SHAPE_ARCHIVE, region), work + "/shape.tar.gz");
        BOOST_CHECK_EQUAL(exportFiles(ResultFile::WORK_DIR, region), work);
    }

    {
        const std::string root{"tests/tmp/wiki-export-unit-test-backup"};
        const std::string work{root + "/" + exportCfg.tag()};
        BOOST_CHECK_EQUAL(exportFiles(BackupFile::JSON_ARCHIVE), work + "/json.tar.gz");
        BOOST_CHECK_EQUAL(exportFiles(BackupFile::ROOT_DIR), root);
        BOOST_CHECK_EQUAL(exportFiles(BackupFile::WORK_DIR), work);
    }
}


BOOST_AUTO_TEST_CASE(test_export_domain_objects)
{
    ExportConfig exportCfg(taskParams, configXml());
    const ExportFiles exportFiles(configXml(), exportCfg.tag(), exportCfg.subset());

    auto tag = exportCfg.tag();
    auto tmpDirPath = fs::path(exportFiles(TmpFile::WORK_DIR));
    auto backupDirPath = fs::path(exportFiles(BackupFile::WORK_DIR));

    std::vector<fs::path> resultsDirPaths;
    for (const auto& region: exportCfg.regions()) {
        resultsDirPaths.push_back(fs::path(exportFiles(ResultFile::WORK_DIR, region)));
    }

    concurrent::ScopedGuard cleanupGuard([&]{
        removeDirs({tmpDirPath, backupDirPath});
        removeDirs(resultsDirPaths);
        auto schemaPrefix = getCfgParam(configXml(), "", "schema-prefix");
        dropSchema(*exportPgPoolPtr, schemaPrefix + tag);
        ExportDatasetWriter datasetWriter(mdsClient, pool());
        datasetWriter.deleteDataset(tag);
    });

    auto datasets = runExport(exportCfg, exportFiles);

    // 1. Check export output URLs
    const auto dumpUrlPattern = MDS_READ_URL_PATTERN_PREFIX + "[a-z_]+/" + tag + "\\w?/([a-z_0-9]+/)?" + DUMP_TAR_GZ;
    BOOST_CHECK_EQUAL(datasets.size(), 3);
    for (const auto& dataset: datasets) {
        size_t checkSumCount = 0;
        size_t dumpCount = 0;
        for (const auto& fileLink : dataset.fileLinks()) {
            if (fileLink.name() == DUMP_TAR_GZ) {
                ++dumpCount;
                BOOST_CHECK(matches(fileLink.readingUrl(), dumpUrlPattern));
            } else if (fileLink.name() == CHECKSUMS_NAME) {
                ++checkSumCount;
            }
        }
        BOOST_CHECK_EQUAL(checkSumCount, 1);
        BOOST_CHECK_EQUAL(dumpCount, 1);
    }

    // 2a. Check files in the export directories
    BOOST_CHECK(fs::exists(backupDirPath / JSON_TAR_GZ));

    for (const auto& resultsDirPath: resultsDirPaths) {
        // 2b. Check files in the export directories
        BOOST_CHECK(fs::exists(resultsDirPath / DUMP_TAR_GZ));
        BOOST_CHECK(fs::exists(resultsDirPath / MD5_SUMS));

        // 3. Check content of the service objects shape file
        auto res = shell::runCmd("cd " + resultsDirPath.string() +
                                 " && tar -zxvf " + DUMP_TAR_GZ);
        BOOST_ASSERT_MSG(res.exitCode == 0, "Can't extract dump archive");

        const std::map<Region, std::map<std::string, size_t>> regions2numObjectsPerCategory {
            {"russia", {{CAT_BLD, 6}, {CAT_RD_EL, 2}}},
            {"and_tr", {}},
            {"turkey_mpro", {}}
        };

        for (const auto& region2numObjectsPerCategory: regions2numObjectsPerCategory) {
            const auto& region = region2numObjectsPerCategory.first;
            const auto& numObjectsPerCategory = region2numObjectsPerCategory.second;

            if (resultsDirPath.string().find(region) == std::string::npos) {
                continue;
            }

            for (const auto& catAndCount : numObjectsPerCategory) {
                auto filePath = resultsDirPath / catAndCount.first;
                BOOST_CHECK(fs::exists(filePath));
                auto items = readLines(filePath.string());
                BOOST_CHECK_EQUAL(items.size(), catAndCount.second);
            }
        }
    }
}


BOOST_AUTO_TEST_CASE(test_export_service_objects)
{
    taskParams.subset = Subset::Service;

    // Set self-intersected geometry for a service object
    setGeometry(pool(), 3000,
        "0103000020430D0000010000000900000095030A3D61FA4941C6ABDDD1F7B05F"
        "4139D0422361FA4941049523F5F7B05F4136DF7C4E62FA49414DC0B25BF6B05F"
        "41F1EC858660FA4941452BAA7BF4B05F41640D565A5DFA494110205CE0F3B05F"
        "414776E61F5AFA4941452BAA7BF4B05F41FF83EF5758FA49414DC0B25BF6B05F"
        "41FD92298359FA4941049523F5F7B05F4195030A3D61FA4941C6ABDDD1F7B05F41"
    );

    auto tag = makeTaskTag(taskParams.taskId, taskParams.branch, headCommitId);
    auto tmpDirPath = fs::path(TMP_BASE_DIR) / (TMP_DIR_PREFIX + tag);
    auto backupDirPath = fs::path(BACKUP_BASE_DIR) / tag;
    auto resultsDirPath = fs::path(RESULTS_BASE_DIR) / tag;

    concurrent::ScopedGuard cleanupGuard([&]{
        removeDirs({tmpDirPath, backupDirPath, resultsDirPath});
        ExportDatasetWriter datasetWriter(mdsClient, pool());
        datasetWriter.deleteDataset(tag);
    });

    ExportConfig exportCfg(taskParams, configXml());
    const ExportFiles exportFiles(configXml(), exportCfg.tag(), exportCfg.subset());

    auto datasets = runExport(exportCfg, exportFiles);

    // 1. Check export output URLs
    BOOST_ASSERT(datasets.size() == 1);
    size_t jsonCount = 0;
    size_t shapeCount = 0;
    const auto jsonUrlPatter = MDS_READ_URL_PATTERN_PREFIX + MDS_DATASET_PATH + "/" + tag + "/" + JSON_TAR_GZ;
    const auto shapeUrlPatter = MDS_READ_URL_PATTERN_PREFIX + MDS_DATASET_PATH + "/" + tag + "/" + SHAPE_TAR_GZ;
    for (const auto& fileLink : datasets.front().fileLinks()) {
        if (fileLink.name() == JSON_TAR_GZ) {
            ++jsonCount;
            BOOST_CHECK(matches(fileLink.readingUrl(), jsonUrlPatter));
        } else if (fileLink.name() == SHAPE_TAR_GZ) {
            ++shapeCount;
            BOOST_CHECK(matches(fileLink.readingUrl(), shapeUrlPatter));
        }
    }
    BOOST_CHECK_EQUAL(jsonCount, 1);
    BOOST_CHECK_EQUAL(shapeCount, 1);

    // 2. Check files in the export directories
    BOOST_CHECK(fs::exists(resultsDirPath / JSON_TAR_GZ));
    BOOST_CHECK(fs::exists(resultsDirPath / SHAPE_TAR_GZ));

    // 3. Check content of the service objects json file
    auto res = shell::runCmd("cd " + resultsDirPath.string() +
                             " && tar -zxvf " + JSON_TAR_GZ);
    BOOST_ASSERT_MSG(res.exitCode == 0, "Can't extract json archive");

    auto json = json::Value::fromFile((resultsDirPath / JSON_FILE).string());
    BOOST_ASSERT(json.hasField("objects"));
    const auto& objects = json["objects"];

    std::map<std::string, size_t> numObjectsPerCategory {
        { CAT_AOI, 0},
        { CAT_ERROR, 0}
    };

    BOOST_CHECK_EQUAL(objects.size(), 9);
    for (const auto& key : objects.fields()) {
        const auto& obj = objects[key];
        BOOST_CHECK(obj.hasField("geometry"));
        BOOST_ASSERT(obj.hasField("attributes"));
        const auto& attrs = obj["attributes"];

        for (auto& catAndCount : numObjectsPerCategory) {
            if (attrs.hasField("cat:" + catAndCount.first))
                ++catAndCount.second;
        }
    }
    BOOST_CHECK_EQUAL(numObjectsPerCategory[CAT_AOI], 2);
    BOOST_CHECK_EQUAL(numObjectsPerCategory[CAT_ERROR], 7);

    // 4. Check content of the service objects shape file
    res = shell::runCmd("cd " + resultsDirPath.string() +
                        " && tar -zxvf " + SHAPE_TAR_GZ);
    BOOST_ASSERT_MSG(res.exitCode == 0, "Can't extract shape archive");

    GDALAllRegister();
    auto *dataSource = OGRSFDriverRegistrar::Open(
        (resultsDirPath / ERROR_SHAPE_FILE).c_str(), FALSE );
    BOOST_ASSERT_MSG(dataSource != NULL, "Can't open shape file");

    auto* layer = dataSource->GetLayerByName("error");
    BOOST_ASSERT_MSG(layer != NULL, "Layer 'error' is missing");

    OGRFeature* feature;

    layer->ResetReading();
    std::set<std::string> jiraLinks;
    while((feature = layer->GetNextFeature()) != NULL) {
        BOOST_CHECK(feature->GetGeometryRef() != NULL);

        const char* jiraLink = feature->GetFieldAsString("jira");
        if (jiraLink != NULL)
            jiraLinks.insert(jiraLink);

        OGRFeature::DestroyFeature(feature);
    }

    std::set<std::string> expectedJiraLinks {
        "https://st.yandex-team.ru/MAPSCONTENT-942302",
        "https://st.yandex-team.ru/MAPSCONTENT-942273",
        "https://st.yandex-team.ru/MAPSCONTENT-1279063",
        "https://st.yandex-team.ru/MAPSCONTENT-172939",
        "https://st.yandex-team.ru/MAPSCONTENT-192910",
        "https://st.yandex-team.ru/MAPSCONTENT-1287473",
        "https://st.yandex-team.ru/MAPSCONTENT-1295147"
    };
    BOOST_CHECK_EQUAL_COLLECTIONS(jiraLinks.begin(), jiraLinks.end(),
        expectedJiraLinks.begin(), expectedJiraLinks.end());

    OGRDataSource::DestroyDataSource(dataSource);
}

BOOST_AUTO_TEST_SUITE_END()

} // test
} // exporter
} // wiki
} // maps
