#include "runtime_data.h"

#include <yandex/maps/wiki/common/retry_duration.h>
#include <yandex/maps/wiki/mds_dataset/dataset_gateway.h>
#include <yandex/maps/wiki/mds_dataset/export_metadata.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/validator/storage/message_attributes_filter.h>
#include <yandex/maps/wiki/validator/storage/results_gateway.h>

#include <maps/libs/http/include/http.h>

#include <boost/iostreams/filter/gzip.hpp>
#include <boost/iostreams/filtering_streambuf.hpp>
#include <boost/optional.hpp>

#include <set>
#include <sstream>
#include <unordered_set>

namespace ds = maps::wiki::mds_dataset;
namespace io = boost::iostreams;

namespace maps::wiki::validation_export {

namespace {

const std::string EXPORT_SUBSET = "domain";
const std::string OBJECT_IDS_ARCHIVE = "oids.txt.gz";

constexpr auto WAIT_TASKS_DURATION = std::chrono::hours(12);
constexpr int HTTP_STATUS_OK = 200;

const std::string SET_TESTED_REQUEST =
R"(<parameters xmlns="http://maps.yandex.ru/mapspro/tasks/1.x">
    <parameter key="tested" value="1"/>
</parameters>)";

std::unordered_set<tasks::ObjectId> readObjectIds(const std::string& url)
{
    http::Client httpClient;
    http::Request request(httpClient, http::GET, url);

    auto response = request.perform();
    REQUIRE(response.status() == HTTP_STATUS_OK,
        "Got wrong response status " << response.status() << " while retrieving url " << url);

    io::filtering_streambuf<io::input> in;
    in.push(io::gzip_decompressor());
    in.push(response.body());
    std::istream stream(&in);

    std::unordered_set<tasks::ObjectId> objectIds;
    while (!stream.eof()) {
        tasks::ObjectId id;
        stream >> id;
        if (stream.fail()) {
            break;
        }
        objectIds.insert(id);
    }
    INFO() << "Received " << objectIds.size() << " object ids from url " << url;
    return objectIds;
}

} // namespace

std::ostream& operator<<(std::ostream& stream, TaskStatus status)
{
    switch (status) {
    case TaskStatus::Success:
        stream << "задача завершилась штатно";
        break;
    case TaskStatus::Failed:
        stream << "задача завершилась аварийно";
        break;
    case TaskStatus::Unknown:
        stream << "неизвестен";
        break;
    }
    return stream;
}

StartedTasks RuntimeData::runTasks()
{
    auto existedObjectIds = common::retryDuration([&] {
        std::set<tasks::ObjectId> result;

        auto txn = coreDbPool.masterReadOnlyTransaction();
        auto branch = revision::BranchManager(*txn).load(branchId);
        revision::RevisionsGateway gateway(*txn, branch);
        auto snapshot = gateway.stableSnapshot(commitId);

        auto isObjectExists = [&](auto objectId) {
            auto rev = snapshot.objectRevision(objectId);
            return rev && !rev->data().deleted;
        };

        for (const auto& param : validationParams) {
            if (param.aoiId && isObjectExists(param.aoiId)) {
                result.emplace(param.aoiId);
            }
            if (param.regionId && isObjectExists(param.regionId)) {
                result.emplace(param.regionId);
            }
        }
        return result;
    });

    auto exportTask = taskManager.startExport(
        branchId, EXPORT_SUBSET, tasks::ExportTested::No, commitId);
    DUAL_INFO("Export task: " << exportTask.id());

    tasks::TaskInfos validationTasks;
    for (const auto& param : validationParams) {
        if (param.aoiId && !existedObjectIds.count(param.aoiId)) {
            DUAL_WARN("Skip validation task (no aoi): " << param.aoiId);
            continue;
        }
        if (param.regionId && !existedObjectIds.count(param.regionId)) {
            DUAL_WARN("Skip validation task (no region): " << param.regionId);
            continue;
        }
        auto aoiId = param.aoiId
            ? boost::optional<tasks::ObjectId>(param.aoiId)
            : tasks::NO_AOI;
        auto regionId = param.regionId
            ? boost::optional<tasks::ObjectId>(param.regionId)
            : tasks::NO_REGION;
        auto validationTask = taskManager.startValidation(
            branchId, param.presetId, aoiId, regionId, commitId);
        DUAL_INFO("Validation task: " << validationTask.id());
        validationTasks.push_back(std::move(validationTask));
    }

    return StartedTasks{exportTask, std::move(validationTasks)};
}

ValidationResult RuntimeData::waitForValidation(
    tasks::TaskInfos currentValidationTasks)
{
    ValidationResult result;

    while (!currentValidationTasks.empty()) {
        auto waitResult = taskManager.waitForTasks(
            currentValidationTasks, WAIT_TASKS_DURATION, checkCancel);

        tasks::TaskInfos childTasks;
        for (const auto& task : currentValidationTasks) {
            if (!waitResult.isSuccess(task.id())) {
                result.failedTaskIds.insert(task.id());
            } else {
                auto objectIds = common::retryDuration([&] {
                    return findObjectIdsWithFatalMessages(task);
                });
                if (!objectIds.empty()) {
                    result.fatalMessagesLinks.push_back(
                        nproFatalMessagesLink(task.id()));
                    result.objectIdsWithFatalMessages.insert(
                        objectIds.begin(), objectIds.end());
                }
            }

            childTasks.splice(childTasks.end(), findChildTasks(task.id()));
        }

        for (const auto& task : childTasks) {
           DUAL_INFO("Child validation task: " << task.id());
        }
        currentValidationTasks = std::move(childTasks);
    }

    return result;
}

TaskStatus RuntimeData::waitForExport(const tasks::TaskInfo& exportTask)
{
    auto waitResult = taskManager.waitForTasks(
        {exportTask}, WAIT_TASKS_DURATION, checkCancel);

    return waitResult.isSuccess(exportTask.id())
        ? TaskStatus::Success
        : TaskStatus::Failed;
}

TaskResult RuntimeData::checkExport(
    const tasks::TaskInfo& exportTask,
    const std::set<tasks::ObjectId>& objectIdsWithFatalMessages)
{
    auto result = checkValidationResult(exportTask.id(), objectIdsWithFatalMessages);
    if (result == TaskResult::ValidatedAndExported) {
        DUAL_INFO("All datasets are marked as tested");
        common::retryDuration([&] {
            taskManager.changeTaskParameters(exportTask.id(), SET_TESTED_REQUEST);
        });
    }
    return result;
}

std::set<tasks::ObjectId>
RuntimeData::findObjectIdsWithFatalMessages(const tasks::TaskInfo& task)
{
    validator::storage::MessageAttributesFilter filter;
    filter.severity = validator::Severity::Max;

    auto validationTxn = validatorDbPool.masterReadOnlyTransaction();

    validator::storage::ResultsGateway validatorGateway(*validationTxn, task.id());
    auto count = validatorGateway.messageCount(filter);
    if (!count) {
        return {};
    }

    auto coreTxn = coreDbPool.masterReadOnlyTransaction();

    auto branch = revision::BranchManager(*coreTxn).load(branchId);
    revision::RevisionsGateway revisionsGateway(*coreTxn, branch);
    auto snapshot = revisionsGateway.snapshot(commitId);

    std::set<tasks::ObjectId> objectIdsWithFatalMessages;

    auto messages = validatorGateway.messages(filter, snapshot, 0, count);
    for (const auto& messageDatum : messages) {
        for (const auto& revId : messageDatum.message().revisionIds()) {
            objectIdsWithFatalMessages.insert(revId.objectId());
        }
    }

    return objectIdsWithFatalMessages;
}

tasks::TaskInfos RuntimeData::findChildTasks(tasks::TaskId parentTaskId)
{
    auto childTaskIds = common::retryDuration([&] {
        auto txn = coreDbPool.masterReadOnlyTransaction();
        auto rows = txn->exec(
            "SELECT id FROM service.task"
            " WHERE parent_id = " + std::to_string(parentTaskId));

        std::set<tasks::TaskId> result;
        for (const auto& row : rows) {
            result.emplace(row[0].as<tasks::TaskId>());
        }
        return result;
    });

    return common::retryDuration([&] {
        tasks::TaskInfos taskInfos;
        for (auto childTaskId : childTaskIds) {
            taskInfos.push_back(taskManager.taskInfo(childTaskId));
        }
        return taskInfos;
    });
}

std::string RuntimeData::nproFatalMessagesLink(tasks::TaskId taskId) const
{
    std::ostringstream link;
    link << "https://"
         << nproHost
         << "/#!/tools/longtasks/"
         << taskId
         << "?branch=" << branchId
         << "&severity=fatal";
    return link.str();
}

TaskResult
RuntimeData::checkValidationResult(
    tasks::TaskId exportTaskId,
    const std::set<tasks::ObjectId>& objectIdsWithFatalMessages)
{
    if (objectIdsWithFatalMessages.empty()) {
        return TaskResult::ValidatedAndExported;
    }

    auto rows = common::retryDuration([&] {
        auto coreTxn = coreDbPool.masterReadOnlyTransaction();
        return coreTxn->exec(
            "SELECT dataset_id FROM service.export_result"
            " WHERE task_id = " + std::to_string(exportTaskId));
    });
    REQUIRE(!rows.empty(), "Failed to find dataset_id for export task " << exportTaskId);
    auto datasetId = rows[0][0].as<std::string>();

    auto datasets = common::retryDuration([&] {
        auto coreTxn = coreDbPool.masterReadOnlyTransaction();
        ds::ExportMetadata::FilterType filter(*coreTxn);
        filter.byId(datasetId);
        filter.byStatus(ds::DatasetStatus::Available);

        return ds::DatasetReader<ds::ExportMetadata>::datasets(*coreTxn, filter);
    });
    REQUIRE(!datasets.empty(), "No datasets found for export task " << exportTaskId);

    std::set<std::string> allRegions;
    std::set<std::string> errorRegions;
    for (const auto& dataset : datasets) {
        for (const auto& fileLink : dataset.fileLinks()) {
            if (fileLink.name() != OBJECT_IDS_ARCHIVE) {
                continue;
            }

            auto objectIds = common::retryDuration([&] {
                return readObjectIds(fileLink.readingUrl());
            });

            auto region = dataset.region();
            allRegions.emplace(region);
            for (auto id : objectIdsWithFatalMessages) {
                if (objectIds.count(id)) {
                    errorRegions.emplace(region);
                    break;
                }
            }
        }
    }
    if (errorRegions.empty()) {
        return TaskResult::ValidatedAndExported;
    }

    for (const auto& region : allRegions) {
        if (!errorRegions.count(region)) {
            DUAL_INFO("Mark dataset '" << datasetId << "' as tested in region '" << region << "'");
            common::retryDuration([&] {
                auto coreTxn = coreDbPool.masterWriteableTransaction();
                coreTxn->exec(
                    "UPDATE mds_dataset.export_meta SET tested = TRUE"
                    " WHERE id = " + coreTxn->quote(datasetId) +
                    " AND region = " + coreTxn->quote(region));
                coreTxn->commit();
            });
        } else {
            DUAL_ERROR("Dataset '" << datasetId << "' has fatal messages in region '" << region << "'");
        }
    }
    return TaskResult::FatalErrorsFound;
}

} // namespace maps::wiki::validation_export
