#include "worker.h"

#include <yandex/maps/wiki/common/retry_duration.h>
#include <yandex/maps/wiki/common/robot.h>
#include <yandex/maps/wiki/common/secrets.h>
#include <yandex/maps/wiki/common/string_utils.h>

#include <maps/libs/common/include/exception.h>

namespace worker = maps::grinder::worker;

namespace maps::wiki::validation_export {

namespace {

const std::string URL_XPATH = "/config/services/tasks/url";
const std::string VALIDATIONS_XPATH = "/config/services/tasks/" + TASK_NAME + "/validations";
const std::string VALIDATION_NODE = "validation";
const std::string AOI_ATTR = "aoi";
const std::string REGION_ATTR = "region";
const std::string PRESET_ID_ATTR = "preset-id";
const std::string RUN_SCHEDULED_TASK_ATTRIBUTE = "run_scheduled_task";

constexpr tasks::ObjectId NO_AOI = 0;
constexpr tasks::ObjectId NO_REGION = 0;

const std::string ST_QUEUE_KEY = "MAPSCONTENT";
const std::string ST_ISSUE_COMPONENT = "releases_export";
const std::string ST_ISSUE_SUMMARY = "Ошибка при экспорте из релизной ветки ";
const std::string ST_ISSUE_DESCRIPTION_VALIDATION = "Фатальные ошибки валидации: ";
const std::string ST_ISSUE_DESCRIPTION_FAILED_VALIDATION = "Упавшие задачи валидации: ";
const std::string ST_ISSUE_DESCRIPTION_VALIDATION_EXPORT = "Задача пакетного экспорта: ";
const std::string ST_ISSUE_DESCRIPTION_EXPORT = "Задача экспорта: ";
const std::string ST_ISSUE_DESCRIPTION_STATUS = ". Статус: ";

const std::string VALIDATED_AND_EXPORTED_RESULT = "validated_and_exported";
const std::string FATAL_ERRORS_FOUND_RESULT = "fatal_errors_found";

bool isTaskScheduled(pgpool3::Pool& pool)
{
    auto attributes = common::retryDuration([&]{
        auto txn = pool.masterReadOnlyTransaction();
        return tasks::attributesForTaskType(*txn, TASK_NAME);
    });

    auto it = attributes.find(RUN_SCHEDULED_TASK_ATTRIBUTE);
    return it != attributes.end() && it->second == "1";
}

void saveResult(
    pgpool3::Pool& pool,
    tasks::TaskId mainTaskId,
    TaskResult result)
{
    common::retryDuration([&]{
        auto txn = pool.masterWriteableTransaction();
        std::ostringstream sql;
        sql << "UPDATE service.validation_export_task"
            << " SET result = " << txn->quote(
                result == TaskResult::ValidatedAndExported
                    ? VALIDATED_AND_EXPORTED_RESULT
                    : FATAL_ERRORS_FOUND_RESULT)
            << " WHERE id = " << mainTaskId;
        txn->exec(sql.str());
        txn->commit();
    });
}

} // namespace

StarTrackIssueData::StarTrackIssueData(tasks::BranchId branchId, tasks::TaskId thisTaskId)
    : branchId(branchId)
    , thisTaskId(thisTaskId)
    , thisTaskStatus(TaskStatus::Failed)
    , exportTaskId(0)
    , exportTaskStatus(TaskStatus::Unknown)
{
}


Worker::Worker(const common::ExtendedXmlDoc& configXml)
    : coreDbHolder_(configXml, "core", "grinder")
    , validatorDbHolder_(configXml, "validation", "grinder")
    , tasksUrl_(configXml.get<std::string>(URL_XPATH))
    , stConfig_(configXml.getAttr<std::string>("/config/common/st", "api-base-url"),
                common::secrets::tokenByKey(common::secrets::Key::RobotWikimapStToken))
    , nproHost_(configXml.get<std::string>("/config/common/npro/host", {}))
{
    auto validationsNode = configXml.node(VALIDATIONS_XPATH, true);
    if (!validationsNode.isNull()) {
        auto validationNodes = validationsNode.nodes(VALIDATION_NODE, true);
        for (size_t i = 0; i < validationNodes.size(); ++i) {
            auto validation = validationNodes[i];
            auto aoiId = validation.attr<tasks::ObjectId>(AOI_ATTR, NO_AOI);
            auto regionId = validation.attr<tasks::ObjectId>(REGION_ATTR, NO_REGION);
            auto presetId = validation.attr<tasks::PresetId>(PRESET_ID_ATTR);
            validationParams_.push_back({aoiId, regionId, presetId});
        }
    }
}

void Worker::run(const worker::Task& grinderTask) const
{
    const auto& args = grinderTask.args();
    auto taskId = args["taskId"].as<tasks::TaskId>();
    auto uid = args["uid"].as<tasks::UserId>();
    auto branchId = args["branchId"].as<tasks::BranchId>();
    auto commitId = args["commitId"].as<tasks::CommitId>();

    tasks::TaskPgLogger logger(coreDbHolder_.pool(), taskId);
    DUAL_INFO("Task " << taskId << " started. " << "Grinder task ID: " << grinderTask.id());
    DUAL_INFO("Branch: " << branchId << ". Commit: " << commitId);

    StarTrackIssueData issueData(branchId, taskId);

    try {
        if (uid == common::ROBOT_UID && !isTaskScheduled(coreDbHolder_.pool())) {
            DUAL_INFO("Scheduled task start disabled");
            return;
        }

        tasks::TaskManager taskManager(tasksUrl_, uid);

        RuntimeData runtimeData{
            coreDbHolder_.pool(),
            validatorDbHolder_.pool(),
            validationParams_,
            branchId,
            commitId,
            logger,
            taskManager,
            [&grinderTask]() {
                if (grinderTask.isCanceled()) {
                    throw common::RetryDurationCancel();
                }
            },
            nproHost_};

        auto startedTasks = runtimeData.runTasks();
        issueData.exportTaskId = startedTasks.exportTask.id();

        issueData.validationResult = runtimeData.waitForValidation(
            std::move(startedTasks.validationTasks));

        issueData.exportTaskStatus = runtimeData.waitForExport(startedTasks.exportTask);
        if (issueData.exportTaskStatus == TaskStatus::Success
                && issueData.validationResult.failedTaskIds.empty()) {
            auto taskResult = runtimeData.checkExport(
                startedTasks.exportTask,
                issueData.validationResult.objectIdsWithFatalMessages);
            saveResult(coreDbHolder_.pool(), taskId, taskResult);
            if (taskResult == TaskResult::ValidatedAndExported) {
                DUAL_INFO("Task finished.");
                return;
            }
        }
        issueData.thisTaskStatus = TaskStatus::Success;
        createIssue(logger, issueData);
        DUAL_INFO("Task finished.");
    } catch (const common::RetryDurationCancel&) {
        DUAL_WARN("Task canceled.");
        throw worker::TaskCanceledException();
    } catch (const worker::TaskCanceledException&) {
        DUAL_WARN("Task canceled.");
        throw;
    } catch (const maps::Exception& e) {
        ERROR() << "Task failed: " << e;
        logger.logError() << "Task failed: " << e.what();
        createIssue(logger, issueData);
        throw;
    } catch (const std::exception& e) {
        DUAL_ERROR("Task failed: " << e.what());
        createIssue(logger, issueData);
        throw;
    }
}

void Worker::createIssue(
    tasks::TaskPgLogger& logger,
    const StarTrackIssueData& data) const
{
    if (!data.exportTaskId) {
        DUAL_ERROR("Export task is not created");
        return;
    }

    st::Gateway stGateway(stConfig_);

    st::IssuePatch issuePatch;
    issuePatch.summary().set(ST_ISSUE_SUMMARY + std::to_string(data.branchId));
    issuePatch.components().set({ST_ISSUE_COMPONENT});

    std::stringstream description;
    description << ST_ISSUE_DESCRIPTION_VALIDATION_EXPORT << data.thisTaskId << ST_ISSUE_DESCRIPTION_STATUS << data.thisTaskStatus;
    description << "\n\n";
    description << ST_ISSUE_DESCRIPTION_EXPORT << data.exportTaskId << ST_ISSUE_DESCRIPTION_STATUS << data.exportTaskStatus;

    if (!nproHost_.empty() && !data.validationResult.fatalMessagesLinks.empty()) {
        description << "\n\n";
        description << ST_ISSUE_DESCRIPTION_VALIDATION << '\n';
        description << common::join(data.validationResult.fatalMessagesLinks, '\n');
    }

    if (!data.validationResult.failedTaskIds.empty()) {
        description << "\n\n";
        description << ST_ISSUE_DESCRIPTION_FAILED_VALIDATION;
        description << common::join(data.validationResult.failedTaskIds, ", ");
    }

    INFO() << "Issue description:\n" << description.str();

    issuePatch.description().set(description.str());

    auto issue = stGateway.createIssue(
        ST_QUEUE_KEY,
        issuePatch,
        ST_ISSUE_COMPONENT + ":" + std::to_string(data.exportTaskId));

    DUAL_INFO("Issue created " << issue.key());
}

} // namespace maps::wiki::validation_export
