#include "../include/common.h"
#include "../include/task.h"
#include "../include/worker.h"
#include "building_lookup.h"
#include "building_searcher.h"

#include <maps/libs/sql_chemistry/include/gateway.h>
#include <maps/libs/sql_chemistry/include/gateway_access.h>
#include <maps/wikimap/mapspro/libs/editor_client/include/instance.h>
#include <maps/wikimap/mapspro/services/editor/src/configs/categories_strings.h>
#include <maps/wikimap/mapspro/services/editor/src/actions/tools/pedestrian/address_of_building.h>
#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/tasks/task_logger.h>
#include <yandex/maps/wiki/common/robot.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>
#include <yandex/maps/wiki/revision/common.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>

#include <boost/lexical_cast.hpp>

namespace mwc = maps::wiki::common;

namespace maps::wiki::guide_pedestrian {

namespace {

const std::string CONFIG_EDITOR_URL = "/config/services/editor/url";
const std::string YT_PROXY = "/config/services/tasks/guide_pedestrian/yt-proxy";
const std::string YT_ADDRESS_PATH = "/config/services/tasks/guide_pedestrian/yt-address-path";
const std::string YT_ENTRANCE_PATH = "/config/services/tasks/guide_pedestrian/yt-entrance-path";

using namespace sql_chemistry;

class TaskEntity {
public:
    int64_t id() const { return id_; }

    uint64_t paramsObjectId() const { return paramsObjectId_; }
    uint64_t paramsCommitId() const { return paramsCommitId_; }
    std::vector<geolib3::Point2> pointsAsArray() const {
        std::vector<geolib3::Point2> retVal;
        for (auto& pointValue : pointsJson_) {
            const auto& coord = pointValue["coordinates"];
            geolib3::Point2 geoPoint = coord.construct<geolib3::Point2>(0, 1);
            retVal.emplace_back(geolib3::convertGeodeticToMercator(geoPoint));
        }
        return retVal;
    }

    static TaskEntity fromTxn(
        pqxx::transaction_base& txnCore,
        revision::DBID dbTaskId);

private:
    int64_t id_{};
    uint64_t paramsObjectId_{};
    uint64_t paramsCommitId_{};
    json::Value pointsJson_{json::null};

    friend class sql_chemistry::GatewayAccess<TaskEntity>;
    TaskEntity() = default;

    template <typename T>
    static auto introspect(T& t) { return std::tie(t.id_, t.paramsObjectId_, t.paramsCommitId_, t.pointsJson_); }
};

struct TaskTbl : Table<TaskEntity> {
    static constexpr std::string_view name_{"service.guide_pedestrian_task"sv};

    static constexpr BigSerialKey id{"id"sv, name_};
    static constexpr UInt64Column paramsObjectId{"params_object_id"sv, name_};
    static constexpr UInt64Column paramsCommitId{"params_commit_id"sv, name_};
    static constexpr JsonColumn points{"points"sv, name_};

    auto columns_() const { return std::tie(id, paramsObjectId, paramsCommitId, points); }
};

using TaskEntityGateway = Gateway<TaskTbl>;

TaskEntity TaskEntity::fromTxn(
    pqxx::transaction_base& txnCore,
    revision::DBID dbTaskId)
{
    TaskEntityGateway gw{txnCore};
    auto taskEntity = gw.tryLoadById(dbTaskId);
    ASSERT(taskEntity);
    return std::move(*taskEntity);
}

void checkTaskIdAttribute(pgpool3::Pool& corePool, revision::DBID objectId)
{
    auto coreTxnHolder = corePool.slaveTransaction();
    pqxx::transaction_base& coreTxn = coreTxnHolder.get();
    revision::RevisionsGateway gtw(coreTxn);
    auto snapshot = gtw.snapshot(gtw.maxSnapshotId());
    auto revision = snapshot.objectRevision(objectId);
    REQUIRE(revision, "Object id='" << objectId << "' not found");
    auto attributes = revision->data().attributes;
    if (!attributes) {
        return;
    }
    REQUIRE(attributes->count(ATTR_PEDESTRIAN_REGION_TASK_ID) == 0,
        "Task has already created for this pedestrian region");
}

template <typename T>
T getValue(
    const std::unordered_map<std::string, std::string>& attrs,
    const std::string& key)
{
    auto taskTypeIt = attrs.find(key);
    REQUIRE(taskTypeIt != attrs.end(),
        "Attribute '" << key << "' expected");
    const std::string& typeString = taskTypeIt->second;
    return boost::lexical_cast<T>(typeString);
}

size_t guideRoutine(
    pgpool3::Pool& corePool,
    const TaskEntity& taskParams,
    const mwc::ExtendedXmlDoc& cfg)
{
    // check task_id attribute. todo: remove this check when HEREBEDRAGONS-169 will be done
    checkTaskIdAttribute(corePool, taskParams.paramsObjectId());

    // read cat:pedestrian_region object
    editor_client::Instance eClient(
        cfg.get<std::string>(CONFIG_EDITOR_URL), common::ROBOT_UID);
    auto paramsObject = eClient.getObjectById(taskParams.paramsObjectId());

    // check object
    REQUIRE(paramsObject.categoryId == CATEGORY_PEDESTRIAN_REGION,
        "Object of '" << CATEGORY_PEDESTRIAN_REGION << "' expected");
    REQUIRE(paramsObject.revisionId && (paramsObject.revisionId->commitId() == taskParams.paramsCommitId()),
        "object id='" << taskParams.paramsObjectId() << "' has wrong last commitId. "
            << "; commitId from task = " << taskParams.paramsCommitId());
    REQUIRE(paramsObject.plainAttributes.count(ATTR_PEDESTRIAN_REGION_TASK_ID) == 0,
        "Task has already created for this pedestrian region");

    const auto taskType = getValue<TaskType>(paramsObject.plainAttributes, ATTR_PEDESTRIAN_REGION_TASK_TYPE);

    auto paramsObjectGeometry = paramsObject.getGeometryInMercator();
    REQUIRE(paramsObjectGeometry && paramsObjectGeometry->geometryType() == geolib3::GeometryType::Polygon,
        "Polygon geometry expected at region object");
    auto& polygon = paramsObjectGeometry->get<geolib3::Polygon2>();

    // select all buildings
    mwc::PoolHolder viewTrunkPoolHolder(cfg, "view-trunk", "view-trunk");
    auto allBuildings = loadAllBuildings(polygon, viewTrunkPoolHolder);

    // make searcher: bldCenterPoint => bldId
    BuildingSearcher buildingSearcher(std::move(allBuildings));

    // match points with bldId
    AddressTasks addressTasks;
    EntranceTasks entranceTasks;
    std::set<revision::DBID> matchedBldId; // each bldId should occurs only ones
    chrono::TimePoint now = chrono::TimePoint::clock::now();
    std::string taskIdString = std::to_string(taskParams.id());
    for (const auto& point : taskParams.pointsAsArray()) {
        auto buildingData = buildingSearcher.search(point);
        REQUIRE(buildingData, "building not found for point = (" << point.x() << ", " << point.y() << ")");
        REQUIRE(matchedBldId.count(buildingData->objectId) == 0,
            "more than one point for buildingData " << buildingData->objectId);
        matchedBldId.insert(buildingData->objectId);

        geolib3::Point2 geoPoint = geolib3::convertMercatorToGeodetic(point);
        switch (taskType) {
            case TaskType::Addresses: {
                    AddressTask addressTask{
                        now,
                        geoPoint.x(),
                        geoPoint.y(),
                        std::to_string(buildingData->objectId),
                        taskIdString,
                        buildingData->polygonMerc
                    };
                    addressTasks.emplace_back(std::move(addressTask));
                }
                break;
            case TaskType::Entrances: {
                    auto viewTrunkTxnHolder = viewTrunkPoolHolder.pool().slaveTransaction();
                    auto addressString = pedestrian::getAddressOfBuilding(
                        buildingData->polygonMerc, viewTrunkTxnHolder.get());
                    EntranceTask entranceTask{
                        now,
                        geoPoint.x(),
                        geoPoint.y(),
                        std::to_string(buildingData->objectId),
                        taskIdString,
                        std::move(addressString),
                        buildingData->polygonMerc
                    };
                    entranceTasks.emplace_back(std::move(entranceTask));
                }
                break;
        }
    }

    // send tasks into YT
    const auto ytProxy = cfg.get<std::string>(YT_PROXY);
    enqueueAddressTasks(addressTasks, ytProxy, cfg.get<std::string>(YT_ADDRESS_PATH));
    enqueueEntranceTasks(entranceTasks, ytProxy, cfg.get<std::string>(YT_ENTRANCE_PATH));

    // set pedestrian_region:task_id for object
    paramsObject.plainAttributes[ATTR_PEDESTRIAN_REGION_TASK_ID] = std::to_string(taskParams.id());
    eClient.saveObject(paramsObject);

    return addressTasks.size() + entranceTasks.size();
}

} // unnamed namespace


Worker::Worker(std::unique_ptr<maps::wiki::common::ExtendedXmlDoc> cfg)
    : cfg_(std::move(cfg))
{}

void Worker::doTask(const grinder::worker::Task& task)
try {
    common::PoolHolder corePool(*cfg_, "core", "grinder");

    auto dbTaskId = task.args()["taskId"].as<revision::DBID>();
    tasks::TaskPgLogger logger(corePool.pool(), dbTaskId);
    logger.logInfo() << "Guide pedestrian started; task id = " << dbTaskId;

    try {
        auto taskParams = TaskEntity::fromTxn(
            *corePool.pool().slaveTransaction(),
            dbTaskId);

        auto pointsCount = guideRoutine(corePool.pool(), taskParams, *cfg_);
        logger.logInfo() << "Successfully created " << pointsCount << " tasks for pedestrian";
    } catch (const std::exception& ex) {
        logger.logError() << ex.what();
        throw;
    }
} catch (const maps::Exception& ex) {
    ERROR() << "Task ERROR: " << ex;
    throw;
} catch (const std::exception& ex) {
    ERROR() << "Task ERROR: " << ex.what();
    throw;
} catch (...) {
    ERROR() << "Task ERROR";
}

} // namespace maps::wiki::guide_pedestrian
