#include "user_mail_info.h"

#include <maps/libs/common/include/exception.h>

#include <maps/libs/locale/include/convert.h>

namespace maps::wiki::notifications_sender {

namespace {

class SocialProfileFacade
{
public:
    SocialProfileFacade(const social::IProfileGateway& gtw, social::TUid puid) :
        gtw_(gtw), puid_(puid)
    {}

    std::string getEmail() const
    {
        return getOptProfile() ? getOptProfile()->email() : "";
    }

    std::string getLocale() const
    {
        return getOptProfile() ? getOptProfile()->locale() : "";
    }

private:
    using OptProfile = std::optional<social::Profile>;

    const OptProfile& getOptProfile() const
    {
        if (!profileCache_) {
            profileCache_ = gtw_.getUserProfile(puid_);
        }
        return profileCache_.value();
    }

    mutable std::optional<OptProfile> profileCache_;
    const social::IProfileGateway& gtw_;
    social::TUid puid_;
};


class BlackboxProfileFacade
{
public:
    BlackboxProfileFacade(const blackbox::IGateway& gtw, social::TUid puid) :
        gtw_(gtw), puid_(puid)
    {}

    std::string getEmail() const
    {
        return getOptUserInfo() ? getOptUserInfo()->email() : "";
    }

    std::string getUserName() const
    {
        return getOptUserInfo()
            ? getOptUserInfo()->username()
                ? getOptUserInfo()->username().value()
                : ""
            : "";
    }

    std::string getLocale() const
    {
        const auto& optUserInfo = getOptUserInfo();
        return optUserInfo
            ? optUserInfo->locale()
                ? locale::toString(optUserInfo->locale()->language())
                : ""
            : "";
    }

private:
    using OptUserInfo = std::optional<blackbox::UserInfo>;

    const OptUserInfo& getOptUserInfo() const
    {
        if (!userInfoCache_) {
            userInfoCache_ = gtw_.defaultUserInfo(puid_);
        }
        return userInfoCache_.value();
    }

    mutable std::optional<OptUserInfo> userInfoCache_;
    const blackbox::IGateway& gtw_;
    social::TUid puid_;
};

} // unnamed namespace

UserMailInfo::UserMailInfo(
    social::TUid puid,
    std::string email,
    std::string userName,
    std::string locale)
        : puid_(puid)
        , email_(std::move(email))
        , userName_(std::move(userName))
        , locale_(std::move(locale))
{
    ASSERT(puid_);
    ASSERT(!email_.empty());
    ASSERT(!userName_.empty());
    ASSERT(!locale_.empty());
}

social::TUid UserMailInfo::puid() const
{
    return puid_;
}

const std::string& UserMailInfo::email() const
{
    return email_;
}

const std::string& UserMailInfo::userName() const
{
    return userName_;
}


const std::string& UserMailInfo::locale() const
{
    return locale_;
}


UserMailInfoProvider::UserMailInfoProvider(
    const social::IProfileGateway& socialProfileGtw,
    const blackbox::IGateway& blackboxGtw)
        : socialProfileGtw_(socialProfileGtw)
        , blackboxGtw_(blackboxGtw)
{}

std::optional<UserMailInfo>
UserMailInfoProvider::getUserMailInfo(social::TUid puid)
{
    SocialProfileFacade socialProfileFacade(socialProfileGtw_, puid);
    BlackboxProfileFacade blackboxProfileFacade(blackboxGtw_, puid);

    std::string email = socialProfileFacade.getEmail().empty()
        ? blackboxProfileFacade.getEmail()
        : socialProfileFacade.getEmail();

    std::string locale = socialProfileFacade.getLocale().empty()
        ? blackboxProfileFacade.getLocale()
        : socialProfileFacade.getLocale();

    std::string userName = blackboxProfileFacade.getUserName();

    if (email.empty() || userName.empty() || locale.empty()) {
        return std::nullopt;
    }

    return UserMailInfo(puid, std::move(email), std::move(userName), std::move(locale));
}

} // namespace maps::wiki::notifications_sender
