#include "worker.h"
#include "consts.h"
#include "parsed_message.h"
#include "sender_mailing.h"
#include "user_mail_info.h"

#include <aws/sqs/model/DeleteMessageRequest.h>
#include <aws/sqs/model/ReceiveMessageRequest.h>

#include <yandex/maps/wiki/social/common.h>
#include <yandex/maps/wiki/social/profile_gateway.h>
#include <yandex/maps/wiki/social/sent_notification.h>
#include <yandex/maps/wiki/social/sent_notification_gateway.h>

#include <maps/libs/auth/include/tvm.h>
#include <maps/libs/log8/include/log8.h>

#include <maps/wikimap/mapspro/libs/sender/include/email_template_params.h>
#include <maps/wikimap/mapspro/libs/sender/include/gateway.h>
#include <maps/wikimap/mapspro/libs/sqs_client/include/configuration.h>
#include <maps/wikimap/mapspro/libs/sqs_client/include/client.h>

namespace maps::wiki::notifications_sender {

namespace {

const std::string SERVICES_CONFIG_BLACKBOX_HOST = "/config/common/blackbox-url";

const std::string NOTIFICATIONS_QUEUE_NAME = "notifications";

const int MAX_NUMBER_OF_MESSAGES_TO_RECEIVE = 10;
const int QUEUE_LONG_POLLING_TIME_SEC = 10;

const int64_t ADDRESSES_EMAIL_PERIOD_DAYS = 30;


template<typename F>
void noExceptExecution(const std::string& processName, F func)
{
    try {
        func();
    } catch (const Exception& ex) {
        ERROR() << processName << " failed: " << ex;
    } catch (const std::exception& ex) {
        ERROR() << processName << " failed: " << ex.what();
    } catch (...) {
        ERROR() << processName << " failed: unknown exception";
    }
}

social::NotificationType notificationTypeFromMailType(const std::string& mailType)
{
    if (mailType == MAIL_TYPE_NEWS_SUBSCRIPTION_WELCOME) {
        return social::NotificationType::WelcomeToService;
    } else if (mailType == MAIL_TYPE_ACHIEVE_EDITS_COUNT) {
        return social::NotificationType::AchievementEditsCount;
    } else if (mailType == MAIL_TYPE_ADDRESSES_SHOWS_COUNT) {
        return social::NotificationType::AddressesShowsCount;
    }
    throw RuntimeError() << "Unknown mail type " << mailType;
}

json::Value notificationArgsFromMessage(const ParsedMessage& parsedMessage)
{
    if (parsedMessage.mailType == MAIL_TYPE_ACHIEVE_EDITS_COUNT) {
        return parsedMessage.args;
    } else {
        return json::Value(json::repr::ObjectRepr{});
    }
}

} // namespace unnamed


Worker::Worker(
    const common::ExtendedXmlDoc& servicesConfig,
    const sender::Config& senderConfig)
    : senderConfig_(senderConfig)
    , socialDbPoolHolder_(servicesConfig, "social", "grinder")
    , sqsConfig_(servicesConfig)
    , sqsClient_(sqs::createSqsClient(sqsConfig_))
    , tvmClient_(auth::TvmtoolSettings().selectClientAlias("maps-core-nmaps-tasks-feedback").makeTvmClient())
    , blackboxClient_(
          blackbox::Configuration(
              servicesConfig.get<std::string>(SERVICES_CONFIG_BLACKBOX_HOST)
          ),
          maps::common::RetryPolicy(),
          [&]() {
              return tvmClient_.GetServiceTicketFor("blackbox");
          }
      )
{}

void Worker::doCycleNoExcept()
{
    noExceptExecution(
        "Worker cycle",
        [&](){ doCycle(); }
    );
}

void Worker::doCycle()
{
    INFO() << "Starting work cycle";

    const auto& queueName = NOTIFICATIONS_QUEUE_NAME;

    Aws::SQS::Model::ReceiveMessageRequest receiveRequest;
    receiveRequest.SetQueueUrl(sqsConfig_.getQueueUrl(queueName));
    receiveRequest.SetMaxNumberOfMessages(MAX_NUMBER_OF_MESSAGES_TO_RECEIVE);
    receiveRequest.SetWaitTimeSeconds(QUEUE_LONG_POLLING_TIME_SEC);

    auto receive = sqsClient_.ReceiveMessage(receiveRequest);
    if (!receive.IsSuccess()) {
        ERROR() << "Unable to receive message from queue <" << queueName << ">. "
                << "Error message: " << receive.GetError().GetMessage();
        return;
    }

    const auto& messagesRaw = receive.GetResult().GetMessages();
    INFO() << "Recieved " << messagesRaw.size() << " messages";

    for (const auto& messageRaw : messagesRaw) {
        noExceptExecution(
            "Processing SQS message " + messageRaw.GetBody(),
            [&]() {
                // Processing message. In case of exceptional situation
                // message won't be deleted from queue and after several
                // attempts it will be automatically put in DLQ.
                //
                processRawMessage(messageRaw);

                // Delete message from queue.
                //
                Aws::SQS::Model::DeleteMessageRequest deleteRequest;
                deleteRequest.SetQueueUrl(sqsConfig_.getQueueUrl(queueName));
                deleteRequest.SetReceiptHandle(messageRaw.GetReceiptHandle());

                auto deleteResult = sqsClient_.DeleteMessage(deleteRequest);
                if (!deleteResult.IsSuccess()) {
                    ERROR() << "Unable to delete message from queue " << queueName << ". "
                            << "Error message: " << deleteResult.GetError().GetMessage();
                }
            }
        );
    }
}

bool noNotificationInLastPeriod(
    social::NotificationType type,
    social::TUid uid,
    int64_t periodDays,
    pqxx::transaction_base& socialTxn
) {
    social::SentNotificationGateway notificationsGateway(socialTxn);

    auto now = chrono::TimePoint::clock::now();
    auto notifications = notificationsGateway.load(
        social::table::SentNotificationTbl::uid == uid &&
        social::table::SentNotificationTbl::type == type
    );

    using days = std::chrono::duration<int64_t, std::ratio<3600 * 24>>;

    for (const auto& notification : notifications) {
        auto sentAgo = now - notification.sentAt;
        auto sentDaysAgo = std::chrono::duration_cast<days>(sentAgo).count();

        if (sentDaysAgo < periodDays) {
            return false;
        }
    }

    return true;
}

bool notificationAllowedInPeriod(
    const ParsedMessage& parsedMessage,
    pqxx::transaction_base& socialTxn
) {
    auto type = notificationTypeFromMailType(parsedMessage.mailType);

    if (type == social::NotificationType::AddressesShowsCount) {
        return noNotificationInLastPeriod(
            type, parsedMessage.puid, ADDRESSES_EMAIL_PERIOD_DAYS, socialTxn);
    } else {
        return true;
    }
}

void Worker::addToNotificationsStorage(const ParsedMessage& parsedMessage)
{
    auto socialTxn = socialDbPoolHolder_.pool().masterWriteableTransaction();
    social::SentNotificationGateway notificationsGateway(*socialTxn);

    social::SentNotification notification;
    notification.uid = parsedMessage.puid;
    notification.channel = social::NotificationChannel::Email;
    notification.type = notificationTypeFromMailType(parsedMessage.mailType);
    notification.args = notificationArgsFromMessage(parsedMessage);
    notification.sentAt = chrono::TimePoint::clock::now();

    notificationsGateway.insert(notification);
    socialTxn->commit();
}

void Worker::processRawMessage(const Aws::SQS::Model::Message& sqsMessage)
{
    INFO() << "Processing message " << sqsMessage.GetBody();
    auto message = parseMessage(sqsMessage);

    // Load user mail info
    //
    auto socialTxnReadHandle = socialDbPoolHolder_.pool().slaveTransaction();
    social::ProfileGateway socialProfileGateway(*socialTxnReadHandle);

    UserMailInfoProvider userMailInfoProvider(socialProfileGateway, blackboxClient_);
    auto userMailInfo = userMailInfoProvider.getUserMailInfo(message.puid);

    if (!userMailInfo) {
        ERROR() << "Sending mail to user " << message.puid << " skipped. "
                << "Reason: unable to load full user info";
        return;
    }

    if (!notificationAllowedInPeriod(message, *socialTxnReadHandle)) {
        ERROR() << "Sending mail to user " << message.puid << " skipped. "
                << "Reason: already sent in current period";
        return;
    }

    // Send mail to user
    //
    sender::Gateway senderClient(
        senderConfig_.endPoint,
        senderConfig_.credentials,
        maps::common::RetryPolicy()
    );

    SenderMailing senderMailing(senderConfig_.campaignSlugs);

    sender::CampaignSlug campaignSlug;
    try {
        campaignSlug = senderMailing.getCampaignSlug(message, userMailInfo.value());
    } catch (const NotExistingCampaignError& err) {
        ERROR() << "Sending mail to user " << message.puid << " skipped. "
                << "Reason: " << err;
        return;
    }

    auto sendResult = senderClient.sendToEmail(
        campaignSlug,
        userMailInfo->email(),
        senderMailing.getEmailParams(message, userMailInfo.value())
    );

    if (sendResult == sender::Result::Failed) {
        ERROR() << "Failed to send email for puid " << message.puid;
        return;
    }

    // Write to notification storage
    //
    noExceptExecution(
        "Adding to notifications storage",
        [&](){ addToNotificationsStorage(message); }
    );
}

} // namespace maps::wiki::notifications_sender
