
#include <yandex/maps/wiki/http/blackbox/blackbox.h>
#include <yandex/maps/wiki/social/profile_gateway.h>

#include <maps/libs/locale/include/locale.h>
#include <maps/libs/locale/include/codes.h>
#include <maps/wikimap/mapspro/services/tasks_feedback/src/notifications_sender/lib/user_mail_info.h>

#include <library/cpp/testing/unittest/registar.h>

namespace maps::wiki::notifications_sender::tests {

namespace {

const social::TUid TEST_UID = 1;

const auto SOCIAL_EMAIL = "so@ya.ru";
const auto SOCIAL_LOCALE_TEXT = "ru";

const auto BLACKBOX_EMAIL = "bb@ya.ru";
const auto BLACKBOX_LOCALE = locale::Locale(locale::Lang::En);
const auto BLACKBOX_LOCALE_TEXT = "en";

const auto USERNAME = "vasya";

bool operator == (const UserMailInfo& lhs, const UserMailInfo& rhs)
{
    return lhs.puid() == rhs.puid() &&
           lhs.email() == rhs.email() &&
           lhs.userName() == rhs.userName() &&
           lhs.locale() == rhs.locale();
}

// Stub implementations for all methods except `getUserProfile`
//
struct ProfileGatewayMidStub : social::IProfileGateway {
    social::Profiles getUserProfiles(const social::TUids&) const override { return {}; }
    social::Profiles getAllUserProfiles() const override { return {}; }
    std::optional<social::Profile> updateProfile(const social::TUid&, const social::ProfileOptionalFields&) override { return {}; }
    social::Profile insertProfile(const social::TUid&, const social::ProfileOptionalFields&) override { return {}; }
    social::Profile upsertProfile(const social::TUid&, const social::ProfileOptionalFields&) override { return {}; }
};

// Stub implementations for all methods except `defaultUserInfo`
//
struct BlackboxGatewayMidStub : blackbox::IGateway {
    bool isEmailValid(const std::string&, blackbox::UID) const override { return true; }
};

struct AbsentUserProfileGateway : ProfileGatewayMidStub {
    std::optional<social::Profile> getUserProfile(social::TUid) const override {
        return std::nullopt;
    }
};

struct AbsentUserBlackboxGateway : BlackboxGatewayMidStub {
     std::optional<blackbox::UserInfo> defaultUserInfo(blackbox::UID) const override {
         return std::nullopt;
     }
};

struct FullUserProfileGateway : ProfileGatewayMidStub {
    std::optional<social::Profile> getUserProfile(social::TUid uid) const override {
        return social::Profile(
            uid, "", SOCIAL_EMAIL, false, false, false,
            std::nullopt, SOCIAL_LOCALE_TEXT
        );
    }
};

struct FullUserBlackboxGateway : BlackboxGatewayMidStub {
     std::optional<blackbox::UserInfo> defaultUserInfo(blackbox::UID uid) const override {
         blackbox::UserInfo userInfo;
         userInfo.setUid(uid);
         userInfo.setEmail(BLACKBOX_EMAIL);
         userInfo.setUsername(USERNAME);
         userInfo.setLocale(BLACKBOX_LOCALE);
         return userInfo;
     }
};

} // unnamed namespace

Y_UNIT_TEST_SUITE(suite) {

Y_UNIT_TEST(no_social_profile__no_blackbox_userinfo)
{
    AbsentUserProfileGateway socialGtw;
    AbsentUserBlackboxGateway blackboxGtw;
    UserMailInfoProvider provider(socialGtw, blackboxGtw);

    UNIT_ASSERT_EQUAL(provider.getUserMailInfo(TEST_UID), std::nullopt);
}

Y_UNIT_TEST(no_social_profile__full_blackbox_userinfo)
{
    AbsentUserProfileGateway socialGtw;
    FullUserBlackboxGateway blackboxGtw;
    UserMailInfoProvider provider(socialGtw, blackboxGtw);

    auto expected = UserMailInfo(TEST_UID, BLACKBOX_EMAIL, USERNAME, BLACKBOX_LOCALE_TEXT);
    auto obtained = provider.getUserMailInfo(TEST_UID);

    UNIT_ASSERT(obtained != std::nullopt);
    UNIT_ASSERT_EQUAL(obtained.value(), expected);
}

Y_UNIT_TEST(no_social_profile__only_email_in_blackbox_userinfo)
{
    struct BlackboxGatewayImpl : BlackboxGatewayMidStub {
         std::optional<blackbox::UserInfo> defaultUserInfo(blackbox::UID uid) const override {
             blackbox::UserInfo userInfo;
             userInfo.setUid(uid);
             userInfo.setEmail(BLACKBOX_EMAIL);
             return userInfo;
         }
    };

    AbsentUserProfileGateway socialGtw;
    BlackboxGatewayImpl blackboxGtw;
    UserMailInfoProvider provider(socialGtw, blackboxGtw);

    UNIT_ASSERT_EQUAL(provider.getUserMailInfo(TEST_UID), std::nullopt);
}

Y_UNIT_TEST(no_social_profile__only_locale_in_blackbox_userinfo)
{
    struct BlackboxGatewayImpl : BlackboxGatewayMidStub {
         std::optional<blackbox::UserInfo> defaultUserInfo(blackbox::UID uid) const override {
             blackbox::UserInfo userInfo;
             userInfo.setUid(uid);
             userInfo.setLocale(BLACKBOX_LOCALE);
             return userInfo;
         }
    };

    AbsentUserProfileGateway socialGtw;
    BlackboxGatewayImpl blackboxGtw;
    UserMailInfoProvider provider(socialGtw, blackboxGtw);

    UNIT_ASSERT_EQUAL(provider.getUserMailInfo(TEST_UID), std::nullopt);
}

Y_UNIT_TEST(full_social_profile)
{
    FullUserProfileGateway socialGtw;
    FullUserBlackboxGateway blackboxGtw;
    UserMailInfoProvider provider(socialGtw, blackboxGtw);

    auto expected = UserMailInfo(TEST_UID, SOCIAL_EMAIL, USERNAME, SOCIAL_LOCALE_TEXT);
    auto obtained = provider.getUserMailInfo(TEST_UID);

    UNIT_ASSERT(obtained != std::nullopt);
    UNIT_ASSERT_EQUAL(obtained.value(), expected);
}

Y_UNIT_TEST(email_is_absent_in_social_profile_but_present_in_blackbox_userinfo)
{
    struct ProfileGatewayImpl : ProfileGatewayMidStub {
        std::optional<social::Profile> getUserProfile(social::TUid uid) const override {
            return social::Profile(
                uid, "", "" /*empty email*/, false, false, false,
                std::nullopt, SOCIAL_LOCALE_TEXT
            );
        }
    };

    ProfileGatewayImpl socialGtw;
    FullUserBlackboxGateway blackboxGtw;
    UserMailInfoProvider provider(socialGtw, blackboxGtw);

    auto expected = UserMailInfo(TEST_UID, BLACKBOX_EMAIL, USERNAME, SOCIAL_LOCALE_TEXT);
    auto obtained = provider.getUserMailInfo(TEST_UID);

    UNIT_ASSERT(obtained != std::nullopt);
    UNIT_ASSERT_EQUAL(obtained.value(), expected);
}

Y_UNIT_TEST(locale_is_absent_in_social_profile_but_present_in_blackbox_userinfo)
{
    struct ProfileGatewayImpl : ProfileGatewayMidStub {
        std::optional<social::Profile> getUserProfile(social::TUid uid) const override {
            return social::Profile(
                uid, "", SOCIAL_EMAIL, false, false, false,
                std::nullopt, "" /*empty locale*/
            );
        }
    };

    ProfileGatewayImpl socialGtw;
    FullUserBlackboxGateway blackboxGtw;
    UserMailInfoProvider provider(socialGtw, blackboxGtw);

    auto expected = UserMailInfo(TEST_UID, SOCIAL_EMAIL, USERNAME, BLACKBOX_LOCALE_TEXT);
    auto obtained = provider.getUserMailInfo(TEST_UID);

    UNIT_ASSERT(obtained != std::nullopt);
    UNIT_ASSERT_EQUAL(obtained.value(), expected);
}

} // suite

} // maps::wiki::notifications_sender::tests
