#include "filters.h"

#include <maps/libs/log8/include/log8.h>
#include <maps/libs/pgpool/include/pgpool3.h>
#include <maps/libs/http/include/http.h>
#include <yandex/maps/wiki/common/batch.h>
#include <yandex/maps/wiki/common/moderation.h>
#include <yandex/maps/wiki/revision/branch.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/commit.h>
#include <yandex/maps/wiki/revision/snapshot.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/revision/snapshot_id.h>
#include <yandex/maps/wiki/revision/exception.h>
#include <yandex/maps/wiki/social/gateway.h>
#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <maps/wikimap/mapspro/libs/gdpr/include/user.h>
#include <maps/wikimap/mapspro/libs/revision_meta/include/commit_regions.h>
#include <maps/wikimap/mapspro/libs/social/include/yandex/maps/wiki/social/published_commits.h>

#include <geos/geom/Geometry.h>
#include <geos/geom/GeometryFactory.h>
#include <geos/io/WKTReader.h>
#include <geos/io/WKTWriter.h>

#include <boost/algorithm/string/predicate.hpp>

#include <algorithm>
#include <iomanip>
#include <sstream>

namespace maps::wiki::releases_notification {

namespace {

const size_t COMMIT_IDS_BATCH_SIZE = 1000;
const auto TIME_ZONE_OFFSET = std::chrono::hours(3);

} // namespace

VecReleaseData getVecReleaseData(
    pqxx::transaction_base& txnCore,
    pqxx::transaction_base& txnSocial,
    Mode mode)
{
    INFO() << "Get vec releases users";

    social::PublishedCommits publishedCommits(txnSocial);
    auto commitIds = publishedCommits.getUnprocessedIds(social::commits_handlers::RELEASES_NOTIFICATION);

    INFO() << "Found " << commitIds.size() << " published commits";

    VecReleaseData releaseData;

    common::applyBatchOp<social::TIds>(
        commitIds,
        COMMIT_IDS_BATCH_SIZE,
        [&](const social::TIds& batch) {
            auto commits = revision::Commit::load(
                txnCore,
                revision::filters::CommitAttr::id().in(batch)
            );
            for (const auto& commit: commits) {
                const gdpr::User user(commit.createdBy());
                if (user.hidden()) {
                    continue;
                }
                auto& vecUserData = releaseData.commitedUsers[user.uid()];
                vecUserData.addCommit();

                auto createdAt = commit.createdAtTimePoint();

                if (createdAt < vecUserData.sinceTimePoint()) {
                    vecUserData.setSinceTimePoint(createdAt);
                }
                if (createdAt > vecUserData.tillTimePoint()) {
                    vecUserData.setTillTimePoint(createdAt);
                }

                if (createdAt < releaseData.firstCommitAt) {
                    releaseData.firstCommitAt = createdAt;
                }
                if (createdAt > releaseData.lastCommitAt) {
                    releaseData.lastCommitAt = createdAt;
                }
            }
            if (mode == Mode::Real) {
                publishedCommits.process(batch, social::commits_handlers::RELEASES_NOTIFICATION);
            }
        });

    INFO() << "Found " << releaseData.commitedUsers.size() << " unique users";
    return releaseData;
}

std::set<revision::UserID> getUsersWithIntersectedSubscriptionZone(
    pqxx::transaction_base& txnCore,
    pqxx::transaction_base& txnSocial,
    const common::Geom& releaseGeom)
{
    std::set<revision::UserID> affectedUsers;
    revision::RevisionsGateway rg(txnCore);
    auto trunkSnapshot = rg.stableSnapshot(rg.headCommitId());

    std::stringstream query;
    query << " SELECT subscriber, feed_id as object_id"
          << " FROM social.subscription";

    std::map<revision::DBID, std::vector<revision::UserID>> feedSubscribers;
    std::set<revision::DBID> subscriptionObjectIds;
    for (const auto& row: txnSocial.exec(query.str())) {
        auto uid = row["subscriber"].as<revision::UserID>();
        const gdpr::User user(uid);
        if (user.hidden()) {
            continue;
        }
        auto objectId = row["object_id"].as<revision::DBID>();

        feedSubscribers[objectId].push_back(uid);
        subscriptionObjectIds.insert(objectId);
    }

    if (subscriptionObjectIds.empty()) {
        return affectedUsers;
    }
    auto userSubscriptionRevisions = trunkSnapshot.objectRevisionsByFilter(
        revision::filters::Geom::defined() &&
        revision::filters::ObjRevAttr::isNotDeleted() &&
        revision::filters::ObjRevAttr::objectId().in(subscriptionObjectIds));

    for (const auto& revision: userSubscriptionRevisions) {
        if (!revision.data().geometry) {
            continue;
        }
        auto subscriptionGeom = common::Geom(*revision.data().geometry);
        if (subscriptionGeom->intersects(releaseGeom.geosGeometryPtr())) {
            for (auto uid: feedSubscribers[revision.id().objectId()]) {
                if (affectedUsers.count(uid) == 0) {
                    affectedUsers.insert(uid);
                    INFO() << "Add user with uid = " << uid << " intersected by subscription";
                }
            }
        }
    }
    return affectedUsers;
}

std::set<revision::UserID> getUsersWithIntersectedModerationZone(
    pqxx::transaction_base& txnCore,
    const common::Geom& releaseGeom)
{
    std::set<revision::UserID> affectedUsers;

    acl::ACLGateway aclGateway(txnCore);
    std::set<acl::ID> aoiIds;

    const auto& moderatorRole = aclGateway.role(common::MODERATION_STATUS_MODERATOR);
    for (const auto& policy: moderatorRole.policies()) {
        if (policy.aoi().wkb().empty()) {
            continue;
        }
        auto moderationGeom = common::Geom(policy.aoi().wkb());
        if (moderationGeom->intersects(releaseGeom.geosGeometryPtr())) {
            aoiIds.insert(policy.aoiId());
        }
    }

    for (auto aoiId: aoiIds) {
        auto users = aclGateway.users(
            0, moderatorRole.id(), aoiId, acl::User::Status::Active, 0, 0).value();
        for (const auto& user: users) {
            affectedUsers.insert(user.uid());
            INFO() << "Add user with uid = " << user.uid() << " intersected by moderation";
        }
    }

    return affectedUsers;
}

std::set<revision::UserID> getSatReleaseUsers(
    pqxx::transaction_base& txnCore,
    pqxx::transaction_base& txnSocial,
    const common::Geom& releaseGeom)
{
    std::set<revision::UserID> affectedUsers;
    auto subscriptionZoneUsers =
        getUsersWithIntersectedSubscriptionZone(txnCore, txnSocial, releaseGeom);
    affectedUsers.insert(subscriptionZoneUsers.begin(), subscriptionZoneUsers.end());

    auto moderationZoneUsers = getUsersWithIntersectedModerationZone(txnCore, releaseGeom);
    affectedUsers.insert(moderationZoneUsers.begin(), moderationZoneUsers.end());

    return affectedUsers;
}

namespace {

std::set<revision::UserID> getUids(
    pqxx::transaction_base& txnCore,
    const revision::filters::FilterExpr& filter,
    size_t minCount)
{
    const auto userToCount = revision::Commit::loadUserCommitsCount(txnCore, filter);
    std::set<revision::UserID> result;
    for (const auto& [userId, count]: userToCount) {
        const gdpr::User user(userId);
        if (user.hidden()) {
            continue;
        }
        if (count >= minCount) {
            result.insert(userId);
        }
    }
    return result;
}

} // namespace

std::set<revision::UserID> getContributorUsers(
    pqxx::transaction_base& txnCore)
{
    auto result = getUids(txnCore, revision::filters::True(), 0);
    INFO() << "Found " << result.size() << " contributors";
    return result;
}

std::set<revision::UserID> getActiveContributorUsers(
    pqxx::transaction_base& txnCore,
    size_t minCorrectionsCount,
    size_t maxSilentPeriodDays)
{
    auto correctionsCountUids = getUids(txnCore, revision::filters::True(), minCorrectionsCount);
    INFO() << "Found " << correctionsCountUids.size() << " contributors"
        << " having more than " << minCorrectionsCount << " corrections";

    auto timePoint = std::chrono::system_clock::now() - std::chrono::hours(24 * maxSilentPeriodDays);
    auto silentPeriodUids = getUids(txnCore,
        revision::filters::CommitCreationTime() >= std::chrono::system_clock::to_time_t(timePoint), 0);
    INFO() << "Found " << silentPeriodUids.size() << " contributors"
        << " being active withing last " << maxSilentPeriodDays << " days";

    std::set<revision::UserID> result;
    std::set_intersection(
        correctionsCountUids.begin(), correctionsCountUids.end(),
        silentPeriodUids.begin(), silentPeriodUids.end(),
        std::inserter(result, result.begin()));

    INFO() << "Total " << result.size() << " active contributors";
    return result;
}


chrono::TimePoint getBranchCreateTime(
    pqxx::transaction_base& txnCore,
    revision::DBID branchId)
{
    revision::BranchManager branchManager(txnCore);
    auto branch = branchManager.load(branchId);

    return chrono::parseSqlDateTime(branch.createdAt());
}

common::Geom
loadSatTaskParams(pqxx::transaction_base& txnCore, revision::DBID dbTaskId)
{
    std::ostringstream query;
    query << " SELECT ST_AsText(geom) as geom"
          << " FROM service.releases_notification_sat_param"
          << " WHERE task_id = " << dbTaskId;
    auto result = txnCore.exec(query.str());
    REQUIRE(not result.empty(), "Cannot find sat params");

    geos::io::WKTReader reader;
    return common::Geom(reader.read(result[0]["geom"].as<std::string>()));
}

std::string loadSurveyTaskParams(pqxx::transaction_base& txnCore, revision::DBID dbTaskId)
{
    std::ostringstream query;
    query << " SELECT subject"
          << " FROM service.releases_notification_survey_param"
          << " WHERE task_id = " << dbTaskId;
    auto result = txnCore.exec(query.str());
    REQUIRE(not result.empty(), "Cannot find survey params");

    return result[0]["subject"].as<std::string>();
}

struct EventTaskParams {
    std::string subject;
    std::string eventName;
};

EventTaskParams loadEventTaskParams(pqxx::transaction_base& txnCore, revision::DBID dbTaskId)
{
    std::ostringstream query;
    query << " SELECT subject, event_name"
          << " FROM service.releases_notification_event_param"
          << " WHERE task_id = " << dbTaskId;
    auto result = txnCore.exec(query.str());
    REQUIRE(not result.empty(), "Cannot find event params");

    return EventTaskParams{
        result[0]["subject"].as<std::string>(),
        result[0]["event_name"].as<std::string>()
    };
}


ReleaseInfo
findSatUsersToNotify(
    const TaskParams& taskParams,
    PgPools& pgPools)
{
    auto slaveCoreTxn = pgPools.longReadPool.slaveTransaction();
    auto slaveSocialTxn = pgPools.socialPool.slaveTransaction();

    auto releaseGeom = loadSatTaskParams(*slaveCoreTxn, taskParams.dbTaskId());

    auto uids = getSatReleaseUsers(*slaveCoreTxn, *slaveSocialTxn, releaseGeom);
    UidToSpecificParamsPtr uidToTemplateParams;
    for (const auto& uid: uids) {
        uidToTemplateParams[uid] = std::unique_ptr<EmptyParams>(new EmptyParams);
    }
    return ReleaseInfo{
        std::move(uidToTemplateParams),
        std::unique_ptr<EmptyParams>(new EmptyParams)
    };
}

ReleaseInfo
findVecUsersToNotify(
    const TaskParams& taskParams,
    PgPools& pgPools,
    tasks::TaskPgLogger& logger)
{
    auto coreTxn = pgPools.longReadPool.slaveTransaction();
    auto socialTxn = pgPools.socialPool.masterWriteableTransaction();

    auto releaseData = getVecReleaseData(*coreTxn, *socialTxn, taskParams.mode());
    if (taskParams.mode() == Mode::Real) {
        socialTxn->commit();
    }

    if (releaseData.commitedUsers.empty()) {
        logger.logWarn() << "No users to notify";
    } else {
        logger.logInfo() << "First commit at: " <<
            chrono::formatIsoDate(releaseData.firstCommitAt + TIME_ZONE_OFFSET);
        logger.logInfo() << "Last commit at: " <<
            chrono::formatIsoDate(releaseData.lastCommitAt + TIME_ZONE_OFFSET);
    }

    UidToSpecificParamsPtr uidToTemplateParams;
    for (const auto& [userId, vecUserData]: releaseData.commitedUsers) {
        uidToTemplateParams[userId] = std::unique_ptr<VecParams>(new VecParams(
            vecUserData.getCommitsCount(),
            vecUserData.sinceTimePoint(),
            vecUserData.tillTimePoint()
        ));
    }
    auto defaultParams = std::make_unique<VecParams>(
        DEFAULT_COMMITS_COUNT,
        std::chrono::system_clock::now(),
        std::chrono::system_clock::now());

    return ReleaseInfo{
        std::move(uidToTemplateParams),
        std::move(defaultParams)
    };
}

ReleaseInfo
findEventUsersToNotify(
    const TaskParams& taskParams,
    PgPools& pgPools)
{
    auto slaveCoreTxn = pgPools.longReadPool.slaveTransaction();
    auto typeSpecificTaskParams = loadEventTaskParams(*slaveCoreTxn, taskParams.dbTaskId());

    std::map<std::string, std::string> paramsMap;
    paramsMap[template_param_names::SUBJECT] = typeSpecificTaskParams.subject;
    paramsMap[template_param_names::EVENT_NAME] = typeSpecificTaskParams.eventName;

    auto uids = getContributorUsers(*slaveCoreTxn);
    UidToSpecificParamsPtr uidToTemplateParams;
    for (const auto& uid: uids) {
        uidToTemplateParams[uid] =  std::unique_ptr<ConstParams>(new ConstParams(paramsMap));
    }
    return ReleaseInfo{
        std::move(uidToTemplateParams),
        std::unique_ptr<ConstParams>(new ConstParams(paramsMap))
    };
}

ReleaseInfo
findNewsUsersToNotify(
    const TaskParams& /*taskParams*/,
    PgPools& pgPools)
{
    auto slaveCoreTxn = pgPools.longReadPool.slaveTransaction();

    sender::EmailTemplateParams emptyTemplateParams;

    auto uids = getContributorUsers(*slaveCoreTxn);
    UidToSpecificParamsPtr uidToTemplateParams;
    for (const auto& uid: uids) {
        uidToTemplateParams[uid] = std::unique_ptr<EmptyParams>(new EmptyParams);
    }
    return ReleaseInfo{
        std::move(uidToTemplateParams),
        std::unique_ptr<EmptyParams>(new EmptyParams)
    };
}

ReleaseInfo
findSurveyUsersToNotify(
    const TaskParams& taskParams,
    PgPools& pgPools)
{
    auto slaveCoreTxn = pgPools.longReadPool.slaveTransaction();
    auto subject = loadSurveyTaskParams(*slaveCoreTxn, taskParams.dbTaskId());

    std::map<std::string, std::string> paramsMap;
    paramsMap[template_param_names::SUBJECT] = subject;

    auto uids = getContributorUsers(*slaveCoreTxn);
    UidToSpecificParamsPtr uidToTemplateParams;
    for (const auto& uid: uids) {
        uidToTemplateParams[uid] =  std::unique_ptr<ConstParams>(new ConstParams(paramsMap));
    }
    return ReleaseInfo{
        std::move(uidToTemplateParams),
        std::unique_ptr<ConstParams>(new ConstParams(paramsMap))
    };
}


ReleaseInfo
findUsersToNotify(
    const TaskParams& taskParams,
    PgPools& pgPools,
    tasks::TaskPgLogger& logger)
{
    switch (taskParams.releaseType()) {
        case ReleaseType::Vec:
            return findVecUsersToNotify(taskParams, pgPools, logger);
        case ReleaseType::Sat:
            return findSatUsersToNotify(taskParams, pgPools);
        case ReleaseType::Event:
            return findEventUsersToNotify(taskParams, pgPools);
        case ReleaseType::News:
            return findNewsUsersToNotify(taskParams, pgPools);
        case ReleaseType::Survey:
            return findSurveyUsersToNotify(taskParams, pgPools);
    }
    throw Exception() << "Unrecognized release type";
}


} // maps::wiki::releases_notification
