#include "releases_notification.h"
#include "common.h"
#include "filters.h"
#include "email_params_rendering.h"

#include <maps/libs/auth/include/tvm.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/pgpool/include/pgpool3.h>
#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <yandex/maps/wiki/tasks/tasks.h>
#include <yandex/maps/wiki/common/moderation.h>
#include <yandex/maps/wiki/common/pg_utils.h>
#include <yandex/maps/wiki/revision/branch.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/commit.h>
#include <yandex/maps/wiki/revision/snapshot.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/revision/snapshot_id.h>
#include <yandex/maps/wiki/revision/exception.h>
#include <maps/libs/locale/include/find_helpers.h>
#include <maps/libs/locale/include/convert.h>

#include <pqxx/pqxx>

#include <algorithm>
#include <list>
#include <optional>
#include <set>
#include <vector>

namespace acl = maps::wiki::acl;
namespace revision = maps::wiki::revision;
namespace common = maps::wiki::common;
namespace social = maps::wiki::social;

namespace maps::wiki::releases_notification {

namespace {

const std::string REASON_NOT_ACTIVE = "not active";
const std::string REASON_DECLINED_SUBSCRIPTION = "declined subscription";
const std::string REASON_NO_BLACKBOX_INFO = "not enough information from blackbox";
const std::string REASON_INVALID_EMAIL = "profile's email is invalid";

inline void logFilteredUser(revision::UserID uid, const std::string& reason)
{
    INFO() << "User with id = " << uid << " was filtered. Reason: " << reason;
}

} // namespace

std::optional<UserInfo>
fromBlackboxUserInfo(
    const std::optional<blackbox::UserInfo>& blackboxInfo,
    const locale::Locale& locale)
{
    if (!blackboxInfo) {
        return std::nullopt;
    }
    return UserInfo{
        blackboxInfo->uid(),
        blackboxInfo->email(),
        locale,
        blackboxInfo->username()
    };
}

UserFilter::UserFilter(
        std::unique_ptr<blackbox::IGateway> blackbox,
        pqxx::transaction_base& txnCore,
        pqxx::transaction_base& txnSocial)
    : blackbox_(std::move(blackbox))
    , aclGateway_(txnCore)
    , socialProfileGateway_(txnSocial)
{}

std::optional<UserInfo>
UserFilter::checkAndGetSocialInfo(revision::UserID uid)
{
    auto user = aclGateway_.user(uid);

    if (user.status() != acl::User::Status::Active) {
        logFilteredUser(uid, REASON_NOT_ACTIVE);
        return std::nullopt;
    }

    const auto& profiles = socialProfileGateway_.getUserProfiles({uid});
    if (profiles.empty()) {
        return fromBlackboxUserInfo(blackbox_->defaultUserInfo(uid), DEFAULT_LOCALE);
    }

    const auto& profile = profiles.front();
    if (!profile.hasBroadcastSubscription()) {
        logFilteredUser(uid, REASON_DECLINED_SUBSCRIPTION);
        return std::nullopt;
    }

    auto defaultUserInfo = blackbox_->defaultUserInfo(uid);
    if (!defaultUserInfo) {
        logFilteredUser(uid, REASON_NO_BLACKBOX_INFO);
        return std::nullopt;
    }

    auto localeStr = profile.locale();
    auto locale = localeStr.empty()
        ? DEFAULT_LOCALE
        : locale::to<locale::Locale>(localeStr);

    if (!profile.email().empty()) {
        if (defaultUserInfo->email() == profile.email() or blackbox_->isEmailValid(profile.email(), uid)) {
            return UserInfo{
                defaultUserInfo->uid(),
                profile.email(),
                locale,
                defaultUserInfo->username()
            };
        } else {
            logFilteredUser(uid, REASON_INVALID_EMAIL);
            return std::nullopt;
        }
    } else {
        return fromBlackboxUserInfo(defaultUserInfo, locale);
    }
}


namespace {

const std::string STATUS_SENT = "sent";
const std::string STATUS_SCHEDULED = "scheduled";
const std::string STATUS_FAILED = "failed";

struct ScheduledEmailData
{
    std::string email;
    sender::EmailTemplateParams templateParams;
    locale::Locale locale;
};

struct EmailData
{
    revision::DBID id;
    std::string email;
    std::string locale;
    std::string attrs;
};

sender::Result notifyAndUpdate(
    PgPools& pgPools,
    const EmailData& emailData,
    sender::BaseGateway& senderGateway,
    const sender::LocalizedCampaign& localizedCampaign)
{
    auto attrs = sender::EmailTemplateParams::fromJson(emailData.attrs);

    sender::CampaignSlug bestCampaign = getBestLocalizedString(
        localizedCampaign.slugs, locale::to<locale::Locale>(emailData.locale));

    auto setStatus = [&](const std::string& status) {
        std::ostringstream query;
        auto txnCore = pgPools.longReadPool.masterWriteableTransaction();
        query << " UPDATE service.releases_notification_email"
              << " SET status = " << txnCore->quote(status)
              << " WHERE id = " << emailData.id;
        txnCore->exec(query.str());
        txnCore->commit();
    };
    // it guarantees that email can't be sent twice
    setStatus(STATUS_SENT);
    try {
        auto result = senderGateway.sendToEmail(bestCampaign, emailData.email, attrs);
        if (result == sender::Result::Sent) {
            INFO() << "Email to " << emailData.email << " has been sent";
        } else {
            INFO() << "Sender failed while sending email to " << emailData.email << " - skip";
            setStatus(STATUS_FAILED);
        }
        return result;
    } catch (const std::exception& ex) {
        setStatus(STATUS_SCHEDULED);
        throw;
    }
}

class ProgressLogger
{
public:
    ProgressLogger(tasks::TaskPgLogger& logger, int totalNum)
        : logger_(logger)
        , totalNum_(totalNum)
        , currentNum_(0)
        , sentNum_(0)
        , failedNum_(0)
    {
        REQUIRE(totalNum_ > 0, "Incorrect logger param");
    }

    void addCompleted(sender::Result senderResult)
    {
        REQUIRE(currentNum_ + 1 <= totalNum_, "Progress overflow");
        int prevPercents = currentNum_ * 100 / totalNum_;
        int curPercents = (currentNum_ + 1) * 100 / totalNum_;
        currentNum_++;
        if (senderResult == sender::Result::Sent) {
            sentNum_++;
        } else {
            failedNum_++;
        }

        if (prevPercents / NOTIFICATION_LOG_FREQUENCY_PERCENTAGE <
            curPercents / NOTIFICATION_LOG_FREQUENCY_PERCENTAGE) {
            logger_.logInfo() << sentNum_ << " of " << totalNum_ << " users have been notified"
                << "; " << failedNum_ << " emails failed";
        }
    }
private:
    tasks::TaskPgLogger& logger_;
    int totalNum_;
    int currentNum_;
    int sentNum_;
    int failedNum_;

    int NOTIFICATION_LOG_FREQUENCY_PERCENTAGE = 10;
};

} // namespace


void notifyUsers(
    PgPools& pgPools,
    revision::DBID taskId,
    sender::BaseGateway& senderGateway,
    const sender::LocalizedCampaign& localizedCampaign,
    tasks::TaskPgLogger& logger)
{
    int numEmailsSent = 0;
    int numEmailsTotal = 0;

    std::vector<EmailData> emailDatas;
    {
        auto mainTxn = pgPools.longReadPool.masterReadOnlyTransaction();
        std::ostringstream query;
        query << " SELECT id, email, hstore_to_json(sender_attrs) as attrs, locale"
              << " FROM service.releases_notification_email"
              << " WHERE task_id = " << taskId
              << " AND status = " << mainTxn->quote(STATUS_SCHEDULED);
        auto result = mainTxn->exec(query.str());
        numEmailsTotal = result.affected_rows();
        if (numEmailsTotal > 0) {
            for (const auto& row: result) {
                emailDatas.push_back({
                    row["id"].as<revision::DBID>(),
                    row["email"].as<std::string>(),
                    row["locale"].as<std::string>(),
                    row["attrs"].as<std::string>()
                });
            }
        } else {
            logger.logInfo() << "No users to be notified";
        }
    }
    if (!emailDatas.empty()) {
        ProgressLogger progressLogger(logger, numEmailsTotal);
        for (const auto& row: emailDatas) {
            auto result = notifyAndUpdate(pgPools, row, senderGateway, localizedCampaign);
            progressLogger.addCompleted(result);
            if (result == sender::Result::Sent) {
                numEmailsSent++;
            }
        }
    }

    auto mainTxn = pgPools.longReadPool.masterWriteableTransaction();
    std::ostringstream query;
    query << " UPDATE service.releases_notification_task"
          << " SET sent_notifications_num = " << numEmailsSent
          << " WHERE id = " << taskId;
    mainTxn->exec(query.str());
    mainTxn->commit();
}


namespace {

std::map<revision::UserID, UserInfo>
filteredUserInfos(
    const blackbox::Configuration& blackboxConfig,
    const maps::common::RetryPolicy& retryPolicy,
    PgPools& pgPools,
    const std::vector<revision::UserID>& uids)
{
    auto slaveCoreTxn = pgPools.longReadPool.slaveTransaction();
    auto slaveSocialTxn = pgPools.socialPool.slaveTransaction();

    auto tvmClient = auth::TvmtoolSettings()
            .selectClientAlias("maps-core-nmaps-tasks-feedback")
            .makeTvmClient();

    auto blackbox = std::make_unique<blackbox::Gateway>(
        blackboxConfig,
        retryPolicy,
        [&]() {return tvmClient.GetServiceTicketFor("blackbox"); }
    );

    UserFilter filter(std::move(blackbox), *slaveCoreTxn, *slaveSocialTxn);

    std::map<revision::UserID, UserInfo> result;
    for (const auto& uid: uids) {
        auto userInfo = filter.checkAndGetSocialInfo(uid);
        if (userInfo) {
            result[uid] = *userInfo;
        }
    }

    return result;
}

void addEmail(
    const TaskParams& params,
    const ScheduledEmailData& scheduledEmailData,
    pqxx::transaction_base& txnCore)
{
    std::ostringstream query;
    query << " INSERT INTO service.releases_notification_email"
          << " (task_id, email, status, sender_attrs, locale)"
          << " VALUES"
          << " ("
            << params.dbTaskId() << ", "
            << txnCore.quote(scheduledEmailData.email) << ", "
            << "'" << STATUS_SCHEDULED << "'" << ", "
            << common::attributesToHstore(txnCore, scheduledEmailData.templateParams.data()) << ", "
            << txnCore.quote(locale::toString(scheduledEmailData.locale))
          << " )";
    txnCore.exec(query.str());
}

void
saveEmailsToDb(
    const std::vector<ScheduledEmailData>& scheduledEmailDatas,
    const TaskParams& params,
    PgPools& pgPools)
{
    auto taskTxn = pgPools.longReadPool.masterWriteableTransaction();
    for (const auto& scheduledEmailData: scheduledEmailDatas) {
        addEmail(params, scheduledEmailData, *taskTxn);
    }

    tasks::freezeTask(*taskTxn, params.dbTaskId());
    taskTxn->commit();
}

} // namespace

void collectEmailsForNotification(
    const common::ExtendedXmlDoc& servicesConfig,
    const sender::Config& senderConfig,
    const TaskParams& taskParams,
    PgPools& pgPools,
    tasks::TaskPgLogger& logger)
{
    logger.logInfo() << "Retrieve users for release";

    auto releaseInfo = findUsersToNotify(taskParams, pgPools, logger);

    blackbox::Configuration blackboxConfig(
        servicesConfig.get<std::string>("/config/common/blackbox-url"));
    auto retryPolicy = defaultRetryPolicy();

    std::vector<revision::UserID> uids;
    for (const auto& [uid, params]: releaseInfo.uidToParams) {
        uids.push_back(uid);
    }

    auto uidToUserInfoFiltered = filteredUserInfos(
        blackboxConfig,
        retryPolicy,
        pgPools,
        uids);


    std::vector<ScheduledEmailData> scheduledEmailDatas;

    if (taskParams.mode() == Mode::Test) {
        std::map<std::string, revision::UserID> emailToUid;
        for (const auto& [uid, userInfo]: uidToUserInfoFiltered) {
            emailToUid[userInfo.email] = uid;
        }

        for (const auto& testEmail: taskParams.testEmails()) {

            auto emailToUidIt = emailToUid.find(testEmail);

            if (emailToUidIt != emailToUid.end()) {
                auto uid = emailToUidIt->second;

                auto userInfoIt = uidToUserInfoFiltered.find(uid);
                ASSERT(userInfoIt != uidToUserInfoFiltered.end());
                const auto& userInfo = userInfoIt->second;

                auto typeSpecificParamsIt = releaseInfo.uidToParams.find(uid);
                ASSERT(typeSpecificParamsIt != releaseInfo.uidToParams.end());

                auto emailTemplateParams = renderEmailParams(
                    *(typeSpecificParamsIt->second),
                    userInfo,
                    senderConfig.constantParams,
                    taskParams);

                scheduledEmailDatas.push_back(ScheduledEmailData{
                    userInfo.email,
                    emailTemplateParams,
                    userInfo.locale
                });

            } else {
                auto emailTemplateParams = renderEmailParamsForTestUser(
                    *releaseInfo.defaultParams,
                    senderConfig.constantParams,
                    taskParams);

               scheduledEmailDatas.push_back(ScheduledEmailData{
                    testEmail,
                    emailTemplateParams,
                    DEFAULT_LOCALE
                });

            }
        }
    } else {
        for (const auto& [uid, typeSpecificParams]: releaseInfo.uidToParams) {

            auto userInfoIt = uidToUserInfoFiltered.find(uid);
            if (userInfoIt == uidToUserInfoFiltered.end()) {
                continue;
            }
            const auto& userInfo = userInfoIt->second;

            auto emailTemplateParams = renderEmailParams(
                *typeSpecificParams,
                userInfo,
                senderConfig.constantParams,
                taskParams);

            scheduledEmailDatas.push_back(ScheduledEmailData{
                userInfo.email,
                emailTemplateParams,
                userInfo.locale
            });
        }
    }

    saveEmailsToDb(scheduledEmailDatas, taskParams, pgPools);

    logger.logInfo() << "Get " << scheduledEmailDatas.size() << " messages to send";

}

} // maps::wiki::releases_notification
