#include <maps/wikimap/mapspro/services/tasks_feedback/src/releases_notification_worker/lib/releases_notification.h>
#include <maps/wikimap/mapspro/services/tasks_feedback/src/releases_notification_worker/lib/common.h>
#include <maps/wikimap/mapspro/services/tasks_feedback/src/releases_notification_worker/lib/filters.h>
#include <maps/wikimap/mapspro/services/tasks_feedback/src/releases_notification_worker/lib/email_params_rendering.h>
#include <maps/wikimap/mapspro/libs/revision_meta/include/commit_regions.h>

#include <maps/wikimap/mapspro/libs/social/include/yandex/maps/wiki/social/published_commits.h>

#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <yandex/maps/wiki/common/geom.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>
#include <yandex/maps/wiki/common/moderation.h>
#include <yandex/maps/wiki/http/blackbox/blackbox.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/commit.h>
#include <yandex/maps/wiki/revision/commit_manager.h>
#include <yandex/maps/wiki/revision/common.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/social/event.h>
#include <yandex/maps/wiki/social/gateway.h>
#include <yandex/maps/wiki/social/profile_gateway.h>
#include <yandex/maps/wiki/unittest/arcadia.h>

#include <library/cpp/testing/unittest/registar.h>

#include <chrono>
#include <optional>
#include <string>
#include <vector>

namespace acl = maps::wiki::acl;
namespace revision = maps::wiki::revision;
namespace social = maps::wiki::social;
namespace unittest = maps::wiki::unittest;

namespace maps::wiki::releases_notification::test {

namespace {

const revision::UserID BRANCH_CREATOR = 1;

const revision::DBID PUBLICATION_REGION_1_ID = 100;
const revision::DBID PUBLICATION_REGION_2_ID = 200;

const revision::UserID COMMIT_1_REG_1_DRAFT_USER_ID = 10;
const revision::UserID COMMIT_2_BR_1_REG_1_USER_ID = 11;
const revision::UserID COMMIT_3_BR_1_REG_2_USER_ID = 12;
const revision::UserID COMMIT_4_BR_2_REG_2_USER_ID = 13;
const revision::UserID COMMIT_5_BR_1_REG_2_REVERT_USER_ID = 14;

const revision::UserID MODERATOR_1_USER_ID = 15;
const revision::UserID MODERATOR_2_USER_ID = 16;

const revision::Wkb AOI_1_WKB = common::wkt2wkb("POLYGON ((0 0, 0 10, 10 10, 10 0, 0 0))");
const revision::Wkb AOI_2_WKB = common::wkt2wkb("POLYGON ((20 20, 20 30, 30 30, 30 20, 20 20))");
const revision::Wkb SAT_RELEASE_WKB = common::wkt2wkb("POLYGON ((5 5, 5 10, 10 10, 10 5, 5 5))");

const std::string BOUNDS_IN = "[0, 0, 1, 1]";
const std::string BOUNDS_OUT = "[10, 10, 11, 11]";

const revision::UserID REAL_USER_UID = 255374041;
const revision::UserID UNREAL_USER_UID = 123;

const Email REAL_USER_EMAIL = "vlkulpinov@yandex.com";
const Email UNREAL_USER_EMAIL = "impossible";

const std::string USER_CREATED_OR_UNBANNED_TIME = "2010-10-10 10:10:10";

struct TestData {
    TestData() {}
    explicit TestData(
            const revision::Wkb& geom,
            const std::optional<revision::Attributes>& attrs = std::nullopt,
            const std::optional<revision::Description>& desc = std::nullopt)
        : data(attrs, desc, geom, std::nullopt, false)
    {}
    explicit TestData(
            const revision::RelationData& rdata,
            const std::optional<revision::Attributes>& attrs = std::nullopt,
            const std::optional<revision::Description>& desc = std::nullopt)
        : data(attrs, desc, std::nullopt, rdata, false)
    {}
    explicit TestData(
            const std::optional<revision::Attributes>& attrs,
            const std::optional<revision::Description>& desc = std::nullopt,
            const std::optional<revision::Wkb>& geom = std::nullopt,
            const std::optional<revision::RelationData>& rdata = std::nullopt)
        : data(attrs, desc, geom, rdata, false)
    {}
    explicit TestData(
            bool deleted,
            const std::optional<revision::Attributes>& attrs = std::nullopt,
            const std::optional<revision::Description>& desc = std::nullopt,
            const std::optional<revision::Wkb>& geom = std::nullopt,
            const std::optional<revision::RelationData>& rdata = std::nullopt)
        : data(attrs, desc, geom, rdata, deleted)
    {}

    operator revision::ObjectData() const
    {
        return data;
    }

    revision::ObjectData data;
};


class VecReleasesNotificationFixture : public unittest::ArcadiaDbFixture
{
public:
    VecReleasesNotificationFixture()
    {
        auto txn = pool().masterWriteableTransaction();
        revision::BranchManager branchManger(*txn);

        std::list<revision::RevisionsGateway::NewRevisionData> lst1 {
            {revision::RevisionID::createNewID(1), {} }
        };
        std::list<revision::RevisionsGateway::NewRevisionData> lst2 {
            {revision::RevisionID::createNewID(2), {}}
        };
        std::list<revision::RevisionsGateway::NewRevisionData> lst3 {
            {revision::RevisionID::createNewID(3), {}}
        };
        std::list<revision::RevisionsGateway::NewRevisionData> lst4 {
            {revision::RevisionID::createNewID(4), {}}
        };
        revision::RevisionsGateway gatewayTrunk(*txn);
        revision::Attributes attrs {
            {"description", "commit"}
        };

        social::Gateway socialGateway(*txn);
        revision_meta::CommitRegions commitRegions(*txn);

        auto commit1 = gatewayTrunk.createCommit(lst1.begin(), lst1.end(),
            COMMIT_1_REG_1_DRAFT_USER_ID, attrs);
        commitRegions.push(commit1.id(), { PUBLICATION_REGION_1_ID });

        auto commit2 = gatewayTrunk.createCommit(lst2.begin(), lst2.end(),
            COMMIT_2_BR_1_REG_1_USER_ID, attrs);
        commitRegions.push(commit2.id(), { PUBLICATION_REGION_1_ID });
        auto event2 = socialGateway.createCommitEvent(
            commit2.id(), social::CommitData(0, commit2.id(), "", BOUNDS_IN), std::nullopt, {});
        socialGateway.createAcceptedTask(event2, USER_CREATED_OR_UNBANNED_TIME);

        // commit2_1 corrects commit2
        std::list<revision::RevisionsGateway::NewRevisionData> lst2_1 {
            {revision::RevisionID(2, commit2.id()), {}}
        };
        auto commit2_1 = gatewayTrunk.createCommit(lst2_1.begin(), lst2_1.end(),
            COMMIT_2_BR_1_REG_1_USER_ID, attrs);
        commitRegions.push(commit2_1.id(), { PUBLICATION_REGION_1_ID });
        auto event2_1 = socialGateway.createCommitEvent(
            commit2_1.id(), social::CommitData(0, commit2_1.id(), "", BOUNDS_IN), std::nullopt, {});
        socialGateway.createAcceptedTask(event2_1, USER_CREATED_OR_UNBANNED_TIME);

        auto commit3 = gatewayTrunk.createCommit(lst3.begin(), lst3.end(),
            COMMIT_3_BR_1_REG_2_USER_ID, attrs);
        commitRegions.push(commit3.id(), { PUBLICATION_REGION_2_ID });
        auto event3 = socialGateway.createCommitEvent(
            commit3.id(), social::CommitData(0, commit3.id(), "", BOUNDS_OUT), std::nullopt, {});
        socialGateway.createAcceptedTask(event3, USER_CREATED_OR_UNBANNED_TIME);

        auto commit4 = gatewayTrunk.createCommit(lst4.begin(), lst4.end(),
            COMMIT_4_BR_2_REG_2_USER_ID, attrs);
        commitRegions.push(commit4.id(), { PUBLICATION_REGION_2_ID });

        auto branchApproved = branchManger.createApproved(BRANCH_CREATOR, revision::Attributes());

        revision::CommitManager commitManager(*txn);

        // commit5 reverts commit3
        auto commit5 = commitManager.revertCommitsInTrunk({commit3.id()},
            COMMIT_5_BR_1_REG_2_REVERT_USER_ID, {}).createdCommit;
        commitRegions.push(commit5.id(), { PUBLICATION_REGION_2_ID });
        auto event5 = socialGateway.createCommitEvent(
            commit5.id(), social::CommitData(0, commit5.id(), "", BOUNDS_OUT), std::nullopt, {});
        socialGateway.createAcceptedTask(event5, USER_CREATED_OR_UNBANNED_TIME);

        // Approve 2, 2.1, 3, 5 commits
        std::list<revision::DBID> approvedCommits1{commit2.id(), commit2_1.id(), commit3.id(), commit5.id()};
        commitManager.approve(approvedCommits1);

        // Merge 2, 3 commits to stable branch and close it
        auto branch1 = branchManger.createStable(BRANCH_CREATOR, revision::Attributes());
        revision::RevisionsGateway gatewayStable1(*txn, branch1);
        commitManager.mergeApprovedToStable(approvedCommits1.back());
        branch1.finish(*txn, BRANCH_CREATOR);

        // Approve 4 commit
        std::list<revision::DBID> approvedCommits2{commit4.id()};
        commitManager.approve(approvedCommits2);

        // Merge 4 commit to stable branch and close it
        auto branch2 = branchManger.createStable(BRANCH_CREATOR, revision::Attributes());
        revision::RevisionsGateway gatewayStable2(*txn, branch2);
        commitManager.mergeApprovedToStable(approvedCommits2.back());
        branch2.finish(*txn, BRANCH_CREATOR);

        // Publish 2, 2.1, 3, 5 commits
        commitRegions.publish({
            commit2.id(),
            commit2_1.id(),
            commit3.id(),
            commit5.id()
        });

        social::PublishedCommits publishedCommits(*txn);
        publishedCommits.push({
            commit2.id(),
            commit2_1.id(),
            commit3.id(),
            commit5.id()
        });

        txn->commit();
    }
};

class SatReleasesNotificationFixture : public unittest::ArcadiaDbFixture
{
public:
    SatReleasesNotificationFixture()
    {
        auto txn = pool().masterWriteableTransaction();

        const int REVISION_1_ID = 1;
        const int REVISION_2_ID = 2;

        std::list<revision::RevisionsGateway::NewRevisionData> lst1 {
            {revision::RevisionID::createNewID(REVISION_1_ID), TestData
                { AOI_1_WKB, revision::Attributes{{"cat:aoi", "1"},
                                              {"aoi:name", "1"}}
                }
            }
        };
        std::list<revision::RevisionsGateway::NewRevisionData> lst2 {
            {revision::RevisionID::createNewID(REVISION_2_ID), TestData
                { AOI_2_WKB, revision::Attributes{{"cat:aoi", "1"},
                                              {"aoi:name", "1"}}
                }
            }
        };
        revision::RevisionsGateway gatewayTrunk(*txn);
        revision::Attributes attrs {
            {"description", "commit"}
        };

        social::Gateway socialGateway(*txn);

        auto commit1 = gatewayTrunk.createCommit(lst1.begin(), lst1.end(),
            COMMIT_2_BR_1_REG_1_USER_ID, attrs);

        auto commit2 = gatewayTrunk.createCommit(lst2.begin(), lst2.end(),
            COMMIT_5_BR_1_REG_2_REVERT_USER_ID, attrs);

        auto subsConsole1 = socialGateway.subscriptionConsole(COMMIT_2_BR_1_REG_1_USER_ID);
        subsConsole1.subscribe(REVISION_1_ID);
        auto subsConsole2 = socialGateway.subscriptionConsole(COMMIT_5_BR_1_REG_2_REVERT_USER_ID);
        subsConsole2.subscribe(REVISION_2_ID);

        acl::ACLGateway aclGateway(*txn);
        auto role = aclGateway.createRole(common::MODERATION_STATUS_MODERATOR, "");

        auto moderator1 = aclGateway.createUser(MODERATOR_1_USER_ID, "moderator1", "", 0);
        aclGateway.createPolicy(moderator1, role, aclGateway.aoi(REVISION_1_ID));

        auto moderator2 = aclGateway.createUser(MODERATOR_2_USER_ID, "moderator2", "", 0);
        aclGateway.createPolicy(moderator2, role, aclGateway.aoi(REVISION_2_ID));

        txn->commit();
    }
};

class FillContributorsFixture : public unittest::ArcadiaDbFixture
{
public:
    FillContributorsFixture()
    {
        auto txn = pool().masterWriteableTransaction();

        std::list<revision::RevisionsGateway::NewRevisionData> lst1 {
            {revision::RevisionID::createNewID(1), {} }
        };
        std::list<revision::RevisionsGateway::NewRevisionData> lst2 {
            {revision::RevisionID::createNewID(2), {}}
        };
        std::list<revision::RevisionsGateway::NewRevisionData> lst3 {
            {revision::RevisionID::createNewID(3), {}}
        };

        revision::RevisionsGateway gatewayTrunk(*txn);
        revision::Attributes attrs {
            {"description", "commit"}
        };

        gatewayTrunk.createCommit(lst1.begin(), lst1.end(),
            CONTRIBUTOR, attrs);
        gatewayTrunk.createCommit(lst2.begin(), lst2.end(),
            ACTIVE_CONTRIBUTOR, attrs);
        gatewayTrunk.createCommit(lst3.begin(), lst3.end(),
            ACTIVE_CONTRIBUTOR, attrs);

        txn->commit();
    }

    const revision::UserID NOT_CONTRIBUTER = 1;
    const revision::UserID CONTRIBUTOR = 2;
    const revision::UserID ACTIVE_CONTRIBUTOR = 3;
};


class BlackboxStub: public blackbox::IGateway
{
public:
    bool isEmailValid(const std::string& /*email*/, blackbox::UID uid) const override
    {
        return uid == REAL_USER_UID;
    }

    std::optional<blackbox::UserInfo> defaultUserInfo(blackbox::UID uid) const override
    {
        if (uid == REAL_USER_UID) {
            blackbox::UserInfo userInfo;
            userInfo.setUid(REAL_USER_UID);
            userInfo.setEmail(REAL_USER_EMAIL);
            return userInfo;
        }
        return std::nullopt;
    }

};

void addProfile(revision::UserID uid, Email email, bool bs, pqxx::transaction_base& txn)
{
    social::ProfileGateway gtw(txn);
    social::ProfileOptionalFields fields;
    fields.email = email;
    fields.hasBroadcastSubscription = bs;
    gtw.insertProfile(uid, fields);
}

} // unnamed namespace

Y_UNIT_TEST_SUITE(filters_and_notifications) {

Y_UNIT_TEST_F(user_filter_valid, unittest::ArcadiaDbFixture)
{
    std::unique_ptr<BlackboxStub> blackbox(new BlackboxStub);

    auto txn = pool().masterWriteableTransaction();

    acl::ACLGateway aclGateway(*txn);

    auto userInvalid = aclGateway.createUser(UNREAL_USER_UID, "user1", "", 0);
    addProfile(UNREAL_USER_UID, UNREAL_USER_EMAIL, true, *txn);

    auto userValid = aclGateway.createUser(REAL_USER_UID, "user2", "", 0);
    addProfile(REAL_USER_UID, REAL_USER_EMAIL, true, *txn);

    txn->commit();

    auto slaveCoreTxn = pool().slaveTransaction();
    auto slaveSocialTxn = pool().slaveTransaction();

    UserFilter filter(std::move(blackbox), *slaveCoreTxn, *slaveSocialTxn);

    UNIT_ASSERT(not filter.checkAndGetSocialInfo(userInvalid.uid()));
    UNIT_ASSERT(filter.checkAndGetSocialInfo(userValid.uid()));
}

Y_UNIT_TEST_F(user_filter_decline, unittest::ArcadiaDbFixture)
{
    std::unique_ptr<BlackboxStub> blackbox(new BlackboxStub);

    auto txn = pool().masterWriteableTransaction();

    social::Gateway socialGateway(*txn);
    acl::ACLGateway aclGateway(*txn);

    auto userDecline = aclGateway.createUser(REAL_USER_UID, "user1", "", 0);
    addProfile(REAL_USER_UID, REAL_USER_EMAIL, false, *txn);

    txn->commit();

    auto slaveCoreTxn = pool().slaveTransaction();
    auto slaveSocialTxn = pool().slaveTransaction();

    UserFilter filter(std::move(blackbox), *slaveCoreTxn, *slaveSocialTxn);

    UNIT_ASSERT(not filter.checkAndGetSocialInfo(userDecline.uid()));
}

Y_UNIT_TEST_F(get_vec_release_users, VecReleasesNotificationFixture)
{
    auto coreTxn = pool().masterReadOnlyTransaction();
    auto socialTxn = pool().masterWriteableTransaction();

    auto releaseData = getVecReleaseData(*coreTxn, *socialTxn, Mode::Dry);
    auto& commitedUsers = releaseData.commitedUsers;

    UNIT_ASSERT_EQUAL(commitedUsers.size(), 3);
    UNIT_ASSERT_EQUAL(commitedUsers[COMMIT_2_BR_1_REG_1_USER_ID].getCommitsCount(), 2);
    UNIT_ASSERT_EQUAL(commitedUsers[COMMIT_3_BR_1_REG_2_USER_ID].getCommitsCount(), 1);
    UNIT_ASSERT_EQUAL(commitedUsers[COMMIT_5_BR_1_REG_2_REVERT_USER_ID].getCommitsCount(), 1);
}

Y_UNIT_TEST_F(get_sat_release_users, SatReleasesNotificationFixture)
{
    auto slaveCoreTxn = pool().slaveTransaction();
    auto slaveSocialTxn = pool().slaveTransaction();

    auto affectedUsers = getSatReleaseUsers(
        *slaveCoreTxn, *slaveSocialTxn, common::Geom(SAT_RELEASE_WKB));

    UNIT_ASSERT_EQUAL(affectedUsers.size(), 2);
    UNIT_ASSERT_EQUAL(affectedUsers.count(COMMIT_2_BR_1_REG_1_USER_ID), 1);
    UNIT_ASSERT_EQUAL(affectedUsers.count(MODERATOR_1_USER_ID), 1);
}

Y_UNIT_TEST_F(get_contributor_users, FillContributorsFixture)
{
    auto slaveTxn = pool().slaveTransaction();

    auto uids = getContributorUsers(*slaveTxn);

    UNIT_ASSERT(!uids.count(NOT_CONTRIBUTER));
    UNIT_ASSERT(uids.count(CONTRIBUTOR));
    UNIT_ASSERT(uids.count(ACTIVE_CONTRIBUTOR));
}

Y_UNIT_TEST_F(get_active_contributor_users, FillContributorsFixture)
{
    auto slaveTxn = pool().slaveTransaction();

    auto uids = getActiveContributorUsers(*slaveTxn, 2, 1);

    UNIT_ASSERT(!uids.count(NOT_CONTRIBUTER));
    UNIT_ASSERT(!uids.count(CONTRIBUTOR));
    UNIT_ASSERT(uids.count(ACTIVE_CONTRIBUTOR));
}

} // filters and notifications test suite

} // maps::wiki::releases_notification::test
