#include "../lib/worker.h"
#include "../lib/yt_last_dates.h"

#include <maps/libs/log8/include/log8.h>
#include <maps/libs/common/include/exception.h>
#include <yandex/maps/pgpool3utils/pg_advisory_mutex.h>
#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pg_advisory_lock_ids.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

#include <mapreduce/yt/interface/client.h>
#include <boost/date_time/gregorian/gregorian.hpp>

namespace bg  = boost::gregorian;
namespace mg  = maps::geolib3;
namespace mwc = maps::wiki::common;
namespace mwr = maps::wiki::route_lost_feedback;

namespace {

const int CODE_SUCCESS = 0;
const int CODE_LOCKED = 1;
const int CODE_FAILURE = 2;

mg::PointsVector worldPolygon {
    {-180, -85}, {-180, 85}, {180, 85}, {180, -85}
};

const int AVERAGING_PERIOD = 7;
const int MAX_FEEDBACK_TASKS = 100;

bg::date_period workerDatesPeriod()
{
    auto lastDate = std::min(
        mwr::lastRouteLostDate(),
        mwr::lastMatchedTracksDate()
    );

    return {
        lastDate - bg::date_duration(AVERAGING_PERIOD - 1),
        lastDate + bg::date_duration(1)
    };
}

} // unnamed namespace

int main(int argc, char* argv[])
{
    NYT::Initialize(argc, const_cast<const char**>(argv));

    try {
        INFO() << "Finding route losts started";

        auto configXml = mwc::loadDefaultConfig();

        mwc::PoolHolder socialPool(*configXml, "social", "grinder");
        maps::pgp3utils::PgAdvisoryXactMutex locker(socialPool.pool(),
            static_cast<int64_t>(mwc::AdvisoryLockIds::ROUTE_LOSTS_PRODUCER));

        if (!locker.try_lock()) {
            INFO() << "Database is already locked. Task interrupted.";
            return CODE_LOCKED;
        }


        mwr::WorkerArgs args;
        args.polygonGeo = mg::Polygon2(worldPolygon);
        args.datesPeriod = workerDatesPeriod();
        args.maxFeedbackTasksToPost = MAX_FEEDBACK_TASKS;

        mwr::Worker worker(args, socialPool.pool());
        worker.createFeedbackTasks();

        INFO() << "Finding route losts finished";

    } catch (const maps::Exception& e) {
        ERROR() << "Worker failed: " << e;
        return CODE_FAILURE;
    } catch (const std::exception& e) {
        ERROR() << "Worker failed: " << e.what();
        return CODE_FAILURE;
    } catch (...) {
        ERROR() << "Worker failed: unknown exception";
        return CODE_FAILURE;
    }

    return CODE_SUCCESS;
}
