#include "score_mapper.h"
#include "column_names.h"

namespace maps::wiki::route_lost_feedback {

namespace {

const int MIN_LOST_PER_DAY = 2;

}

ScoreMapper::ScoreMapper(uint64_t countDays) :
    countDays_(countDays)
{
}

// Score is ratio of route-lost number to track number,
// multiplied by -1. Inversion is done because we want
// losts with maximal score to be in the beginning of sorted
// result table
//
void ScoreMapper::Do(TReader* reader, TWriter* writer)
{
    for (; reader->IsValid(); reader->Next()) {
        const NYT::TNode& row = reader->GetRow();

        uint64_t lostsNumber = row[column_names::LOSTS_NUMBER].AsUint64();
        uint64_t tracksNumber = row[column_names::TRACKS_NUMBER].AsUint64();

        if (lostsNumber >= MIN_LOST_PER_DAY * countDays_ && tracksNumber > lostsNumber) {
            double score = -static_cast<double>(lostsNumber) / tracksNumber;
            writer->AddRow(NYT::TNode(row)(column_names::INVERTED_SCORE, score));
        }
    }
}

REGISTER_MAPPER(ScoreMapper);

} // namespace maps::wiki::route_lost_feedback
