#include "worker.h"
#include "yt_consts.h"
#include "table_names.h"
#include "yt_operations.h"
#include "yt_row_to_feedback.h"

#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/common/yt.h>
#include <yandex/maps/wiki/common/robot.h>
#include <yandex/maps/wiki/social/feedback/task.h>
#include <yandex/maps/wiki/social/feedback/twins.h>
#include <yandex/maps/wiki/social/feedback/attribute_names.h>
#include <yandex/maps/wiki/social/feedback/attributes.h>
#include "serialization.h"

namespace bg = boost::gregorian;
namespace sf = maps::wiki::social::feedback;

namespace maps::wiki::route_lost_feedback {

namespace {

const std::vector<std::string> ROUTE_LOST_ATTRS {
    sf::attrs::BEFORE_LOST_SEGMENT,
    sf::attrs::AFTER_LOST_ROUTE_SEGMENT,
    sf::attrs::AFTER_LOST_TRACK_SEGMENT
};

bool tasksAreDuplicated(const sf::TaskNew& newTask, const sf::Task& oldTask) {
    const auto& newTaskAttrs = newTask.attrs.get(sf::AttrType::UserData);
    const auto& oldTaskAttrs = oldTask.attrs().get(sf::AttrType::UserData);

    for (const auto& attrName : ROUTE_LOST_ATTRS) {
        if (fromJson(newTaskAttrs[attrName]) != fromJson(oldTaskAttrs[attrName])) {
            return false;
        }
    }
    return true;
}

void printTableNames(const std::vector<std::string>& tables)
{
    for (const auto& table : tables) {
        INFO() << table;
    }
}

} // unnamed namespace


WorkerArgs::WorkerArgs() :
    datesPeriod(bg::date(), bg::date())
{}

Worker::Worker(
    WorkerArgs args,
    pgpool3::Pool& socialPool) :
        args_(std::move(args)),
        socialPool_(socialPool)
{}

uint64_t Worker::createFeedbackTasks()
{
    INFO() << "YT operations started";

    auto ytClient = common::yt::createYtClient(TString(YT_PROXY));
    auto ytInputTables = createYtInputTableNames(args_.datesPeriod, *ytClient.Get());
    auto ytTmpTables = createYtTmpTableNames();

    INFO() << "Route lost input tables:";
    printTableNames(ytInputTables.routeLost);
    INFO() << "Tracks input tables:";
    printTableNames(ytInputTables.tracks);

    YtOperations ytOps(
        ytClient,
        args_.polygonGeo,
        ytInputTables,
        ytTmpTables);

    ytOps.runAll();

    INFO() << "YT operations finished";
    INFO() << "Publishing feedback to social started";

    auto ytReader = ytClient->CreateTableReader<NYT::TNode>(
        TString(ytTmpTables.routeLostScoredSorted)
    );

    auto socialTxn = socialPool_.masterWriteableTransaction();
    sf::GatewayRW gatewayRw(socialTxn.get());

    sf::Tasks insertedTasks;
    for (const auto& ytRow = ytReader->GetRow(); ytReader->IsValid(); ytReader->Next()) {
        if (insertedTasks.size() == args_.maxFeedbackTasksToPost) {
            break;
        }
        auto insertedTask = sf::addTaskIfNotTwin(
            gatewayRw,
            common::ROBOT_UID,
            ytRowToFeedback(ytRow, args_.datesPeriod),
            tasksAreDuplicated);

        if (insertedTask) {
            insertedTasks.push_back(std::move(insertedTask.value()));
        }
    }

    socialTxn->commit();

    INFO() << "Publishing feedback to social finished";
    INFO() << "In total " << insertedTasks.size() << " new feedback";

    return insertedTasks.size();
}

} // namespace maps::wiki::route_lost_feedback
