#pragma once

#include "address.h"
#include <maps/search/geocoder/segmenter/interface/factory.h>
#include <unicode/regex.h>
#include <optional>

// Address normalizing motivation
//
// Building addresses can be generated by users and
// even by programmers in arbitrary form. When we want
// to process or to compare such addresses we should
// have them in similar form - canonized or normalized.
//
// E.g. entries below are practically the same address:
//  улица Ленина, 5к1
//  улица Ленина 5 к1
//  Ленина улица, 5к1
//  Ленина ул. дом 5 к 1
//

namespace maps::wiki::schedule_feedback {

class IAddressNormalizer
{
public:
    virtual std::optional<std::string>
    normalizeStreet(const std::string& street) const = 0;

    virtual std::optional<std::string>
    normalizeHouse(const std::string& house) const = 0;

    virtual std::optional<NormalizedAddress>
    normalize(const RawAddress& address) const = 0;

    virtual std::optional<NormalizedAddress>
    normalize(const std::string& address) const = 0;

    virtual ~IAddressNormalizer() = default;
};

class AddressNormalizer : public IAddressNormalizer
{
public:
    AddressNormalizer(const std::string& grammarPath);

    std::optional<std::string>
    normalizeStreet(const std::string& street) const override;

    std::optional<std::string>
    normalizeHouse(const std::string& house) const override;

    std::optional<NormalizedAddress>
    normalize(const RawAddress& address) const override;

    std::optional<NormalizedAddress>
    normalize(const std::string& address) const override;

private:
    std::unique_ptr<const segmenter::AddressSplitter> splitter_;
    std::vector<std::unique_ptr<icu::RegexMatcher>> streetMarkersMatchers_;
};

} // namespace maps::wiki::schedule_feedback
