#include "report_addr_strategy.h"

#include "address_normalizer.h"
#include "log_action.h"
#include "report_addr.h"
#include "report_addr_exist.h"
#include "report_addr_has_street.h"
#include "report_addr_obj_is_addr.h"

namespace maps::wiki::schedule_feedback {

namespace {

const bool STREET_VICINITY_HIDE_INSTEAD_REJECT_ENABLED = true;

} // unnamed namespace

ReportAddressStrategy::ReportAddressStrategy(
        std::unique_ptr<IAddressNormalizer> normalizer,
        std::unique_ptr<IObjectIsAddressChecker> isAddrChecker,
        std::unique_ptr<IAddressExistsChecker> existsChecker,
        std::unique_ptr<IHasStreetChecker> hasStreetChecker) :
    normalizer_(std::move(normalizer)),
    isAddrChecker_(std::move(isAddrChecker)),
    existsChecker_(std::move(existsChecker)),
    hasStreetChecker_(std::move(hasStreetChecker))
{}

ReportAddrAction
ReportAddressStrategy::action(
    const ReportAddressFb& fb) const try
{
    auto fbAddrNorm = normalizer_->normalize(fb.address);
    if (!fbAddrNorm) {
        return ReportAddrRejectAction{ReportAddrRejectAction::Reason::NonNormalizable};
    }

    if (!isAddrChecker_->isAddressObject(fb.addressObjectId)) {
        return ReportAddrRejectAction{ReportAddrRejectAction::Reason::ObjectIsNotAddress};
    }

    if (existsChecker_->addressExists(fb.addressObjectId, *fbAddrNorm)) {
        return ReportAddrAcceptAction{};
    }

    if (!hasStreetChecker_->hasStreetInVicinity(fb.positionMerc, *fbAddrNorm)) {
        if (STREET_VICINITY_HIDE_INSTEAD_REJECT_ENABLED) {
            return ReportAddrHideAction{};
        } else {
            return ReportAddrRejectAction{ReportAddrRejectAction::Reason::NoStreetNearby};
        }
    }

    return ReportAddrNoneAction{};

} catch (const std::exception&) {
    // It happens when some assertions fail during
    // work with TDS, namely when 'Address' object
    // doesn't have corresponding street name
    // TODO: razberis
    //
    return ReportAddrNoneAction{};
}

} // namespace maps::wiki::schedule_feedback
