#include "traits.h"
#include "consts.h"
#include <yandex/maps/wiki/social/feedback/attribute_names.h>
#include <maps/libs/http/include/client.h>
#include <maps/libs/http/include/request.h>
#include <maps/libs/json/include/exception.h>
#include <maps/libs/log8/include/log8.h>

namespace maps::wiki::schedule_feedback {

namespace sf = social::feedback;
namespace sfa = social::feedback::attrs;

bool isFbapiTask(const sf::Task& task)
{
    return task.source() == FBAPI;
}

bool isAddEntranceTask(const sf::Task& task)
{
    return task.type() == sf::Type::Entrance &&
           task.subSource() == "qid__add_object__aid__entrance";
}

bool isAddressReportAddressTask(const sf::Task& task)
{
    return task.type() == sf::Type::Address &&
           task.subSource() == "qid__wrong_address__aid__report_address";
}

// completely deprecated: check database for existing such feedback
bool isAddressReportLocationTask(const sf::Task& task)
{
    return task.type() == sf::Type::Address &&
           task.subSource() == "qid__wrong_address__aid__report_location";
}

bool isAddressTask(const sf::Task& task)
{
    return task.type() == sf::Type::Address;
}

namespace {

const std::string CLEANWEB_URL = "http://router-prod.clean-web.yandex.net/v2";
const std::set<std::string> BAD_VERDICTS{
    "media_auto_bad", // media verdicts for photos
    "media_auto_erotica",
    "media_auto_porno",
    "media_auto_shock",
    "strict", // verdict for antifraud
};

std::string createCleanWebRequestBody(const std::string& photoUrl, const std::string& photoId)
{
    return R"({
        "jsonrpc": "2.0",
        "method": "process",
        "params": {
            "service": "maps_feedback",
            "type": "image",
            "key": "maps_feedback_photo",
            "body": {"image_url": ")" + photoUrl + R"("}
        },
        "id": ")" + photoId + R"("
    })";
}

std::string createAntifraudRequestBody(const sf::Task& task)
{
    maps::json::Builder builder;
    builder << [&](maps::json::ObjectBuilder b) {
        b["id"] = std::to_string(task.id());
        b["jsonrpc"] = "2.0";
        b["method"] = "process";
        b["params"] << [&](maps::json::ObjectBuilder b) {
            b["key"] = std::to_string(task.id());
            b["type"] = "text";
            // Permalink and altay_feedback is crutch: antifraud treats request as
            // feedback for organization
            // Settings for separate service for maps feedback in antifraud
            // take more time
            b["service"] = "altay_feedback";
            b["body"] << [&](maps::json::ObjectBuilder b) {
                b["permalink"] = "";
                b["text"] = task.attrs().existCustom(sfa::USER_DATA_COMMENT)
                    ? task.attrs().getCustom(sfa::USER_DATA_COMMENT)
                    : "";
                b["lat"] = task.position().y();
                b["lon"] = task.position().x();
                if (task.attrs().existCustom(sfa::IP)) {
                    b["ip"] = task.attrs().getCustom(sfa::IP);
                }
                if (task.attrs().existCustom(sfa::UID)) {
                    b["puid"] = task.attrs().getCustom(sfa::UID);
                }
                if (task.attrs().existCustom(sfa::YANDEXUID)) {
                    b["yandex_uid"] = task.attrs().getCustom(sfa::YANDEXUID);
                }
                if (task.attrs().existCustom(sfa::USER_AGENT)) {
                    b["user_agent"] = task.attrs().getCustom(sfa::USER_AGENT);
                }
                if (task.attrs().existCustom(sfa::FINGERPRINT)) {
                    b["fingerprint"] = task.attrs().getCustom(sfa::FINGERPRINT);
                }
            };
        };
    };
    return builder.str();
}

bool hasInappropriateVerdict(const std::string& response)
try {
    const auto& json = maps::json::Value::fromString(response);
    for (const auto& verdict : json["result"]["verdicts"]) {
        if (BAD_VERDICTS.contains(verdict["name"].as<std::string>())) {
            return true;
        }
    }
    return false;
} catch (maps::json::Error& e) {
    WARN() << "Cannot parse json in response: " << e << "\nResponse: " << response;
    // treat photo as good if cannot parse response
    return false;
}

} // namespace

bool hasBadPhotos(const sf::Task& task)
{
    if (task.attrs().exist(sf::AttrType::UserDataPhotoUrls)) {
        return internal::hasBadPhotos(task.attrs().get(sf::AttrType::UserDataPhotoUrls));
    }
    return false;
}

bool isFraudFeedback(const sf::Task& task)
{
    maps::http::Client client;
    const auto& body = createAntifraudRequestBody(task);

    maps::http::Request request(client, maps::http::POST, CLEANWEB_URL);
    request.addHeader("Connection", "close");
    request.addHeader("Content-Type", "application/json");
    request.setContent(body);
    maps::http::Response response = request.perform();

    if (response.status() == 200 && hasInappropriateVerdict(response.readBody())) {
        return true;
    }
    return false;
}

namespace internal {

bool hasBadPhotos(const json::Value& jsonPhotos)
{
    maps::http::Client client;

    for (const auto& jsonPhoto : jsonPhotos) {
        const auto& strPhotoUrl = jsonPhoto.as<std::string>();
        const auto& body = createCleanWebRequestBody(strPhotoUrl, strPhotoUrl);

        maps::http::Request request(client, maps::http::POST, CLEANWEB_URL);
        request.addHeader("Connection", "close");
        request.addHeader("Content-Type", "application/json");
        request.setContent(body);
        maps::http::Response response = request.perform();

        if (response.status() == 200 && hasInappropriateVerdict(response.readBody())) {
            return true;
        }
    }
    return false;
}

} // namespace internal

} // namespace maps::wiki::schedule_feedback
