#include "../fixtures.h"
#include "../../lib/fbapi/address.h"
#include "../../lib/fbapi/address_normalizer.h"

#include <library/cpp/testing/unittest/env.h>
#include <library/cpp/testing/unittest/registar.h>

namespace maps::wiki::schedule_feedback::tests {

namespace {

// BuildRoot here is crucial. Because segmenter files
// are brought with DEPENDS, we should garantee that
// test run takes place after the files are generated.
// In case of ArcadiaRoot the first test run will fail due to
// lack of grammar files, but the second will succeed
//
const std::string ARCADIA_GRAMMAR_PATH =
    BuildRoot() + "/maps/search/geocoder/segmenter/data";

struct AddressNormalizerFixture : AddressNormalizer, NUnitTest::TBaseFixture
{
    AddressNormalizerFixture() :
        AddressNormalizer(ARCADIA_GRAMMAR_PATH)
    {}
};

} // namespace anonymous

Y_UNIT_TEST_SUITE_F(address_normalizer_street, AddressNormalizerFixture) {

Y_UNIT_TEST(base_street)
{
    auto raw = "улица ленина";
    auto norm = normalizeStreet(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "ленина");
}

Y_UNIT_TEST(lower_case)
{
    auto raw = "улица Ленина";
    auto norm = normalizeStreet(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "ленина");
}

Y_UNIT_TEST(shortened_marker)
{
    auto raw = "ул Ленина";
    auto norm = normalizeStreet(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "ленина");
}

Y_UNIT_TEST(no_street_marker_case)
{
    auto raw = "Ленина";
    auto norm = normalizeStreet(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "ленина");
}

Y_UNIT_TEST(other_street_marker_case)
{
    auto raw = "проспект Ленина";
    auto norm = normalizeStreet(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "ленина");
}

Y_UNIT_TEST(complex_street)
{
    auto raw = "улица 4-й добрынинский";
    auto norm = normalizeStreet(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "4 добрынинский");
}

Y_UNIT_TEST(complex_street_other_marker)
{
    auto raw = "4-й добрынинский переулок";
    auto norm = normalizeStreet(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "4 добрынинский");
}

Y_UNIT_TEST(more_complex_street)
{
    auto raw = "улица 40 лет Октября";
    auto norm = normalizeStreet(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "40 лет октября");
}

Y_UNIT_TEST(street_with_number)
{
    auto raw = "улица ленина, 7";
    auto norm = normalizeStreet(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "ленина");
}

Y_UNIT_TEST(street_with_house_number)
{
    auto raw = "улица ленина, дом 7";
    auto norm = normalizeStreet(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "ленина");
}

Y_UNIT_TEST(degenerate_street)
{
    auto raw = "4-я линия";
    auto norm = normalizeStreet(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "4");
}

Y_UNIT_TEST(trash_street)
{
    auto raw = "город абинск станица холмская улица Горького";
    auto norm = normalizeStreet(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "горького");
}

} // address_normalizer_street


Y_UNIT_TEST_SUITE_F(address_normalizer_house, AddressNormalizerFixture) {

Y_UNIT_TEST(base_house)
{
    auto raw = "8";
    auto norm = normalizeHouse(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "8");
}

Y_UNIT_TEST(big_house)
{
    auto raw = "999";
    auto norm = normalizeHouse(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "999");
}

Y_UNIT_TEST(block)
{
    auto raw = "5к1";
    auto norm = normalizeHouse(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "5к1");
}

Y_UNIT_TEST(block_caps)
{
    auto raw = "5К1";
    auto norm = normalizeHouse(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "5к1");
}

Y_UNIT_TEST(block_space_left)
{
    auto raw = "5 к1";
    auto norm = normalizeHouse(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "5к1");
}

Y_UNIT_TEST(block_space_right)
{
    auto raw = "5к 1";
    auto norm = normalizeHouse(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "5к1");
}

Y_UNIT_TEST(block_space_both)
{
    auto raw = "5 к 1";
    auto norm = normalizeHouse(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "5к1");
}

Y_UNIT_TEST(block_name_mid)
{
    auto raw = "5 корп 1";
    auto norm = normalizeHouse(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "5к1");
}

Y_UNIT_TEST(block_name_full)
{
    auto raw = "5 корпус 1";
    auto norm = normalizeHouse(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "5к1");
}

Y_UNIT_TEST(block_name_full_stick)
{
    auto raw = "5 корпус1";
    auto norm = normalizeHouse(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "5к1");
}

Y_UNIT_TEST(building_name_base)
{
    auto raw = "5с1";
    auto norm = normalizeHouse(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "5с1");
}

Y_UNIT_TEST(building_name_mid)
{
    auto raw = "5стр1";
    auto norm = normalizeHouse(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "5с1");
}

Y_UNIT_TEST(building_name_full)
{
    auto raw = "5строение1";
    auto norm = normalizeHouse(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "5с1");
}

Y_UNIT_TEST(building_name_house)
{
    auto raw = "дом 1";
    auto norm = normalizeHouse(raw);
    UNIT_ASSERT(norm != std::nullopt);
    UNIT_ASSERT_STRINGS_EQUAL(*norm, "1");
}

} // address_normalizer_house


Y_UNIT_TEST_SUITE_F(address_normalizer_other_countries, AddressNormalizerFixture) {

Y_UNIT_TEST(ukr)
{
    RawAddress raw1{"Політехнічна вулиця", "37к9"};
    UNIT_ASSERT(normalize(raw1) == (NormalizedAddress{"політехнічна", "37к9"}));

    RawAddress raw2{"вулиця Жамбила Жабаєва", "7"};
    UNIT_ASSERT(normalize(raw2) == (NormalizedAddress{"жамбила жабаєва", "7"}));

    RawAddress raw3{"провулок Авіаконструктора Ігоря Сікорського", "8"};
    UNIT_ASSERT(normalize(raw3) == (NormalizedAddress{"авіаконструктора ігоря сікорського", "8"}));
}

Y_UNIT_TEST(tr)
{
    RawAddress raw1{"Yavuz Cad.", "3A"};
    UNIT_ASSERT(normalize(raw1) == (NormalizedAddress{"yavuz", "3a"}));

    RawAddress raw2{"Hasan Tahsin Cad.", "128"};
    UNIT_ASSERT(normalize(raw2) == (NormalizedAddress{"tahsin", "128"}));

    RawAddress raw3{"Atatürk Blv.", "2/47"};
    UNIT_ASSERT(normalize(raw3) == (NormalizedAddress{"atatürk", "2/47"}));
}

} // address_normalizer_other_countries


Y_UNIT_TEST_SUITE(address_normalizer_string) {

Y_UNIT_TEST_F(alltogether, AddressNormalizerFixture)
{
    std::string raw1 = "Салютний проїзд, 5";
    UNIT_ASSERT(normalize(raw1) == (NormalizedAddress{"салютний", "5"}));

    std::string raw2 = "Антоновская улица, 6";
    UNIT_ASSERT(normalize(raw2) == (NormalizedAddress{"антоновская", "6"}));

    std::string raw3 = "Волго-Донской проспект, 19 корп 3";
    UNIT_ASSERT(normalize(raw3) == (NormalizedAddress{"волго-донской", "19к3"}));
}

} // address_normalizer_string

} // namespace maps::wiki::schedule_feedback::tests
