#include "../mocks.h"
#include "../../lib/fbapi/report_addr.h"

#include <library/cpp/testing/unittest/registar.h>

namespace maps::wiki::schedule_feedback::tests {

namespace {

const double VICINITY_RADIUS = 10.;
geolib3::Point2 COORDS_ORIGIN(0, 0);
const std::string NORM_STREET = "нормальный";
const std::string NORM_STREET_CAPS = "НОРМАЛЬНЫЙ";
const NormalizedAddress NORM_ADDRESS{NORM_STREET, "короче"};

} // namespace anonymous

Y_UNIT_TEST_SUITE(report_address_has_street) {

Y_UNIT_TEST(bbox_is_empty)
{
    auto rev = std::make_unique<RevisionAdapterMock>();
    EXPECT_CALL(*rev, streetNamesFromBBox(_))
        .WillOnce(Return(std::vector<std::string>{}));

    auto norm = std::make_unique<AddressNormalizerMock>();
    EXPECT_CALL(*norm, normalizeStreet(_))
        .Times(0);

    HasStreetChecker check(VICINITY_RADIUS, std::move(rev), std::move(norm));
    UNIT_ASSERT(check.hasStreetInVicinity(COORDS_ORIGIN, NORM_ADDRESS) == false);
}

Y_UNIT_TEST(bbox_contains_other_streets)
{
    const std::string STREET_1 = "street 1";
    const std::string STREET_2 = "street 2";

    const RawAddress RAW_ADDRESS_1{STREET_1, ""};
    const RawAddress RAW_ADDRESS_2{STREET_2, ""};

    auto rev = std::make_unique<RevisionAdapterMock>();
    EXPECT_CALL(*rev, streetNamesFromBBox(_))
        .WillOnce(Return(std::vector<std::string>{STREET_1, STREET_2}));

    // identical normalizer
    auto norm = std::make_unique<AddressNormalizerMock>();
    EXPECT_CALL(*norm, normalizeStreet(_))
        .Times(2)
        .WillRepeatedly(ReturnArg<0>());

    HasStreetChecker check(VICINITY_RADIUS, std::move(rev), std::move(norm));
    UNIT_ASSERT(check.hasStreetInVicinity(COORDS_ORIGIN, NORM_ADDRESS) == false);
}

Y_UNIT_TEST(normalization_fail)
{
    auto rev = std::make_unique<RevisionAdapterMock>();
    EXPECT_CALL(*rev, streetNamesFromBBox(_))
        .WillOnce(Return(std::vector<std::string>{NORM_STREET}));

    // failing normalizer
    auto norm = std::make_unique<AddressNormalizerMock>();
    EXPECT_CALL(*norm, normalizeStreet(_))
        .WillOnce(Return(std::nullopt));

    HasStreetChecker check(VICINITY_RADIUS, std::move(rev), std::move(norm));
    UNIT_ASSERT(check.hasStreetInVicinity(COORDS_ORIGIN, NORM_ADDRESS) == false);
}

Y_UNIT_TEST(street_exists_and_normalizer_works)
{
    auto rev = std::make_unique<RevisionAdapterMock>();
    EXPECT_CALL(*rev, streetNamesFromBBox(_))
        .WillOnce(Return(std::vector<std::string>{NORM_STREET_CAPS}));

    auto norm = std::make_unique<AddressNormalizerMock>();
    EXPECT_CALL(*norm, normalizeStreet(NORM_STREET_CAPS))
        .WillOnce(Return(NORM_STREET));

    HasStreetChecker check(VICINITY_RADIUS, std::move(rev), std::move(norm));
    UNIT_ASSERT(check.hasStreetInVicinity(COORDS_ORIGIN, NORM_ADDRESS));
}

} // Y_UNIT_TEST_SUITE

} // namespace maps::wiki::schedule_feedback::tests
