#include "../mocks.h"
#include "../../lib/fbapi/report_addr.h"
#include "../../lib/fbapi/report_addr_action.h"
#include "../../lib/fbapi/report_addr_strategy.h"

#include <library/cpp/testing/unittest/registar.h>

using namespace std::chrono_literals;

namespace maps::wiki::schedule_feedback::tests {

namespace {

const chrono::TimePoint CREATED_AT = chrono::TimePoint() + 24h;
geolib3::Point2 COORDS_ORIGIN(0, 0);
const TId FB_ID = 321;
const ObjId ADDR_OBJ_ID = 123;
const RawAddress RAW_ADDR{"ВРАНГЕЛЯ", "23"};
const NormalizedAddress NORM_ADDR{"врангеля", "23"};

ReportAddressFb REPORT_ADDR{
    FB_ID,
    CREATED_AT,
    COORDS_ORIGIN,
    ADDR_OBJ_ID,
    RAW_ADDR
};

} // namespace anonymous

Y_UNIT_TEST_SUITE(report_address_reject_checker) {

Y_UNIT_TEST(rejected_by_normalizer)
{
    auto norm = std::make_unique<AddressNormalizerMock>();
    EXPECT_CALL(*norm, normalize(An<const RawAddress&>()))
        .WillOnce(Return(std::nullopt));

    auto isAddr = std::make_unique<ObjectIsAddressCheckerMock>();
    EXPECT_CALL(*isAddr, isAddressObject(_)).Times(0);

    auto exists = std::make_unique<AddressExistsCheckerMock>();
    EXPECT_CALL(*exists, addressExists(_, _)).Times(0);

    auto hasStreet = std::make_unique<HasStreetCheckerMock>();
    EXPECT_CALL(*hasStreet, hasStreetInVicinity(_, _)).Times(0);

    ReportAddressStrategy strategy(
        std::move(norm),
        std::move(isAddr),
        std::move(exists),
        std::move(hasStreet));

    auto action = strategy.action(REPORT_ADDR);
    UNIT_ASSERT(std::holds_alternative<ReportAddrRejectAction>(action));
    UNIT_ASSERT(std::get<ReportAddrRejectAction>(action).reason ==
                ReportAddrRejectAction::Reason::NonNormalizable);
}

Y_UNIT_TEST(rejected_by_object_is_address_checker)
{
    auto norm = std::make_unique<AddressNormalizerMock>();
    EXPECT_CALL(*norm, normalize(RAW_ADDR))
        .WillOnce(Return(NORM_ADDR));

    auto isAddr = std::make_unique<ObjectIsAddressCheckerMock>();
    EXPECT_CALL(*isAddr, isAddressObject(ADDR_OBJ_ID))
        .WillOnce(Return(false));

    auto exists = std::make_unique<AddressExistsCheckerMock>();
    EXPECT_CALL(*exists, addressExists(_, _)).Times(0);

    auto hasStreet = std::make_unique<HasStreetCheckerMock>();
    EXPECT_CALL(*hasStreet, hasStreetInVicinity(_, _)).Times(0);

    ReportAddressStrategy strategy(
        std::move(norm),
        std::move(isAddr),
        std::move(exists),
        std::move(hasStreet));

    auto action = strategy.action(REPORT_ADDR);
    UNIT_ASSERT(std::holds_alternative<ReportAddrRejectAction>(action));
    UNIT_ASSERT(std::get<ReportAddrRejectAction>(action).reason ==
                ReportAddrRejectAction::Reason::ObjectIsNotAddress);
}

Y_UNIT_TEST(rejected_by_address_exist_checker)
{
    auto norm = std::make_unique<AddressNormalizerMock>();
    EXPECT_CALL(*norm, normalize(RAW_ADDR))
        .WillOnce(Return(NORM_ADDR));

    auto isAddr = std::make_unique<ObjectIsAddressCheckerMock>();
    EXPECT_CALL(*isAddr, isAddressObject(ADDR_OBJ_ID))
        .WillOnce(Return(true));

    auto exists = std::make_unique<AddressExistsCheckerMock>();
    EXPECT_CALL(*exists, addressExists(ADDR_OBJ_ID, NORM_ADDR))
        .WillOnce(Return(true));

    auto hasStreet = std::make_unique<HasStreetCheckerMock>();
    EXPECT_CALL(*hasStreet, hasStreetInVicinity(_, _)).Times(0);

    ReportAddressStrategy strategy(
        std::move(norm),
        std::move(isAddr),
        std::move(exists),
        std::move(hasStreet));

    auto action = strategy.action(REPORT_ADDR);
    UNIT_ASSERT(std::holds_alternative<ReportAddrAcceptAction>(action));
}

Y_UNIT_TEST(hide_instead_of_rejected_by_no_street_checker)
{
    auto norm = std::make_unique<AddressNormalizerMock>();
    EXPECT_CALL(*norm, normalize(RAW_ADDR))
        .WillOnce(Return(NORM_ADDR));

    auto isAddr = std::make_unique<ObjectIsAddressCheckerMock>();
    EXPECT_CALL(*isAddr, isAddressObject(_))
        .WillOnce(Return(true));

    auto exists = std::make_unique<AddressExistsCheckerMock>();
    EXPECT_CALL(*exists, addressExists(_, _))
        .WillOnce(Return(false));

    auto hasStreet = std::make_unique<HasStreetCheckerMock>();
    EXPECT_CALL(*hasStreet, hasStreetInVicinity(COORDS_ORIGIN, NORM_ADDR))
        .WillOnce(Return(false));

    ReportAddressStrategy strategy(
        std::move(norm),
        std::move(isAddr),
        std::move(exists),
        std::move(hasStreet));

    auto action = strategy.action(REPORT_ADDR);
    UNIT_ASSERT(std::holds_alternative<ReportAddrHideAction>(action));
}

Y_UNIT_TEST(should_not_be_rejected)
{
    auto norm = std::make_unique<AddressNormalizerMock>();
    EXPECT_CALL(*norm, normalize(RAW_ADDR))
        .WillOnce(Return(NORM_ADDR));

    auto isAddr = std::make_unique<ObjectIsAddressCheckerMock>();
    EXPECT_CALL(*isAddr, isAddressObject(_))
        .WillOnce(Return(true));

    auto exists = std::make_unique<AddressExistsCheckerMock>();
    EXPECT_CALL(*exists, addressExists(_, _))
        .WillOnce(Return(false));

    auto hasStreet = std::make_unique<HasStreetCheckerMock>();
    EXPECT_CALL(*hasStreet, hasStreetInVicinity(_, _))
        .WillOnce(Return(true));

    ReportAddressStrategy strategy(
        std::move(norm),
        std::move(isAddr),
        std::move(exists),
        std::move(hasStreet));

    auto action = strategy.action(REPORT_ADDR);
    UNIT_ASSERT(std::holds_alternative<ReportAddrNoneAction>(action));
}

} // Y_UNIT_TEST_SUITE

} // namespace maps::wiki::schedule_feedback::tests
