#include <maps/wikimap/mapspro/services/tasks_feedback/src/sprav_pedestrian_feedback/lib/common.h>
#include <maps/wikimap/mapspro/services/tasks_feedback/src/sprav_pedestrian_feedback/lib/print_info.h>
#include <maps/wikimap/mapspro/services/tasks_feedback/src/sprav_pedestrian_feedback/lib/load_from_yt.h>
#include <maps/wikimap/mapspro/services/tasks_feedback/src/sprav_pedestrian_feedback/lib/addresses/common.h>
#include <maps/wikimap/mapspro/services/tasks_feedback/src/sprav_pedestrian_feedback/lib/entrances/common.h>
#include <maps/wikimap/mapspro/services/tasks_feedback/src/sprav_pedestrian_feedback/lib/feedback/common.h>

#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/libs/common/include/exception.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/pgpool3utils/pg_advisory_mutex.h>
#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

#include <optional>

namespace mwc = maps::wiki::common;
namespace mwt = maps::wiki::tasks_feedback;
namespace mws = maps::wiki::sprav_feedback;

namespace {

const int EXIT_LOCKED = 5;

const std::string CONFIG_SOCIAL_BACKOFFICE_URL = "/config/services/social-backoffice/url";
const std::string CONFIG_EDITOR_URL = "/config/services/editor/url";

const std::string ACT_PUBLISH_FEEDBACK = "publish-feedback";
const std::string ACT_ACCEPT_ENTRANCES = "accept-entrances";
const std::string ACT_ACCEPT_ADDRESSES = "accept-addresses";

template<typename RawDataStruct>
int worker(
    maps::pgpool3::Pool& corePool,
    std::optional<mwt::TimeIntervalMs> timeIntervalMs,
    const mws::ProcessParams<RawDataStruct>& config)
{
    // Exclusive process execution between hosts
    //
    maps::pgp3utils::PgAdvisoryXactMutex dbCoreLocker(corePool,
        static_cast<int64_t>(mws::lockId<RawDataStruct>()));

    if (!dbCoreLocker.try_lock()) {
        INFO() << "Database is already locked. Task interrupted.";
        return EXIT_LOCKED;
    }

    mws::printWorkerArguments(
        mws::lastTimeTableName<RawDataStruct>(),
        mws::ytTableName<RawDataStruct>());

    if (timeIntervalMs) {
        mws::loadByHand<RawDataStruct>(
            dbCoreLocker.writableTxn(),
            timeIntervalMs.value(),
            config);
    } else {
        mws::loadPeriodical<RawDataStruct>(
            corePool,
            config);
    }
    return EXIT_SUCCESS;
}

} // unnamed namespace

int main(int argc, char* argv[]) try
{
    maps::cmdline::Parser parser;

    auto actionString = parser.string("action").required().
        help("Action: '" + ACT_PUBLISH_FEEDBACK
            + "' or '" + ACT_ACCEPT_ENTRANCES + "'"
            + "' or '" + ACT_ACCEPT_ADDRESSES + "'");

    auto configPath = parser.string("config-path").
        help("Path to configuration file, where path to "
             "social service exists");

    auto fromTime = parser.size_t("from-time").
        help("Util will process data from YT\n"
             "from submit time interval ['from-time', 'till-time').\n"
             "Goes in conjunction with 'till-time'. In unix ms.");

    auto tillTime = parser.size_t("till-time").
        help("Goes in conjunction with 'from-time'");

    parser.parse(argc, argv);

    // Time interval verification
    //
    std::optional<mwt::TimeIntervalMs> timeInterval;
    {
        bool bothTimePresent = fromTime.defined() && tillTime.defined();
        bool bothTimeAbsent = !fromTime.defined() && !tillTime.defined();
        REQUIRE(bothTimePresent || bothTimeAbsent, "Time interval is partially specified");
        if (bothTimePresent) {
            timeInterval = {fromTime, tillTime};
        }
    }

    // Load config
    //
    std::unique_ptr<mwc::ExtendedXmlDoc> configXml;
    if (configPath.defined()) {
        configXml.reset(new mwc::ExtendedXmlDoc(configPath));
    } else {
        configXml = mwc::loadDefaultConfig();
    }

    //
    mwc::PoolHolder corePoolHolder(*configXml, "core", "grinder");

    int retVal = EXIT_FAILURE;
    if (ACT_PUBLISH_FEEDBACK == actionString) {
        retVal = worker<mws::SpravFeedback>(
            corePoolHolder.pool(),
            timeInterval,
            {configXml->get<std::string>(CONFIG_SOCIAL_BACKOFFICE_URL)});
    } else if (ACT_ACCEPT_ENTRANCES == actionString) {
        mwc::PoolHolder viewTrunkPoolHolder(*configXml, "view-trunk", "view-trunk");
        auto viewTrunkTxnHolder = viewTrunkPoolHolder.pool().masterReadOnlyTransaction();
        viewTrunkTxnHolder->exec("SET search_path=vrevisions_trunk,public");

        retVal = worker<mws::EntrancesData>(
            corePoolHolder.pool(),
            timeInterval,
            {
                configXml->get<std::string>(CONFIG_SOCIAL_BACKOFFICE_URL),
                configXml->get<std::string>(CONFIG_EDITOR_URL),
                viewTrunkTxnHolder.get()
            });
    } else if (ACT_ACCEPT_ADDRESSES == actionString) {
        retVal = worker<mws::AddressData>(
            corePoolHolder.pool(),
            timeInterval,
            {configXml->get<std::string>(CONFIG_SOCIAL_BACKOFFICE_URL)});
    } else {
        parser.printUsageAndExit(EXIT_FAILURE);
    }

    return retVal;

} catch (const maps::Exception& e) {
    FATAL() << e;
    return EXIT_FAILURE;
} catch (const std::exception& e) {
    FATAL() << e.what();
    return EXIT_FAILURE;
}
