#include "publish.h"
#include <maps/wikimap/mapspro/services/tasks_feedback/src/sprav_pedestrian_feedback/lib/parser.h>
#include <maps/wikimap/mapspro/services/tasks_feedback/src/sprav_pedestrian_feedback/lib/social.h>
#include <maps/wikimap/mapspro/services/tasks_feedback/src/sprav_pedestrian_feedback/lib/social_constants.h>

#include <maps/libs/http/include/http.h>
#include <maps/wikimap/mapspro/libs/http/http_utils.h>
#include <maps/libs/geolib/include/serialization.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/json/include/builder.h>
#include <yandex/maps/wiki/revision/common.h>

#include <chrono>
#include <sstream>

namespace maps::wiki::sprav_feedback {

namespace {

const std::string FEEDBACK_HANDLE = "/feedback/tasks";

std::string generateDescription(const AddressData& addressData)
{
    std::stringstream retVal;

    if (addressData.state == AddressData::State::Address) {
        retVal << "Пешеход нашёл адрес.\n\n";
    } else {
        retVal << "Пешеход НЕ нашёл адрес.\n\n";
    }

    if (addressData.address) {
        retVal << "Адрес:\n\"" << *addressData.address << "\"\n\n";
    }

    if (addressData.house) {
        retVal << "Дом: \"" << *addressData.house << "\"\n";
    }

    if (addressData.probAddress) {
        retVal << "Возможный адрес:\n\"" << *addressData.probAddress << "\"\n\n";
    }

    if (addressData.comment) {
        retVal << "Комментарий:\n\"" << *addressData.comment << "\"\n\n";
    }

    if (!addressData.photoUrls.empty()) {
        retVal << "Фото:\n";
        int count = 1;
        for (const auto& url : addressData.photoUrls) {
            retVal << std::to_string(count) << ") " << url << "\n";
            count++;
        }
    }

    return retVal.str();
}

const std::string getSource(const std::string& pedestrianType)
{
    if (pedestrianType == "walkers") {
        return "sprav-pedestrian-addresses";
    } else if (pedestrianType == "toloka") {
        return "sprav-pedestrian-addresses-toloka";
    } else if (pedestrianType == "yang") {
        return "sprav-pedestrian-addresses-toloka";
    } else {
        throw maps::RuntimeError() << "Unexpected pedestrianType: '" << pedestrianType << "'";
    }
}

void addOptionalString(
    json::ObjectBuilder& builder,
    const std::string& fieldName,
    const std::optional<std::string>& value)
{
    if (value) {
        builder[fieldName] = *value;
    }
}

std::string postRequestBody(const AddressData& addressData)
{
    json::Builder bodyBuilder;
    bodyBuilder << [&](json::ObjectBuilder bodyBuilder) {
        bodyBuilder[jf::TYPE] = TYPE_ADDRESS;
        bodyBuilder[jf::HIDDEN] = true;
        bodyBuilder[jf::SOURCE] = getSource(addressData.pedestrianType);
        bodyBuilder[jf::WORKFLOW] = WORKFLOW_FEEDBACK;
        bodyBuilder[jf::POSITION] = geolib3::geojson(addressData.inputGeo);
        bodyBuilder[jf::DESCR] = generateDescription(addressData);

        bodyBuilder[jf::USER_ATTRS] = [&](json::ObjectBuilder userAttrsBuilder) {
            addOptionalString(userAttrsBuilder, "spravTaskId", addressData.assignmentId);
            userAttrsBuilder["submitTimeUnixMs"] = addressData.submitTimeUnixMs;
            userAttrsBuilder["createdAtTimeUnixMs"] = addressData.createdAtTimeUnixMs;
            addOptionalString(userAttrsBuilder, "spravUserId", addressData.spravUserId);
        };
    };

    return bodyBuilder.str();
}

} // unnamed namespace

void publishAddress(const std::string& socialUrl, const AddressData& addressData)
{
    if (addressData.state == AddressData::State::NeverDone) {
        INFO() << "State = never-done. No action needed.";
        return;
    }
    if (addressData.state == AddressData::State::NoAddress
        && !addressData.address
        && !addressData.house
        && !addressData.probAddress) {
        INFO() << "No address found by pedestrian. No action needed.";
        return;
    }

    http::URL socialHttpUrl(socialUrl);
    socialHttpUrl.setPath(FEEDBACK_HANDLE);

    const size_t maxAttempts = 6;
    const std::chrono::seconds initialTimeout{1};
    const double timeoutBackoff = 2;

    auto retryPolicy = maps::common::RetryPolicy()
        .setTryNumber(maxAttempts)
        .setInitialCooldown(initialTimeout)
        .setCooldownBackoff(timeoutBackoff);

    http::Client httpClient;
    auto [response, status] = httpClient.post(
        socialHttpUrl,
        socialBackofficeHttpHeaders(),
        postRequestBody(addressData),
        retryPolicy);

    REQUIRE(status == http_status::OK,
        "Unable to post feedback; Response status: " << status);

    auto fbId = getFeedbackIdFromResponse(response);
    INFO() << "Feedback " << fbId << " published.";
}

} // namespace maps::wiki::sprav_feedback
