#include "common.h"
#include "data.h"
#include <maps/wikimap/mapspro/services/tasks_feedback/src/sprav_pedestrian_feedback/lib/parser.h>

#include <maps/libs/enum_io/include/enum_io.h>
#include <maps/libs/common/include/exception.h>

#include <boost/algorithm/string/trim.hpp>
#include <boost/lexical_cast.hpp>

namespace maps::wiki::sprav_feedback {

namespace {

const std::string ADDRESS = "address";
const std::string ASSIGNMENT = "assignmentId";
const std::string COMMENT = "comment";
const std::string COORDS = "coordinates";
const std::string PHOTOS = "photos";
const std::string WORKER = "workerId";
const std::string STATE = "state";
const std::string CITY = "city";
const std::string ENTRANCES = "entrances";

const enum_io::Representations<EntrancesData::State>
    PREDEFINED_STATES_STRINGS {
    {EntrancesData::State::Entry, "entry"},
    {EntrancesData::State::NoEntry, "no-entry"},
    {EntrancesData::State::NeverDone, "never-done"}
};

std::vector<Entrance> entrancesFromNode(const NYT::TNode& node)
{
    if (node.IsNull()) {
        return {};
    }

    std::vector<Entrance> retVal;
    for (const NYT::TNode& entranceNode : node.AsList()) {
        const auto& entranceMap = entranceNode.AsMap();
        auto pointGeo = reversedCoordsToPoint(entranceMap.at("coordinates").AsString());
        std::string entranceNameString = entranceMap.at("name").AsString();
        boost::trim(entranceNameString);
        retVal.push_back({pointGeo, entranceNameString});
    }

    return retVal;
}

} // unnamed namespace

DEFINE_ENUM_IO(EntrancesData::State, PREDEFINED_STATES_STRINGS);

template<>
std::optional<EntrancesData> buildDataFromTNode(const NYT::TNode& tableRow)
{
    EntrancesData retVal;

    retVal.address = optionalStringFromNode(tableRow[ADDRESS]);
    retVal.comment = optionalStringFromNode(tableRow[COMMENT]);
    retVal.pointGeo  = reversedCoordsToPoint(tableRow[COORDS].AsString());
    retVal.photoUrls = photoUrlsFromNode(tableRow[PHOTOS]);
    retVal.submitTimeUnixMs = submitTime(tableRow);
    retVal.createdAtTimeUnixMs = rowTime(tableRow);
    retVal.assignmentId = tableRow[ASSIGNMENT].AsString();
    retVal.spravUserId = tableRow[WORKER].AsString();
    retVal.state = boost::lexical_cast<EntrancesData::State >(tableRow[STATE].AsString());
    retVal.city = optionalStringFromNode(tableRow[CITY]);
    retVal.entrances = entrancesFromNode(tableRow[ENTRANCES]);

    return retVal;
}

} // namespace maps::wiki::sprav_feedback
