#include "process_entrance.h"
#include "revision_lookup.h"

#include <maps/libs/enum_io/include/enum_io.h>
#include <maps/libs/geolib/include/distance.h>
#include <yandex/maps/wiki/common/geom_utils.h>

#include <boost/lexical_cast.hpp>
#include <algorithm>

namespace maps::wiki::sprav_feedback {
namespace {

constexpr double SEARCH_RADIUS = 30.0; // meters
constexpr double NEARNESS_RADIUS = 7.0; // meters
constexpr double INTERSECT_ACCURACY = 0.5; // in meters

constexpr int MAX_NOT_SUSPICIOUS_ENTRANCE_NUM = 10;

constexpr enum_io::Representations<SkipReason>
    SKIP_REASON_STRINGS {
    {SkipReason::Duplicate, "Duplicate"}
};

bool isEntranceNameSuspicious(const std::string& name)
{
    if (name.empty()) {
        return false;
    }
    try {
        auto num = boost::lexical_cast<int>(name);
        return !(0 < num && num <= MAX_NOT_SUSPICIOUS_ENTRANCE_NUM);
    } catch (const boost::bad_lexical_cast& ex) {
        return true;
    }
}

double distanceInMeters(const geolib3::Point2& p1, const geolib3::Point2& p2)
{
    return geolib3::toMeters(geolib3::distance(p1, p2), p1);
}

} // unnamed namespace

DEFINE_ENUM_IO(SkipReason, SKIP_REASON_STRINGS);

PublishData processEntrance(
    pqxx::transaction_base& coreTxn,
    pqxx::transaction_base& viewTrunkTxn,
    const Entrance& entrance)
{
    if (isEntranceNameSuspicious(entrance.name)) {
        return FeedbackPublishData{entrance, FeedbackReason::SuspiciousName};
    }

    auto edgePoint = getEdgePoint(viewTrunkTxn, entrance.pointGeo, SEARCH_RADIUS);

    if (!edgePoint) {
        return FeedbackPublishData{entrance, FeedbackReason::NoBuildingFound};
    }

    revision::RevisionsGateway gtw(coreTxn);
    auto existingEntrances = findEntrances(gtw, edgePoint->bldId, INTERSECT_ACCURACY);
    std::sort(existingEntrances.begin(), existingEntrances.end(),
        [&](const RevisionEntrance& e1, const RevisionEntrance& e2){
            double dist1 = geolib3::distance(e1.point, edgePoint->point);
            double dist2 = geolib3::distance(e2.point, edgePoint->point);
            return dist1 < dist2;
    });

    for (const auto& existingEntrance : existingEntrances) {
        auto distance = distanceInMeters(existingEntrance.point, edgePoint->point);
        if (distance <= NEARNESS_RADIUS) {
            if (entrance.name.empty()
                || (existingEntrance.name
                && existingEntrance.name.value() == entrance.name)
                ) {
                return SkipReason::Duplicate;
            }
            if (!existingEntrance.name) {
                return CommitPublishData{existingEntrance.poiEntranceId, entrance.name};
            }

            return FeedbackPublishData{entrance, FeedbackReason::NameConflict};
        }

        if (existingEntrance.name
            && existingEntrance.name.value() == entrance.name
            && !entrance.name.empty()) {
            return FeedbackPublishData{entrance, FeedbackReason::DuplicateName};
        }
    }

    // commit entrance
    return CommitPublishData{edgePoint->point, entrance.name};
}

} // namespace maps::wiki::sprav_feedback
