#pragma once

#include "common.h"
#include "print_info.h"

#include <maps/wikimap/mapspro/services/tasks_feedback/src/common/time.h>
#include <maps/wikimap/mapspro/services/tasks_feedback/src/common/last_processed_time.h>

#include <maps/libs/common/include/exception.h>
#include <yandex/maps/wiki/common/yt.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>

#include <pqxx/pqxx>
#include <algorithm>
#include <optional>
#include <vector>
#include <cstdint>

namespace mwc = maps::wiki::common;
namespace mwt = maps::wiki::tasks_feedback;

namespace maps::wiki::sprav_feedback {

const TString YT_PROXY_NAME = "hahn";

    inline bool inTimeInterval(mwt::TimeIntervalMs timeIntervalMs, uint64_t fbTime)
{
    return timeIntervalMs.fromTime <= fbTime && fbTime < timeIntervalMs.tillTime;
}

template <typename RawDataStruct>
bool lessByTime(const RawDataStruct& a, const RawDataStruct& b)
{
    return getRowTimeMs(a) < getRowTimeMs(b);
}

template <typename RawDataStruct>
std::vector<RawDataStruct> loadFromYT(mwt::TimeIntervalMs timeIntervalMs)
{
    REQUIRE(timeIntervalMs.fromTime <= timeIntervalMs.tillTime, "Start time is greater than end time");

    auto client = mwc::yt::createYtClient(YT_PROXY_NAME);
    auto clientTxn = client->StartTransaction();

    std::vector<RawDataStruct> feedback;

    auto reader = clientTxn->CreateTableReader<NYT::TNode>(
        TString(ytTableName<RawDataStruct>()));
    for (; reader->IsValid(); reader->Next()) {
        const NYT::TNode& row = reader->GetRow();
        auto fbTime = rowTime(row);
        if (inTimeInterval(timeIntervalMs, fbTime)) {
            auto rawDataOptional = buildDataFromTNode<RawDataStruct>(row);
            if (rawDataOptional) {
                feedback.push_back(rawDataOptional.value());
            }
        }
    }

    clientTxn->Commit();
    return feedback;
}

template <typename RawDataStruct>
void loadPeriodical(
    maps::pgpool3::Pool& corePool,
    const ProcessParams<RawDataStruct>& config)
{
    auto loadLastProcessedTimeMs = [&] {
        auto coreTxnHolder = corePool.masterReadOnlyTransaction();

        // '+ 1' - because we will process timeInterval [from, till)
        // and we don't want to process 'from' twice
        return mwt::lastProcessedTimeMs(coreTxnHolder.get(),
            lastTimeTableName<RawDataStruct>())
            + 1;
    };

    mwt::TimeIntervalMs timeIntervalMs{
        loadLastProcessedTimeMs(),
        mwt::currentTimeUnixMs()
    };

    printLoadFromYtInfo(timeIntervalMs);
    auto fbs = loadFromYT<RawDataStruct>(timeIntervalMs);
    printLoadedSizeInfo(fbs.size());
    std::sort(fbs.begin(), fbs.end(), &lessByTime<RawDataStruct>);

    for (const auto& fb : fbs) {
        uint64_t itemTime = getRowTimeMs(fb);
        printProcessingInfo(itemTime);

        auto coreTimeTxnHolder = corePool.masterWriteableTransaction();
        mwt::setLastProcessedTimeMs(
            itemTime,
            coreTimeTxnHolder.get(),
            lastTimeTableName<RawDataStruct>());

        processFunc(coreTimeTxnHolder.get(), fb, config);

        coreTimeTxnHolder.get().commit();

        printSettingLastProccessedInfo(itemTime);
    }
    printSuccessPublishInfo();
}

template <typename RawDataStruct>
void loadByHand(
    pqxx::transaction_base& coreTxn,
    mwt::TimeIntervalMs timeIntervalMs,
    const ProcessParams<RawDataStruct>& config)
{
    printLoadFromYtInfo(timeIntervalMs);
    auto fbs = loadFromYT<RawDataStruct>(timeIntervalMs);
    printLoadedSizeInfo(fbs.size());
    std::sort(fbs.begin(), fbs.end(), &lessByTime<RawDataStruct>);

    if (!fbs.empty()) {
        for (const auto& fb : fbs) {
            printProcessingInfo(getRowTimeMs(fb));
            processFunc(coreTxn, fb, config);
        }
        printSuccessPublishInfo();
    }
}

} // namespace maps::wiki::sprav_feedback
