#include "fbapi_issue_parser.h"
#include "feedback_subsource.h"
#include "magic_strings.h"
#include "position_diff.h"
#include "route_context.h"
#include "site_and_spots_as_object_diff.h"
#include "decode_coordinates.h"

#include <maps/libs/geolib/include/conversion.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/enum_io/include/enum_io.h>
#include <maps/libs/json/include/builder.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/social/feedback/attribute_names.h>
#include <yandex/maps/wiki/social/feedback/description_keys.h>
#include <yandex/maps/wiki/social/feedback/description_producers.h>

#include <sstream>
#include <iomanip>

using boost::lexical_cast;

namespace maps::wiki::sync_fbapi_feedback {

namespace sf = social::feedback;
namespace sfa = social::feedback::attrs;

namespace {

#define PARSING_REQUIRE(cond, msg) \
    if (Y_LIKELY(cond)) {} else \
        throw IssueParseError() << msg;


// more convenient interface for fbapi::Toponym
struct ToponymAddressAdapter
{
    ToponymAddressAdapter(const fbapi::Toponym& toponym)
        : addressString(toponym.addressString())
    {
        for (const auto& component: toponym.addressComponents()) {
            switch (component.kind()) {
                case fbapi::ToponymAddressComponentKind::House:
                    house = component.name();
                    break;
                case fbapi::ToponymAddressComponentKind::Street:
                    street = component.name();
                    break;
            }
        }
    }

    std::optional<std::string> house;
    std::optional<std::string> street;
    std::optional<std::string> addressString;
};

enum class Dialog
{
    Question,
    Answer
};

constexpr enum_io::Representations<Dialog> DIALOG_STRINGS {
    {Dialog::Question, "question"},
    {Dialog::Answer, "answer"}
};

DEFINE_ENUM_IO(Dialog, DIALOG_STRINGS);


template <typename CONTEXT>
fbapi::Toponym extractToponym(
    const std::optional<CONTEXT>& context,
    Dialog dialog)
{
    PARSING_REQUIRE(context, dialog << " context expected");
    return context->template as<fbapi::Toponym>();
}

ToponymAddressAdapter extractToponymWithMandatoryAddress(
    const std::optional<fbapi::AnswerContext>& answerContext,
    Dialog dialog)
{
    PARSING_REQUIRE(answerContext, dialog << " context expected");

    auto toponym = ToponymAddressAdapter(answerContext->as<fbapi::Toponym>());
    PARSING_REQUIRE(toponym.house || toponym.addressString,
        "toponym house expected");

    return toponym;
}

fbapi::Toponym extractToponymWithMandatoryPosition(
    const fbapi::OriginalTask& originalTask,
    Dialog dialog)
{
    auto toponym = [&]() {
        switch (dialog) {
            case Dialog::Question:
                return extractToponym(originalTask.questionContext(), dialog);
            case Dialog::Answer:
                return extractToponym(originalTask.answerContext(), dialog);
        }
    }();

    PARSING_REQUIRE(
        toponym.center(),
        dialog << " entrance center expected"
    );

    return toponym;
}

template <typename Ctx>
fbapi::Entrance extractEntranceWithMandatoryPosition(
    const std::optional<Ctx>& context,
    Dialog dialog)
{
    PARSING_REQUIRE(context, dialog << " context expected");

    auto entrance = context->template as<fbapi::Entrance>();
    PARSING_REQUIRE(entrance.center(),
        dialog << " entrance center expected");

    return entrance;
};

fbapi::RouteQuestionContext extractRouteQuestion(
    const std::optional<fbapi::QuestionContext>& question)
{
    PARSING_REQUIRE(question, "question context expected");
    return question->as<fbapi::RouteQuestionContext>();
};

fbapi::RouteAnswerContext extractRouteAnswer(
    const std::optional<fbapi::AnswerContext>& answer)
{
    PARSING_REQUIRE(answer, "answer context expected");
    return answer->as<fbapi::RouteAnswerContext>();
};

template<typename T>
json::Value objectToJsonValue(const T& object)
{
    json::Builder jsonBuilder;
    jsonBuilder << object;
    return json::Value::fromString(jsonBuilder.str());
}

std::optional<std::string>
supportOrUserMessage(const fbapi::OriginalTask& originalTask)
{
    // uncomment when MAPSHTTPAPI-1328 will be done
//    if (auto supportMessage = originalTask_.supportMessage()) {
//        return supportMessage;
//    }

    return originalTask.message();
}

social::feedback::Type getRouteType(
    const fbapi::TravelMode travelMode,
    const fbapi::RouteAnswerContext& answer)
{
    const auto errors = answer.errors();
    if (!errors.empty()) {
        const auto& routeError = errors[0];
        switch (routeError.type()) {
            case fbapi::RouteErrorType::Barrier:
            case fbapi::RouteErrorType::Obstruction:
                return sf::Type::Barrier;

            case fbapi::RouteErrorType::PoorCondition:
                return sf::Type::RoadSurface;

            case fbapi::RouteErrorType::NoRoad:
                return sf::Type::NoRoad;

            case fbapi::RouteErrorType::ProhibitingSign:
                return sf::Type::TrafficProhibitedSign;

            case fbapi::RouteErrorType::UTurnProhibited:
            case fbapi::RouteErrorType::RightTurnProhibited:
            case fbapi::RouteErrorType::LeftTurnProhibited:
                return sf::Type::ProhibitedTurnSign;

            case fbapi::RouteErrorType::Other:
                ; // no code
        }
    }

    switch (travelMode) {
        case fbapi::TravelMode::Bicycle:
            return sf::Type::BicycleRoute;

        case fbapi::TravelMode::Scooter:
            return sf::Type::ScooterRoute;

        case fbapi::TravelMode::Pedestrian:
            return sf::Type::PedestrianRoute;

        case fbapi::TravelMode::Auto:
            return sf::Type::CarRoute;

        case fbapi::TravelMode::Taxi:
            return sf::Type::TaxiRoute;

        case fbapi::TravelMode::Truck:
            return sf::Type::TruckRoute;

        case fbapi::TravelMode::Masstransit:
            return sf::Type::MtRoute;
    }
}

std::string getRouteDescriptionKey(
    const fbapi::TravelMode travelMode,
    const fbapi::RouteAnswerContext& answer)
{
    const auto errors = answer.errors();
    if (!errors.empty()) {
        const auto& routeError = errors[0];
        switch (routeError.type()) {
            case fbapi::RouteErrorType::Barrier:
            case fbapi::RouteErrorType::ProhibitingSign:
            case fbapi::RouteErrorType::Obstruction:
                return travelMode == fbapi::TravelMode::Pedestrian
                    ? sf::tanker::fb_desc::ROUTE_NO_PASSAGEWAY_KEY
                    : sf::tanker::fb_desc::ROUTE_NO_DRIVEWAY_KEY;

            case fbapi::RouteErrorType::PoorCondition:
                return sf::tanker::fb_desc::ROUTE_POOR_CONDITION_KEY;

            case fbapi::RouteErrorType::NoRoad:
                return sf::tanker::fb_desc::ROUTE_NOROAD_KEY;

            case fbapi::RouteErrorType::UTurnProhibited:
            case fbapi::RouteErrorType::RightTurnProhibited:
            case fbapi::RouteErrorType::LeftTurnProhibited:
                return sf::tanker::fb_desc::ROUTE_TURN_PROHIBITED_KEY;

            case fbapi::RouteErrorType::Other:
                return sf::tanker::fb_desc::ROUTE_ERROR_KEY;
        }
    }
    const auto seg = answer.segments();
    if (!answer.segments().empty()) {
        return sf::tanker::fb_desc::ROUTE_CORRECTION_KEY;
    }
    return sf::tanker::fb_desc::ROUTE_OTHER_KEY;
}

json::Value toJsonValue(const std::vector<std::string>& items)
{
    json::Builder builder;
    builder << [&](json::ArrayBuilder builder) {
        for (const auto& item : items) {
            builder << item;
        }
    };
    return json::Value::fromString(builder.str());
}

} // namespace

IUriResolver::~IUriResolver()
{}

void FbapiIssueParser::initCommonFields(
    IUriResolver& uriResolver,
    const fbapi::Task& task)
{
    position_ = originalTask_.formPoint();
    std::optional<std::string> objectIdText = originalTask_.objectId();
    if (!objectIdText && originalTask_.objectUri()) {
        objectIdText = uriResolver.resolveObjectUri(originalTask_.objectUri().value());
    }
    if (objectIdText) try {
        objectId_ = lexical_cast<social::TId>(*objectIdText);
    } catch (const boost::bad_lexical_cast& ex) {
        throw IssueParseError() << "Incorrect object_id "
            << "('" << *objectIdText << "')"
            << "at fbapi::Task '" << task.id() << "'";
    }

    const auto attachedPhotos = task.attachedPhotos();
    if (!attachedPhotos.empty()) {
        hidden_ = true;
        attrs_.add(sf::AttrType::UserDataPhotoUrls, toJsonValue(attachedPhotos));
    }

    attrs_.addCustom(
        sfa::SUBSOURCE,
        generateSubsource(
            originalTask_.questionId(),
            originalTask_.answerId()
        )
    );

    auto userEmail = originalTask_.userEmail();
    if (userEmail) {
        attrs_.addCustom(sfa::USER_EMAIL, *userEmail);
    }

    auto objectUri = originalTask_.objectUri();
    if (objectUri) {
        attrs_.addCustom(sfa::OBJECT_URI, *objectUri);
    }

    auto userMessage = originalTask_.message();
    if (userMessage && *userMessage != "") {
        hidden_ = true;
        attrs_.addCustom(sfa::USER_DATA_COMMENT, *userMessage);
    }

    auto formContext = originalTask_.formContext();
    if (formContext) {
        auto searchText = formContext->searchText();
        if (searchText) {
            hidden_ = true;
            attrs_.addCustom(sfa::USER_DATA_SEARCH_REQUEST, *searchText);
        }
        if (formContext->fromPush()) {
            attrs_.setFlag(sfa::FROM_PUSH);
        }
    }

    const auto& metadata = originalTask_.metadata();
    if (metadata.ip()) {
        attrs_.addCustom(sfa::IP, *metadata.ip());
    }
    if (metadata.fingerprint()) {
        attrs_.addCustom(sfa::FINGERPRINT, *metadata.fingerprint());
    }
    if (metadata.uid()) {
        attrs_.addCustom(sfa::UID, std::to_string(*metadata.uid()));
    }
    if (metadata.yandexuid()) {
        attrs_.addCustom(sfa::YANDEXUID, *metadata.yandexuid());
    }
    if (metadata.userAgent()) {
        attrs_.addCustom(sfa::USER_AGENT, *metadata.userAgent());
    }


    if (
        originalTask_.questionId() == question_id::OTHER &&
        originalTask_.answerId() != answer_id::COMMENT
    ) {
        source_ = "fbapi-samsara";
    } else if (userEmail || metadata.uid() || metadata.deviceId()) {
        source_ = "fbapi";
    } else {
        source_ = "fbapi-nologin";
        hidden_ = true;
    }
}

void FbapiIssueParser::initWrongAddressDontKnow()
{
    type_ = sf::Type::Address;
    description_ = sf::DescriptionI18n(sf::tanker::fb_desc::ADDRESS_UNKNOWN_KEY);
}

void FbapiIssueParser::initAddressRelatedProperties(
    sf::AddressDescriptionType addressDescriptionType)
{
    // Parsing original task
    //
    auto answerToponym = extractToponymWithMandatoryAddress(
        originalTask_.answerContext(), Dialog::Answer);

    // Filling feedback task
    //
    type_ = sf::Type::Address;

    if (answerToponym.street) {
        attrs_.addCustom(sfa::STREET, *answerToponym.street);
    }
    if (answerToponym.house) {
        attrs_.addCustom(sfa::HOUSE, *answerToponym.house);
    }

    description_ = sf::AddressDescr{
        addressDescriptionType,
        answerToponym.street,
        answerToponym.house,
        answerToponym.addressString
    }.toDescription();
}

void FbapiIssueParser::initAddObjectToponym()
{
    initAddressRelatedProperties(sf::AddressDescriptionType::NewAddress);

    auto answerContext = originalTask_.answerContext();
    if (answerContext) {
        auto toponym = answerContext->as<fbapi::Toponym>();
        auto center = toponym.center();
        if (center) {
            position_ = *center;
        }
    }
}

void FbapiIssueParser::initAddObjectStop()
{
    // Parsing original task
    //
    auto answerToponym = extractToponymWithMandatoryPosition(
        originalTask_, Dialog::Answer
    );

    // Filling feedback task
    //
    type_ = sf::Type::PublicTransportStop;
    position_ = *answerToponym.center();
    description_ = sf::NoPublicTransportStopDescr{
        answerToponym.name()
    }.toDescription();
}

void FbapiIssueParser::initAddObjectBarrier()
{
    // Parsing original task
    //
    auto answerToponym = extractToponymWithMandatoryPosition(
        originalTask_, Dialog::Answer
    );

    // Filling feedback task
    //
    type_ = sf::Type::Barrier;
    position_ = *answerToponym.center();
    description_ = sf::DescriptionI18n(sf::tanker::fb_desc::ABSENT_BARRIER_KEY);
}

void FbapiIssueParser::initAddObjectCrosswalk()
{
    // Parsing original task
    //
    auto answerToponym = extractToponymWithMandatoryPosition(
        originalTask_, Dialog::Answer
    );

    // Filling feedback task
    //
    type_ = sf::Type::PedestrianRoute;
    position_ = *answerToponym.center();
    description_ = sf::DescriptionI18n(sf::tanker::fb_desc::ABSENT_CROSSWALK_KEY);
}

void FbapiIssueParser::initAddObjectParking()
{
    // Parsing original task
    //
    auto answerContext = originalTask_.answerContext();

    std::string descriptionKey = sf::tanker::fb_desc::ABSENT_PARKING_KEY;
    if (const auto parking = answerContext->as<std::optional<fbapi::Parking>>()) {
        PARSING_REQUIRE(parking->center(), "parking center expected");
        position_ = parking->center().value();

        const auto entrances = parking->entrances();
        if (!entrances.empty()) {
            descriptionKey = sf::tanker::fb_desc::ABSENT_PARKING_ENTRANCES_KEY;
            attrs_.add(
                sf::AttrType::ObjectDiff,
                objectToJsonValue(
                    SiteAndSpotsAsObjectDiff().setMercatorPointsAfter(entrances)
                )
            );
        }
    } else {
        auto answerToponym = extractToponymWithMandatoryPosition(
            originalTask_, Dialog::Answer
        );
        position_ = *answerToponym.center();
    }

    // Filling feedback task
    //
    type_ = sf::Type::Parking;
    description_ = sf::DescriptionI18n(descriptionKey);
}

void FbapiIssueParser::initAddObjectOther()
{
    // Parsing original task
    //
    auto answerToponym = extractToponymWithMandatoryPosition(
        originalTask_, Dialog::Answer
    );

    // Filling feedback task
    //
    type_ = sf::Type::Other;
    position_ = *answerToponym.center();
    auto message = supportOrUserMessage(originalTask_);
    description_ = sf::AbsentObjectDescr{
        answerToponym.name()
    }.toDescription();
}

void FbapiIssueParser::initAddObjectRoad()
{
    // Parsing original task
    //
    std::optional<std::string> roadName;

    auto answerContext = originalTask_.answerContext();
    if (auto road = answerContext->as<std::optional<fbapi::Road>>()) {
        roadName = road->name();
        position_ = road->center();

        auto points = road->points();
        if (!points.empty()) {
            attrs_.add(
                sf::AttrType::ObjectDiff,
                objectToJsonValue(
                    SiteAndSpotsAsObjectDiff().setMercatorSiteAfter(geolib3::Polyline2{points})
                )
            );
        }
    } else {
        auto toponym = extractToponymWithMandatoryPosition(
            originalTask_, Dialog::Answer
        );

        position_ = *toponym.center();

        auto toponymAdapter = ToponymAddressAdapter(toponym);
        if (toponymAdapter.street) {
            roadName = toponymAdapter.street.value();
        } else if (toponymAdapter.addressString) {
            roadName = toponymAdapter.addressString.value();
        }
    }

    // Filling feedback task
    //
    type_ = sf::Type::Road;
    description_ = sf::AbsentRoadDescr{
        roadName
    }.toDescription();
}

void FbapiIssueParser::initRemoveObjectToponym()
{
    // Parsing original task
    //
    const auto questionContext = originalTask_.questionContext();

    if (const auto stop = questionContext->as<std::optional<fbapi::Stop>>()) {
        type_ = sf::Type::PublicTransportStop;
        position_ = stop->center();
        description_ = sf::DescriptionI18n(sf::tanker::fb_desc::REMOVE_STOP_KEY);
    } else {
        const auto questionToponym = extractToponymWithMandatoryPosition(
            originalTask_, Dialog::Question
        );
        type_ = sf::Type::Other;
        position_ = questionToponym.center().value();
        description_ = sf::RemoveObjectDescr{
            questionToponym.name()
        }.toDescription();
    }
}

void FbapiIssueParser::initWrongNameReportName()
{
    // Parsing original task
    //
    const auto questionContext = originalTask_.questionContext();

    if (const auto questionStop = questionContext->as<std::optional<fbapi::Stop>>()) {
        const auto answerStop =
            originalTask_.answerContext()->as<std::optional<fbapi::Stop>>();
        PARSING_REQUIRE(answerStop, "Stop in answerContext expected");

        type_ = sf::Type::PublicTransportStop;
        position_ = questionStop->center();
        description_ = sf::WrongNameDescr{
            questionStop->name(),
            answerStop->name()
        }.toDescription();
    } else {
        const auto questionToponym =
            extractToponymWithMandatoryPosition(originalTask_, Dialog::Question);
        const auto answerToponym =
            extractToponymWithMandatoryPosition(originalTask_, Dialog::Answer);

        type_ = sf::Type::Other;
        position_ = questionToponym.center().value();
        description_ = sf::WrongNameDescr{
            questionToponym.name(),
            answerToponym.name()
        }.toDescription();
    }
}

void FbapiIssueParser::initWrongAddressReportAddress()
{
    initAddressRelatedProperties(sf::AddressDescriptionType::AddressCorrection);
}

void FbapiIssueParser::initReportLocation()
{
    // Parsing original task
    //
    auto questionToponym =
        extractToponymWithMandatoryPosition(originalTask_, Dialog::Question);
    auto answerToponym =
        extractToponymWithMandatoryPosition(originalTask_, Dialog::Answer);

    // Filling feedback task
    //
    position_ = *questionToponym.center();

    attrs_.add(
        sf::AttrType::ObjectDiff,
        objectToJsonValue(PositionDiff(
            geolib3::convertMercatorToGeodetic(*questionToponym.center()),
            geolib3::convertMercatorToGeodetic(*answerToponym.center())
        ))
    );
}

void FbapiIssueParser::initWrongAddressReportLocation()
{
    type_ = sf::Type::Address;
    description_ = sf::DescriptionI18n(sf::tanker::fb_desc::ADDR_WRONG_POS_KEY);
    initReportLocation();
}

void FbapiIssueParser::initWrongBarrierReportLocation(const std::string& descriptionKey)
{
    type_ = sf::Type::Barrier;
    description_ = sf::DescriptionI18n(descriptionKey);
    initReportLocation();
}

void FbapiIssueParser::initWrongSubwayNotFound()
{
    type_ = sf::Type::Subway;
    description_ = sf::DescriptionI18n(sf::tanker::fb_desc::SUBWAY_NOT_FOUND_KEY);
}

void FbapiIssueParser::initWrongBarrierNotFound(const std::string& descriptionKey)
{
    const auto questionToponym = extractToponymWithMandatoryPosition(
        originalTask_, Dialog::Question
    );
    type_ = sf::Type::Barrier;
    position_ = questionToponym.center().value();
    description_ = sf::DescriptionI18n(descriptionKey);
}

void FbapiIssueParser::initWrongPositionDemolished()
{
    type_ = sf::Type::Building;
    description_ = sf::DescriptionI18n(sf::tanker::fb_desc::BUILDING_DEMOLISHED_KEY);
}

void FbapiIssueParser::initWrongPositionNeverBeenHere()
{
    type_ = sf::Type::Building;
    description_ = sf::DescriptionI18n(sf::tanker::fb_desc::NO_BUILDING_HERE_KEY);
}

void FbapiIssueParser::initWrongPositionNotFound()
{
    type_ = sf::Type::Building;
    description_ = sf::DescriptionI18n(sf::tanker::fb_desc::NO_BUILDING_HERE_KEY);
}

void FbapiIssueParser::initWrongPositionOther()
{
    hidden_ = true;
    type_ = sf::Type::Building;

    PARSING_REQUIRE(
        attrs_.existCustom(sfa::USER_DATA_COMMENT) || attrs_.exist(sf::AttrType::UserDataPhotoUrls),
        "Message from user or attached photos expected"
    );
    description_ = sf::DescriptionI18n(sf::tanker::fb_desc::NO_BUILDING_HERE_KEY);
}

std::optional<geolib3::Point2>
getStartPoint(const fbapi::RouteAnswerSegments& routeAnswerSergents)
{
    if (routeAnswerSergents.empty()) {
        return std::nullopt;
    }
    const auto& points = routeAnswerSergents.front().points();
    if (points.empty()) {
        return std::nullopt;
    }
    return points[0];
}

void FbapiIssueParser::initWrongRouteReportRoute(
    IUriResolver& uriResolver)
{
    auto question = extractRouteQuestion(originalTask_.questionContext());
    auto answer = extractRouteAnswer(originalTask_.answerContext());

    fbapi::TravelMode travelMode = question.travelMode();
    geolib3::PointsVector routePointsGeodetic;

    if (question.uri()) {
        switch (travelMode) {
            case fbapi::TravelMode::Auto:
            case fbapi::TravelMode::Taxi:
            case fbapi::TravelMode::Truck: {
                auto route = uriResolver.resolveDrivingRouteUri(*question.uri());
                REQUIRE(route, "unresolved drivingRoute-uri: " << *question.uri());
                routePointsGeodetic = route->routePointsGeodetic().points();
                break;
            }
            case fbapi::TravelMode::Masstransit: {
                auto route = uriResolver.resolveMtRouteUri(*question.uri());
                REQUIRE(route, "unresolved mtRoute-uri: " << *question.uri());
                routePointsGeodetic = route->routePointsGeodetic().points();
                break;
            }
            case fbapi::TravelMode::Pedestrian: {
                auto route = uriResolver.resolvePedestrianRouteUri(*question.uri());
                REQUIRE(route, "unresolved pedestrianRoute-uri: " << *question.uri());
                routePointsGeodetic = route->routePointsGeodetic().points();
                break;
            }
            case fbapi::TravelMode::Bicycle: {
                auto route = uriResolver.resolveBicycleRouteUri(*question.uri());
                REQUIRE(route, "unresolved bicycleRoute-uri: " << *question.uri());
                routePointsGeodetic = route->routePointsGeodetic().points();
                break;
            }
            case fbapi::TravelMode::Scooter: {
                auto route = uriResolver.resolveScooterRouteUri(*question.uri());
                REQUIRE(route, "unresolved scooterRoute-uri: " << *question.uri());
                routePointsGeodetic = route->routePointsGeodetic().points();
                break;
            }
        }

    } else {
        REQUIRE(question.encodedPoints(), "encodedPoints expected in fbapi issue");
        routePointsGeodetic = decodeCoordinates(*question.encodedPoints());
    }

    RouteContext routeContext(
        question.travelMode(),
        question.trafficJams(),
        question.wayPoints(),
        routePointsGeodetic,
        question.segments());
    routeContext.setViaPointsMercator(question.viaPoints())
        .setUserErrors(answer.errors())
        .setUserRoute(answer.segments());
    if (question.uri()) {
        routeContext.setUri(*question.uri());
    }
    if (question.departureAt()) {
        routeContext.setDepartureAt(*question.departureAt());
    }
    if (question.vehicleRestrictions()) {
        routeContext.setVehicleRestrictions(*question.vehicleRestrictions());
    }
    attrs_.add(sf::AttrType::SourceContext, objectToJsonValue(routeContext));

    type_ = getRouteType(travelMode, answer);
    description_ = sf::DescriptionI18n(getRouteDescriptionKey(travelMode, answer));
}

void FbapiIssueParser::initAddObjectEntrance()
{
    // objectId in this case is associated with 'Address' object
    //
    auto answerContext = originalTask_.answerContext();
    PARSING_REQUIRE(answerContext, "Answer context expected");
    auto entrance = answerContext->as<fbapi::Entrance>();

    if (entrance.center()) {
        position_ = *entrance.center();
    }
    type_ = sf::Type::Entrance;

    if (entrance.name()) {
        attrs_.addCustom(sfa::ENTRANCE_NAME, *entrance.name());
    }

    description_ = sf::AbsentEntranceDescr{
        entrance.name()
    }.toDescription();
}

void FbapiIssueParser::initAddObjectFence()
{
    // objectId in this case is associated with 'Address' object
    //
    auto answerContext = originalTask_.answerContext();
    PARSING_REQUIRE(answerContext, "Answer context expected");
    auto fence = answerContext->as<fbapi::Fence>();

    position_ = boundingBox(fence.points()).center();
    type_ = sf::Type::Fence;

    geolib3::PointsVector gatesPoints;
    for (const auto& gate : fence.gates()) {
        gatesPoints.emplace_back(gate.center());
    }
    attrs_.add(
        sf::AttrType::ObjectDiff,
        objectToJsonValue(
            SiteAndSpotsAsObjectDiff()
                .setMercatorSiteBefore(geolib3::Polyline2(fence.points()))
                .setMercatorPointsAfter(gatesPoints)
        )
    );

    description_ = sf::DescriptionI18n(sf::tanker::fb_desc::ABSENT_FENCE_KEY);
}

void FbapiIssueParser::initAddObjectGate()
{
    // objectId in this case is associated with 'Address' object
    //
    auto answerToponym = extractToponymWithMandatoryPosition(
        originalTask_, Dialog::Answer
    );

    position_ = *answerToponym.center();
    type_ = sf::Type::Barrier;

    description_ = sf::DescriptionI18n(sf::tanker::fb_desc::ABSENT_GATE_KEY);
}

void FbapiIssueParser::initWrongEntranceNotFound()
{
    // Parsing original task
    //
    auto entrance = extractEntranceWithMandatoryPosition(
        originalTask_.questionContext(), Dialog::Question);

    // Filling feedback task
    //
    position_ = *entrance.center();
    type_ = sf::Type::Entrance;

    if (entrance.name()) {
        attrs_.addCustom(sfa::ENTRANCE_NAME, *entrance.name());
    }

    description_ = sf::DeleteEntranceDescr{
        entrance.name()
    }.toDescription();
}

void FbapiIssueParser::initWrongLinesReportLines()
{
    // Parsing original task
    //
    const auto stop =
        originalTask_.questionContext()->as<std::optional<fbapi::Stop>>();
    PARSING_REQUIRE(stop, "Stop in questionContext expected");

    type_ = sf::Type::PublicTransportStop;
    position_ = stop->center();
    description_ = sf::DescriptionI18n(sf::tanker::fb_desc::WRONG_LINES_SET);
}

void FbapiIssueParser::initWrongLocationReportLocation()
{
    // Parsing original task
    //
    const auto questionContext = originalTask_.questionContext();

    if (const auto questionStop = questionContext->as<std::optional<fbapi::Stop>>()) {
        const auto answerStop =
            originalTask_.answerContext()->as<std::optional<fbapi::Stop>>();
        PARSING_REQUIRE(answerStop, "Stop in answerContext expected");

        type_ = sf::Type::PublicTransportStop;
        position_ = questionStop->center();
        description_ = sf::DescriptionI18n(sf::tanker::fb_desc::LOCATION_WRONG_POS_KEY);
        attrs_.add(
            sf::AttrType::ObjectDiff,
            objectToJsonValue(PositionDiff(
                geolib3::convertMercatorToGeodetic(questionStop->center()),
                geolib3::convertMercatorToGeodetic(answerStop->center())
            ))
        );
    } else {
        const auto questionLocation = extractToponymWithMandatoryPosition(
            originalTask_, Dialog::Question);
        const auto answerLocation = extractToponymWithMandatoryPosition(
            originalTask_, Dialog::Answer);

        type_ = sf::Type::Poi;
        position_ = questionLocation.center().value();
        description_ = sf::DescriptionI18n(sf::tanker::fb_desc::LOCATION_WRONG_POS_KEY);
        attrs_.add(
            sf::AttrType::ObjectDiff,
            objectToJsonValue(PositionDiff(
                geolib3::convertMercatorToGeodetic(*questionLocation.center()),
                geolib3::convertMercatorToGeodetic(*answerLocation.center())
            ))
        );
    }
}

void FbapiIssueParser::initWrongEntranceReportEntrance()
{
    // Parsing original task
    //
    auto questionEntrance = extractEntranceWithMandatoryPosition(
        originalTask_.questionContext(), Dialog::Question);

    auto answerEntrance = extractEntranceWithMandatoryPosition(
        originalTask_.answerContext(), Dialog::Answer);

    // Filling feedback task
    //
    position_ = *questionEntrance.center();
    type_ = sf::Type::Entrance;

    description_ = sf::EntranceCorrectionDescr{
        questionEntrance.name(),
        answerEntrance.name()
    }.toDescription();

    if (questionEntrance.name()) {
        attrs_.addCustom(sfa::ENTRANCE_NAME, *questionEntrance.name());
    }
    attrs_.add(
        sf::AttrType::ObjectDiff,
        objectToJsonValue(PositionDiff(
            geolib3::convertMercatorToGeodetic(*questionEntrance.center()),
            geolib3::convertMercatorToGeodetic(*answerEntrance.center())
        ))
    );
}

void FbapiIssueParser::initWrongSubwayReportSubway()
{
    auto dialog = Dialog::Answer;
    auto answerToponym = extractToponym(originalTask_.answerContext(), dialog);

    type_ = sf::Type::Subway;

    if (answerToponym.center()) {
        attrs_.add(
            sf::AttrType::ObjectDiff,
            objectToJsonValue(PositionDiff(
                geolib3::convertMercatorToGeodetic(position_),
                geolib3::convertMercatorToGeodetic(*answerToponym.center())
            ))
        );

        if (answerToponym.name()) {
            description_ = sf::SubwayCorrectionNameAndPositionDescr{
                answerToponym.name().value()
            }.toDescription();
        } else {
            description_ = sf::DescriptionI18n(sf::tanker::fb_desc::SUBWAY_WRONG_POSITION_KEY);
        }
    } else if (answerToponym.name()) {
        description_ = sf::SubwayCorrectionNameDescr{
            answerToponym.name().value()
        }.toDescription();
    } else {
        PARSING_REQUIRE(false, dialog << " entrance center or name expected");
    }
}

void FbapiIssueParser::initWrongEntranceReportLocation()
{
    // Parsing original task
    //
    auto questionEntrance = extractEntranceWithMandatoryPosition(
        originalTask_.questionContext(), Dialog::Question);

    auto answerEntrance = extractEntranceWithMandatoryPosition(
        originalTask_.answerContext(), Dialog::Answer);

    // Filling feedback task
    //
    position_ = *questionEntrance.center();
    type_ = sf::Type::Entrance;

    description_ = sf::EntranceWrongPositionDescr{
        questionEntrance.name()
    }.toDescription();

    if (questionEntrance.name()) {
        attrs_.addCustom(sfa::ENTRANCE_NAME, *questionEntrance.name());
    }
    attrs_.add(
        sf::AttrType::ObjectDiff,
        objectToJsonValue(PositionDiff(
            geolib3::convertMercatorToGeodetic(*questionEntrance.center()),
            geolib3::convertMercatorToGeodetic(*answerEntrance.center())
        ))
    );
}

void FbapiIssueParser::initFromOtherContext(
    sf::Type type,
    const std::string& tankerKey)
{
    hidden_ = true;
    type_ = type;

    auto answerContext = originalTask_.answerContext();
    PARSING_REQUIRE(answerContext, "Answer context expected");
    auto otherContext = answerContext->as<fbapi::OtherContext>();
    attrs_.addCustom(sfa::USER_DATA_COMMENT, otherContext.text());

    description_ = sf::DescriptionI18n(tankerKey);
}
// init
void FbapiIssueParser::initWithCommentOrPhotoRequired(
    sf::Type type,
    const std::string& tankerKey)
{
    hidden_ = true;
    type_ = type;

    PARSING_REQUIRE(
        attrs_.existCustom(sfa::USER_DATA_COMMENT) || attrs_.exist(sf::AttrType::UserDataPhotoUrls),
        "Message from user or attached photos expected"
    );

    description_ = sf::DescriptionI18n(tankerKey);
}

FbapiIssueParser::FbapiIssueParser(
    IUriResolver& uriResolver,
    const fbapi::Task& task)
    : originalTask_(task.originalTask())
{
    initCommonFields(uriResolver, task);

    namespace qid = question_id;
    namespace aid = answer_id;

    auto qId = originalTask_.questionId();
    auto aId = originalTask_.answerId();
    if (qId == qid::ADD_OBJECT && aId == aid::BARRIER) {
        initAddObjectBarrier();
    } else if (qId == qid::ADD_OBJECT && aId == aid::CROSSWALK) {
        initAddObjectCrosswalk();
    } else if (qId == qid::ADD_OBJECT && aId == aid::ENTRANCE) {
        initAddObjectEntrance();
    } else if (qId == qid::ADD_OBJECT && aId == aid::FENCE) {
        initAddObjectFence();
    } else if (qId == qid::ADD_OBJECT && aId == aid::GATE) {
        initAddObjectGate();
    } else if (qId == qid::ADD_OBJECT && aId == aid::OTHER) {
        initAddObjectOther();
    } else if (qId == qid::ADD_OBJECT && aId == aid::PARKING) {
        initAddObjectParking();
    } else if (qId == qid::ADD_OBJECT && aId == aid::ROAD) {
        initAddObjectRoad();
    } else if (qId == qid::ADD_OBJECT && aId == aid::SETTLEMENT_SCHEME) {
        initWithCommentOrPhotoRequired(sf::Type::SettlementScheme, sf::tanker::fb_desc::SETTLEMENT_SCHEME_KEY);
    } else if (qId == qid::ADD_OBJECT && aId == aid::STOP) {
        initAddObjectStop();
    } else if (qId == qid::ADD_OBJECT && aId == aid::TOPONYM) {
        initAddObjectToponym();
    } else if (qId == qid::OTHER && aId == aid::AUTO) {
        initFromOtherContext(sf::Type::CarRoute, sf::tanker::fb_desc::OTHER_AUTO_KEY);
    } else if (qId == qid::OTHER && aId == aid::AUTO_ANNOTATIONS) {
        initFromOtherContext(sf::Type::CarRoute, sf::tanker::fb_desc::OTHER_AUTO_ANNOTATIONS_KEY);
    } else if (qId == qid::OTHER && aId == aid::AUTO_BAD_PAVEMENT) {
        initFromOtherContext(sf::Type::CarRoute, sf::tanker::fb_desc::OTHER_AUTO_BAD_PAVEMENT_KEY);
    } else if (qId == qid::OTHER && aId == aid::COMMENT) {
        initWithCommentOrPhotoRequired(sf::Type::Other, sf::tanker::fb_desc::USER_COMMENT_KEY);
    } else if (qId == qid::OTHER && aId == aid::BICYCLE) {
        initFromOtherContext(sf::Type::BicycleRoute, sf::tanker::fb_desc::OTHER_BICYCLE_KEY);
    } else if (qId == qid::OTHER && aId == aid::INDOOR) {
        initFromOtherContext(sf::Type::Indoor, sf::tanker::fb_desc::OTHER_INDOOR_KEY);
    } else if (qId == qid::OTHER && aId == aid::MAP_OTHER) {
        initFromOtherContext(sf::Type::Other, sf::tanker::fb_desc::OTHER_MAP_KEY);
    } else if (qId == qid::OTHER && aId == aid::MASSTRANSIT) {
        initFromOtherContext(sf::Type::MtRoute, sf::tanker::fb_desc::OTHER_MASSTRANSIT_KEY);
    } else if (qId == qid::OTHER && aId == aid::PEDESTRIAN) {
        initFromOtherContext(sf::Type::PedestrianRoute, sf::tanker::fb_desc::OTHER_PEDESTRIAN_KEY);
    } else if (qId == qid::OTHER && aId == aid::POI) {
        initFromOtherContext(sf::Type::Poi, sf::tanker::fb_desc::OTHER_POI_KEY);
    } else if (qId == qid::OTHER && aId == aid::TOPONYM) {
        initFromOtherContext(sf::Type::Address, sf::tanker::fb_desc::OTHER_TOPONYM_KEY);
    } else if (qId == qid::OTHER && aId == aid::TRUCK) {
        initFromOtherContext(sf::Type::TruckRoute, sf::tanker::fb_desc::OTHER_TRUCK_KEY);
    } else if (qId == qid::REMOVE_OBJECT && aId == aid::TOPONYM) {
        initRemoveObjectToponym();
    } else if (qId == qid::WRONG_ADDRESS && aId == aid::DONT_KNOW) {
        initWrongAddressDontKnow();
    } else if (qId == qid::WRONG_ADDRESS && aId == aid::REPORT_ADDRESS) {
        initWrongAddressReportAddress();
    } else if (qId == qid::WRONG_ADDRESS && aId == aid::REPORT_LOCATION) {
        initWrongAddressReportLocation();
    } else if (qId == qid::WRONG_BARRIER && aId == aid::NOT_FOUND) {
        initWrongBarrierNotFound(sf::tanker::fb_desc::REMOVE_BARRIER_KEY);
    } else if (qId == qid::WRONG_BARRIER && aId == aid::REPORT_LOCATION) {
        initWrongBarrierReportLocation(sf::tanker::fb_desc::BARRIER_WRONG_POS_KEY);
    } else if (qId == qid::WRONG_GATE && aId == aid::NOT_FOUND) {
        initWrongBarrierNotFound(sf::tanker::fb_desc::REMOVE_GATE_KEY);
    } else if (qId == qid::WRONG_GATE && aId == aid::REPORT_LOCATION) {
        initWrongBarrierReportLocation(sf::tanker::fb_desc::GATE_WRONG_POS_KEY);
    } else if (qId == qid::WRONG_ENTRANCE && aId == aid::NOT_FOUND) {
        initWrongEntranceNotFound();
    } else if (qId == qid::WRONG_ENTRANCE && aId == aid::REPORT_ENTRANCE) {
        initWrongEntranceReportEntrance();
    } else if (qId == qid::WRONG_ENTRANCE && aId == aid::REPORT_LOCATION) {
        initWrongEntranceReportLocation();
    } else if (qId == qid::WRONG_LINES && aId == aid::REPORT_LINES) {
        initWrongLinesReportLines();
    } else if (qId == qid::WRONG_LOCATION && aId == aid::REPORT_LOCATION) {
        initWrongLocationReportLocation();
    } else if (qId == qid::WRONG_NAME && aId == aid::REPORT_NAME) {
        initWrongNameReportName();
    } else if (qId == qid::WRONG_POSITION && aId == aid::DEMOLISHED) {
        initWrongPositionDemolished();
    } else if (qId == qid::WRONG_POSITION && aId == aid::NEVER_BEEN_HERE) {
        initWrongPositionNeverBeenHere();
    } else if (qId == qid::WRONG_POSITION && aId == aid::NOT_FOUND) {
        initWrongPositionNotFound();
    } else if (qId == qid::WRONG_POSITION && aId == aid::OTHER) {
        initWrongPositionOther();
    } else if (qId == qid::WRONG_ROUTE && aId == aid::REPORT_ROUTE) {
        initWrongRouteReportRoute(uriResolver);
    } else if (qId == qid::WRONG_SUBWAY && aId == aid::NOT_FOUND) {
        initWrongSubwayNotFound();
    } else if (qId == qid::WRONG_SUBWAY && aId == aid::REPORT_SUBWAY) {
        initWrongSubwayReportSubway();
    } else if (qId == qid::WRONG_SUBWAY && aId == aid::OTHER) {
        initWithCommentOrPhotoRequired(sf::Type::Subway, sf::tanker::fb_desc::SUBWAY_OTHER_KEY);
    } else {
        throw IssueParseError() << "Unexpected pair "
            << "question_id='" << originalTask_.questionId() << "' "
            << "answer_id='" << originalTask_.answerId() << "'";
    }
}

} // namespace maps::wiki::sync_fbapi_feedback
