#include "import_routine.h"

#include "common_routines.h"
#include "fbapi_issue_parser.h"

#include <maps/wikimap/mapspro/libs/social/include/yandex/maps/wiki/social/feedback/task.h>
#include <maps/wikimap/mapspro/libs/social/include/yandex/maps/wiki/social/feedback/task_new.h>

#include <maps/wikimap/mapspro/libs/http/include/yandex/maps/wiki/http/fbapi/models/common.h>

#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/common/robot.h>
#include <yandex/maps/wiki/social/feedback/agent.h>
#include <yandex/maps/wiki/social/feedback/task_filter.h>
#include <yandex/maps/wiki/social/fbapi_issue.h>
#include <yandex/maps/wiki/social/gateway.h>

#include <algorithm>
#include <string>
#include <set>
#include <unordered_map>
#include <vector>

namespace maps::wiki::sync_fbapi_feedback {

namespace sf = social::feedback;

namespace {

const std::string ERROR_MESSAGE_NO_INFO_PROVIDED = "No info provided in fbapi answer.";

sf::TaskNew convertFbapiTaskToSocial(
    IUriResolver& uriResolver,
    const fbapi::Task& fbapiTask)
{
    FbapiIssueParser fbapiIssueParser(uriResolver, fbapiTask);

    sf::TaskNew socialTask(
        fbapiIssueParser.position(),
        fbapiIssueParser.type(),
        fbapiIssueParser.source(),
        fbapiIssueParser.description());

    socialTask.objectId = fbapiIssueParser.objectId();
    socialTask.hidden = fbapiIssueParser.hidden();
    socialTask.attrs = fbapiIssueParser.attrs();

    return socialTask;
}

void importTask(
    const fbapi::Task& fbapiTask,
    fbapi::IGateway& fbapiGateway,
    pqxx::transaction_base& socialTxn,
    IUriResolver& uriResolver)
{
    INFO() << "Importing: " << fbapiTask.id();
    // 1) Add new task to 'feedback_task' and 'fbapi_issue' social tables
    //
    auto socialTask = convertFbapiTaskToSocial(uriResolver, fbapiTask);
    auto dbSocialTask = sf::GatewayRW(socialTxn).addTask(common::ROBOT_UID, socialTask);

    auto fbapiIssue = social::addFbapiIssue(
        socialTxn,
        fbapiTask.id(),
        dbSocialTask.id(),
        fbapiTask.data());

    social::updateFbapiIssueHashValue(
        socialTxn,
        fbapiIssue.id(),
        dbSocialTask.hashValue());

    // 2) Update fbapi task status in fbapi service
    //
    auto params = makeUpdateStatusParams(fbapi::TaskStatus::InProgress);
    params.setServiceObjectId(std::to_string(dbSocialTask.id()));
    fbapiGateway.changeTaskById(fbapiTask.id(), params);

    INFO() << "Task with id=" << fbapiTask.id()
           << " successfully imported; social task id=" << dbSocialTask.id();
}

std::string getProvidedMessageFromNew(const fbapi::TaskChanges& sortedHistory)
{
    const auto rIter = std::find_if(
        std::rbegin(sortedHistory),
        std::rend(sortedHistory),
        [](const fbapi::TaskChange& item) {
            return item.status() == fbapi::TaskStatus::New;
        }
    );
    if (rIter != std::rend(sortedHistory) && rIter->message()) {
        return rIter->message().value();
    }
    return ERROR_MESSAGE_NO_INFO_PROVIDED;
}

void reopenTask(
    const fbapi::Task& fbapiTask,
    const social::FbapiIssue& fbapiIssue,
    fbapi::IGateway& fbapiGateway,
    pqxx::transaction_base& socialTxn)
{
    INFO() << "Reopening: " << fbapiIssue.issueId();

    sf::Agent agent(socialTxn, common::ROBOT_UID);
    auto socialTask = agent.taskForUpdateById(fbapiIssue.feedbackTaskId());
    ASSERT(socialTask);

    auto validOperations = sf::Agent::validOperations(*socialTask, common::ROBOT_UID);
    // open feedback::Task if it possible;
    // even if it isn't in NeedInfo status
    if (validOperations.count(sf::TaskOperation::Open)) {
        // reopen feedback task with comment
        social::Gateway gw(socialTxn);
        auto comment = gw.createComment(
            common::ROBOT_UID,
            social::CommentType::Info,
            getProvidedMessageFromNew(fbapiTask.sortedHistory()),
            0,
            0,
            fbapiIssue.feedbackTaskId(),
            {}
        );

        socialTask = agent.openTask(*socialTask, comment.id());
        ASSERT(socialTask);

        socialTask = agent.hideTask(*socialTask);
        ASSERT(socialTask);
    } else {
        WARN() << "Failed to reopen feedback when needed info received. FbapiIssue id: " << fbapiIssue.issueId()
           << ", social task id=" << socialTask->id();
    }

    fbapiGateway.changeTaskById(
        fbapiTask.id(),
        makeUpdateStatusParams(fbapiStatusFromSocialTask(*socialTask)));

    INFO() << "Task with id=" << fbapiIssue.issueId()
           << " successfully updated; social task id=" << socialTask->id();
}

} // namespace

void importRoutine(
    fbapi::IGateway& fbapiGateway,
    pgpool3::Pool& socialPool,
    tasks::StatusWriter& statusWriter,
    IUriResolver& uriResolver)
{
    std::vector<fbapi::Task> tasksToImport;
    fbapi::TasksFilter filter(fbapi::Service::Nmaps, fbapi::TaskStatus::New);
    fbapi::visitTasks(
        fbapiGateway,
        filter,
        fbapi::WithHistory::Yes,
        [&](const fbapi::Task& task) {
            tasksToImport.push_back(task);
            return true;
        }
    );

    INFO() << "Number of 'new' fbapi tasks: " << tasksToImport.size();

    int errors = 0;
    int rejected = 0;

    std::unordered_map<std::string, social::FbapiIssue> issueIdToFbapiIssue;
    {
        std::set<std::string> issueIds;
        for (const auto& task : tasksToImport) {
            issueIds.insert(task.id());
        }

        social::FbapiIssueFilter filter;
        filter.issueIds(std::move(issueIds));
        auto issues = fbapiIssuesByFilter(socialPool.slaveTransaction().get(), filter);
        for (auto& issue: issues) {
            issueIdToFbapiIssue.emplace(issue.issueId(), std::move(issue));
        }
    }

    for (const auto& task : tasksToImport) try {
        auto issueIt = issueIdToFbapiIssue.find(task.id());
        if (issueIt == issueIdToFbapiIssue.end()) {
            // create new task
            auto txnSocialWrite = socialPool.masterWriteableTransaction();
            importTask(task, fbapiGateway, txnSocialWrite.get(), uriResolver);
            txnSocialWrite->commit();
        } else { // update existing task
            auto txnSocialWrite = socialPool.masterWriteableTransaction();
            reopenTask(
                task, issueIt->second, fbapiGateway,
                txnSocialWrite.get());
            txnSocialWrite->commit();
        }
    } catch (const IssueParseError& ex) {
        WARN() << "Task '" << task.id() << "' can't be parsed. "
               << "Its status remains 'new'. Parsing error: " << ex.what();
        errors++;
    } catch (const std::exception& ex) {
        WARN() << "Failed to import task with id=" << task.id() << " - skipped; ex: " << ex.what();
        errors++;
    }

    if (errors) {
        INFO() << "Errors count: " << errors;
        statusWriter.warn(
           "Import new tasks errors: " + std::to_string(errors));
    }
    INFO() << "Rejected: " << rejected << " tasks";
}

} // namespace maps::wiki::sync_fbapi_feedback
