#include "worker.h"

#include "import_routine.h"
#include "sync_routine.h"

#include <maps/libs/auth/include/tvm.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/pgpool3utils/pg_advisory_mutex.h>
#include <yandex/maps/wiki/common/pg_advisory_lock_ids.h>

#include <optional>
#include <string>

namespace maps::wiki::sync_fbapi_feedback {

Worker::Worker(const common::ExtendedXmlDoc& cfg,
        tasks::StatusWriter& statusWriter)
    : socialPool_(cfg, "social", "grinder")
    , tvmClient_(auth::TvmtoolSettings().selectClientAlias("maps-core-nmaps-tasks-feedback").makeTvmClient())
    , fbapiGateway_(
        fbapi::Configuration(
            cfg.get<std::string>("/config/common/fbapi/base-url"))
            .setTvmTicketProvider([&]() {
                return tvmClient_.GetServiceTicketFor("feedback-api");
            }),
        maps::common::RetryPolicy()
            .setTryNumber(1)
            .setInitialCooldown(std::chrono::seconds{1})
            .setCooldownBackoff(2))
    , statusWriter_(statusWriter)
    , uriResolver_(
        geosearch_client::Client(
            geosearch_client::Configuration(
                cfg.get<std::string>("/config/common/geocoder/base-url"),
                "maps-core-nmaps-tasks-feedback-sync-fbapi"
            ).setTimeout(std::chrono::seconds(3)
            ).setTvmTicketProvider([&]() {
                return tvmClient_.GetServiceTicketFor("geocoder");
            })
        ),
        ResolveUriFromRouterClient(
            RouterConfig(
                cfg.get<std::string>("/config/common/driving-router/base-url")
            ).setTimeout(std::chrono::seconds(3)
            ).setTvmTicketProvider([&]() {
                return tvmClient_.GetServiceTicketFor("driving-router");
            })
        ),
        MTRouterClient(
            RouterConfig(
                cfg.get<std::string>("/config/common/mt-router/base-url")
            ).setTimeout(std::chrono::seconds(3)
            ).setTvmTicketProvider([&]() {
                return tvmClient_.GetServiceTicketFor("mt-router");
            })
        ),
        ResolveUriFromRouterClient(
            RouterConfig(
                cfg.get<std::string>("/config/common/bicycle-router/base-url")
            ).setTimeout(std::chrono::seconds(3)
            ).setTvmTicketProvider([&]() {
                return tvmClient_.GetServiceTicketFor("bicycle-router");
            })
        )
    )
{}

void Worker::doTask()
{
    statusWriter_.reset();

    std::optional<std::string> err;
    try {
        // Exclusive execution between hosts
        //
        pgp3utils::PgAdvisoryXactMutex xactMutex(
            socialPool_.pool(),
            static_cast<int64_t>(common::AdvisoryLockIds::SYNC_FBAPI_FEEDBACK));
        if (!xactMutex.try_lock()) {
            INFO() << "Database is already locked. Task interrupted.";
            return;
        }

        importRoutine(fbapiGateway_, socialPool_.pool(), statusWriter_, uriResolver_);
        syncRoutine(fbapiGateway_, socialPool_.pool(), statusWriter_);

    } catch (const maps::Exception &ex) {
        err = (std::stringstream() << ex).str();
    } catch (const std::exception& ex) {
        err = ex.what();
    } catch (...) {
        err = "unknown exception";
    }

    if (err) {
        ERROR() << "Task ERROR: " << err.value();
        statusWriter_.err(err.value());
    }

    statusWriter_.flush();
}

} // namespace maps::wiki::sync_fbapi_feedback
