#include "mock_fbapi_gtw.h"

#include <maps/wikimap/mapspro/services/tasks_feedback/src/sync_fbapi_feedback_worker/lib/import_routine.h>
#include <maps/wikimap/mapspro/services/tasks_feedback/src/sync_fbapi_feedback_worker/lib/sync_routine.h>

#include <maps/libs/introspection/include/comparison.h>
#include <maps/libs/introspection/include/stream_output.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/social/gateway.h>
#include <yandex/maps/wiki/social/fbapi_issue.h>
#include <yandex/maps/wiki/social/feedback/agent.h>
#include <yandex/maps/wiki/social/feedback/gateway_ro.h>
#include <yandex/maps/wiki/unittest/localdb.h>

#include <boost/test/unit_test.hpp>

#include <map>

namespace maps::wiki::sync_fbapi_feedback::tests {

namespace sf = social::feedback;

namespace {

struct FbapiChangeParams
{
    FbapiChangeParams() = default;
    explicit FbapiChangeParams(const fbapi::ChangeTaskParams& params)
        : newService(params.newService())
        , newStatus(params.newStatus())
        , actorService(params.actorService())
        , message(params.message())
        , requestTemplateId(params.requestTemplateId())
        , resolution(params.resolution())
    {}

    std::optional<fbapi::Service> newService;
    std::optional<fbapi::TaskStatus> newStatus;
    std::optional<fbapi::Service> actorService;
    std::optional<std::string> message;
    std::optional<std::string> requestTemplateId;
    std::optional<fbapi::TaskResolution> resolution;

    auto introspect() const {
        return std::tie(
            newService,
            newStatus,
            actorService,
            message,
            requestTemplateId,
            resolution);
    }
};

using introspection::operator==;
using introspection::operator<<;

class StubUriResolver : public IUriResolver
{
public:
    StubUriResolver() {}

    std::optional<std::string> resolveObjectUri(const std::string&) override {
        return std::nullopt;
    }
    std::optional<Route> resolveDrivingRouteUri(const std::string&) override {
        return std::nullopt;
    }
    std::optional<Route> resolveMtRouteUri(const std::string&) override {
        return std::nullopt;
    }
    std::optional<Route> resolvePedestrianRouteUri(const std::string&) override {
        return std::nullopt;
    }
    std::optional<Route> resolveBicycleRouteUri(const std::string&) override {
        return std::nullopt;
    }
    std::optional<Route> resolveScooterRouteUri(const std::string&) override {
        return std::nullopt;
    }
};

struct SetLogLevelFixture
{
    SetLogLevelFixture() {
        maps::log8::setLevel(maps::log8::Level::INFO);
    }
};

class DBFixture : public unittest::MapsproDbFixture, public SetLogLevelFixture
{
public:
    DBFixture()
        : unittest::MapsproDbFixture()
    {}
};

std::string loadMandatoryCommentText(
    pqxx::transaction_base& socialTxn,
    social::TId commentId)
{
    social::Gateway gw(socialTxn);
    const auto comments = gw.loadComments({commentId});
    ASSERT(!comments.empty() && !comments.front().data().empty());
    return comments.front().data();
}

const std::string FBAPI_ISSUE_KEY1 {"2d9d8fbb2a014c718fab7eb9c1f8a0ff"};
const std::string FBAPI_ISSUE_KEY2 {"80fabc4131794c688a8a3bb0462e3b88"};
const social::TUid USER_ID = 11;
StubUriResolver stubUriResolver;
tasks::StatusWriter statusWriter(std::nullopt);

} // anonymous namespace

BOOST_FIXTURE_TEST_CASE(test_import_and_sync, DBFixture)
{
    {
        MockFeedbackGateway mockGateway("new");
        importRoutine(mockGateway, pool(), statusWriter, stubUriResolver);
        std::set<std::string> expectedChangedTaskIds{
                FBAPI_ISSUE_KEY1,
                FBAPI_ISSUE_KEY2
        };
        BOOST_CHECK(mockGateway.changedTaskIds() == expectedChangedTaskIds);
    }

    {
        // reveal one feedback task
        auto txn = pool().masterWriteableTransaction();
        auto fbapiIssue = social::fbapiIssueById(txn.get(), 1);
        BOOST_REQUIRE(fbapiIssue);
        BOOST_REQUIRE(fbapiIssue->issueId() == FBAPI_ISSUE_KEY1);

        sf::Agent agent(txn.get(), USER_ID);
        auto task = agent.revealTaskByIdCascade(fbapiIssue->feedbackTaskId());

        txn->commit();
    }

    {
        MockFeedbackGateway mockGateway("in-progress");
        syncRoutine(mockGateway, pool(), statusWriter);

        std::set<std::string> expectedChangedTaskIds{
            // empty set
        };
        BOOST_CHECK(mockGateway.changedTaskIds() == expectedChangedTaskIds);
    }

    {
        // accept one feedback task
        auto txn = pool().masterWriteableTransaction();
        auto fbapiIssue = social::fbapiIssueById(txn.get(), 1);
        BOOST_REQUIRE(fbapiIssue);
        BOOST_REQUIRE(fbapiIssue->issueId() == FBAPI_ISSUE_KEY1);

        sf::Agent agent(txn.get(), USER_ID);
        auto task = agent.taskForUpdateById(fbapiIssue->feedbackTaskId());
        BOOST_ASSERT(task);
        agent.resolveTaskCascade(task.value(), sf::Resolution::createAccepted());

        txn->commit();
    }

    {
        MockFeedbackGateway mockGateway("in-progress");
        syncRoutine(mockGateway, pool(), statusWriter);

        std::set<std::string> expectedChangedTaskIds{
                FBAPI_ISSUE_KEY1
        };
        BOOST_CHECK(mockGateway.changedTaskIds() == expectedChangedTaskIds);
    }
}

BOOST_FIXTURE_TEST_CASE(test_need_info, DBFixture)
{
    const std::string REQUEST_TEMPLATE = "need-full-address";
    const std::string REQUEST_MESSAGE = "request message";
    const std::string RESPONSE_MESSAGE = "response message";

    {
        MockFeedbackGateway mockGateway("new");
        importRoutine(mockGateway, pool(), statusWriter, stubUriResolver);
        std::set<std::string> expectedChangedTaskIds{
            FBAPI_ISSUE_KEY1,
            FBAPI_ISSUE_KEY2
        };
        BOOST_CHECK(mockGateway.changedTaskIds() == expectedChangedTaskIds);
    }

    {
        // move one feedback task into need-info
        auto txn = pool().masterWriteableTransaction();
        auto fbapiIssue = social::fbapiIssueById(txn.get(), 2);
        BOOST_REQUIRE(fbapiIssue);
        BOOST_REQUIRE(fbapiIssue->issueId() == FBAPI_ISSUE_KEY2);

        sf::Agent agent(txn.get(), USER_ID);
        auto task = agent.revealTaskByIdCascade(fbapiIssue->feedbackTaskId());
        BOOST_ASSERT(task);

        social::Gateway gw(txn.get());
        auto comment = gw.createComment(
            USER_ID,
            social::CommentType::Info,
            REQUEST_MESSAGE,
            0,
            0,
            task->id(),
            {}
        );

        agent.needInfoTask(task.value(), comment.id(), REQUEST_TEMPLATE);

        txn->commit();
    }

    {
        MockFeedbackGateway mockGateway("in-progress");
        syncRoutine(mockGateway, pool(), statusWriter);

        std::set<std::string> expectedChangedTaskIds{
            FBAPI_ISSUE_KEY2
        };
        BOOST_CHECK(mockGateway.changedTaskIds() == expectedChangedTaskIds);

        BOOST_CHECK_EQUAL(
            FbapiChangeParams(mockGateway.changedTaskParams(FBAPI_ISSUE_KEY2)),
            FbapiChangeParams(
                fbapi::ChangeTaskParams()
                    .setNewStatus(fbapi::TaskStatus::NeedInfo)
                    .setActorService(fbapi::Service::Nmaps)
                    .setMessage(REQUEST_MESSAGE)
                    .setRequestTemplateId(REQUEST_TEMPLATE)
            )
        );
    }

    {
        MockFeedbackGateway mockGateway("need-info");
        importRoutine(mockGateway, pool(), statusWriter, stubUriResolver);

        std::set<std::string> expectedChangedTaskIds{
            FBAPI_ISSUE_KEY2
        };
        BOOST_CHECK(mockGateway.changedTaskIds() == expectedChangedTaskIds);

        BOOST_CHECK_EQUAL(
            FbapiChangeParams(mockGateway.changedTaskParams(FBAPI_ISSUE_KEY2)),
            FbapiChangeParams(
                fbapi::ChangeTaskParams()
                    .setNewService(fbapi::Service::Nmaps)
                    .setNewStatus(fbapi::TaskStatus::InProgress)
                    .setActorService(fbapi::Service::Nmaps)
            )
        );
    }

    {
        auto txn = pool().masterWriteableTransaction();
        auto fbapiIssue = social::fbapiIssueById(txn.get(), 2);
        BOOST_REQUIRE(fbapiIssue);
        BOOST_REQUIRE(fbapiIssue->issueId() == FBAPI_ISSUE_KEY2);
        sf::GatewayRO gwRO(txn.get());
        auto task = gwRO.taskById(fbapiIssue->feedbackTaskId());
        BOOST_ASSERT(task);
        const auto& history = gwRO.history(task->id());

        const auto& historyItems = history.items();
        const auto iter = std::find_if(
            std::rbegin(historyItems),
            std::rend(historyItems),
            [](const sf::HistoryItem& item) {
                return item.operation() == sf::TaskOperation::Open;
            }
        );
        BOOST_ASSERT(iter != std::rend(historyItems));
        BOOST_ASSERT(iter->commentId());
        const auto commentMessage =
            loadMandatoryCommentText(txn.get(), iter->commentId().value());

        BOOST_CHECK_EQUAL(commentMessage, RESPONSE_MESSAGE);
    }
}

BOOST_FIXTURE_TEST_CASE(test_reject_reason, DBFixture)
{
    {
        MockFeedbackGateway mockGateway("new");
        importRoutine(mockGateway, pool(), statusWriter, stubUriResolver);
        std::set<std::string> expectedChangedTaskIds{
            FBAPI_ISSUE_KEY1,
            FBAPI_ISSUE_KEY2
        };
        BOOST_CHECK(mockGateway.changedTaskIds() == expectedChangedTaskIds);
    }

    {
        // reject one feedback task
        auto txn = pool().masterWriteableTransaction();
        auto fbapiIssue = social::fbapiIssueById(txn.get(), 1);
        BOOST_REQUIRE(fbapiIssue);
        BOOST_REQUIRE(fbapiIssue->issueId() == FBAPI_ISSUE_KEY1);

        sf::Agent agent(txn.get(), USER_ID);
        auto task = agent.revealTaskByIdCascade(fbapiIssue->feedbackTaskId());
        BOOST_ASSERT(task);
        agent.resolveTaskCascade(
            task.value(),
            sf::Resolution::createRejected(sf::RejectReason::Spam)
        );

        txn->commit();
    }

    {
        MockFeedbackGateway mockGateway("in-progress");
        syncRoutine(mockGateway, pool(), statusWriter);

        std::set<std::string> expectedChangedTaskIds{
            FBAPI_ISSUE_KEY1
        };
        BOOST_CHECK(mockGateway.changedTaskIds() == expectedChangedTaskIds);

        BOOST_CHECK_EQUAL(
            FbapiChangeParams(mockGateway.changedTaskParams(FBAPI_ISSUE_KEY1)),
            FbapiChangeParams(
                fbapi::ChangeTaskParams()
                    .setNewService(fbapi::Service::Nmaps)
                    .setNewStatus(fbapi::TaskStatus::Rejected)
                    .setActorService(fbapi::Service::Nmaps)
                    .setResolution(fbapi::TaskResolution::Spam)
            )
        );
    }

    {
        // reopen and close with the same reject reason (fbapi resolution)
        auto txn = pool().masterWriteableTransaction();
        auto fbapiIssue = social::fbapiIssueById(txn.get(), 1);
        BOOST_REQUIRE(fbapiIssue);
        BOOST_REQUIRE(fbapiIssue->issueId() == FBAPI_ISSUE_KEY1);

        sf::Agent agent(txn.get(), USER_ID);
        auto task = agent.revealTaskByIdCascade(fbapiIssue->feedbackTaskId());
        BOOST_ASSERT(task);
        agent.resolveTaskCascade(
            task.value(),
            sf::Resolution::createRejected(sf::RejectReason::Spam)
        );

        txn->commit();
    }

    {
        MockFeedbackGateway mockGateway("rejected");
        syncRoutine(mockGateway, pool(), statusWriter);

        std::set<std::string> expectedChangedTaskIds{
            // empty
        };
        BOOST_CHECK(mockGateway.changedTaskIds() == expectedChangedTaskIds);
    }

    {
        // reopen and change reject reason
        auto txn = pool().masterWriteableTransaction();
        auto fbapiIssue = social::fbapiIssueById(txn.get(), 1);
        BOOST_REQUIRE(fbapiIssue);
        BOOST_REQUIRE(fbapiIssue->issueId() == FBAPI_ISSUE_KEY1);

        sf::Agent agent(txn.get(), USER_ID);
        auto task = agent.revealTaskByIdCascade(fbapiIssue->feedbackTaskId());
        BOOST_ASSERT(task);
        agent.resolveTaskCascade(
            task.value(),
            sf::Resolution::createRejected(sf::RejectReason::RedirectToSupport)
        );

        txn->commit();
    }

    {
        MockFeedbackGateway mockGateway("rejected");
        syncRoutine(mockGateway, pool(), statusWriter);

        std::set<std::string> expectedChangedTaskIds{
            FBAPI_ISSUE_KEY1
        };
        BOOST_CHECK(mockGateway.changedTaskIds() == expectedChangedTaskIds);

        BOOST_CHECK_EQUAL(
            FbapiChangeParams(mockGateway.changedTaskParams(FBAPI_ISSUE_KEY1)),
            FbapiChangeParams(
                fbapi::ChangeTaskParams()
                    .setNewService(fbapi::Service::Nmaps)
                    .setNewStatus(fbapi::TaskStatus::Rejected)
                    .setActorService(fbapi::Service::Nmaps)
                    .setResolution(fbapi::TaskResolution::RedirectToSupport)
            )
        );
    }
}

} // namespace maps::wiki::sync_fbapi_feedback::tests
