#include "worker.h"

#include <maps/libs/json/include/value.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/pgpool3utils/pg_advisory_mutex.h>
#include <yandex/maps/wiki/common/pg_advisory_lock_ids.h>
#include <yandex/maps/wiki/common/robot.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/social/feedback/converter_checks.h>
#include <yandex/maps/wiki/social/feedback/description.h>
#include <yandex/maps/wiki/social/feedback/description_keys.h>
#include <yandex/maps/wiki/social/feedback/gateway_rw.h>
#include <yandex/maps/wiki/social/feedback/twins.h>
#include <yandex/maps/wiki/tasks/task_logger.h>

#include <maps/libs/introspection/include/hashing.h>

#include <algorithm>
#include <sstream>
#include <thread>
#include <boost/algorithm/string/replace.hpp>

namespace std {

template<>
class hash<maps::wiki::validator::storage::MessageId> {
public:
    size_t operator()(const maps::wiki::validator::storage::MessageId& message) const
    {
        return maps::introspection::computeHash(message);
    }
};

} // namespace std

namespace maps::wiki::validation_feedback_converter {

namespace {

namespace sf  = maps::wiki::social::feedback;
namespace rf = revision::filters;

const std::string ATTR_IDS = "ids";
const std::string ATTR_CONTEXT = "sourceContext";
const std::string ATTR_DESC = "description";

const uint LIMIT_MESSAGES = 30000;

constexpr const std::chrono::seconds POLLING_INTERVAL{5};

const std::string NEW_TASK_SOURCE_PREFIX = "validation-";

std::string getSourceFromCheckId(const std::string& checkId)
{
    std::string source = NEW_TASK_SOURCE_PREFIX + checkId;
    boost::replace_all(source, "_", "-");
    return source;
}

std::optional<sf::DescriptionI18n> descriptionFromMessage(
    const std::string & description)
{
    auto validatorKey = sf::tankerKeyFromValidationDescription(description);
    if (!validatorKey) {
        validatorKey = sf::tankerKeyFromValidationDescriptionRoadWithoutGeometry(description);
    }
    if (validatorKey) {
        return sf::DescriptionI18n(
            sf::tanker::fb_desc::VALIDATION_SIMPLE_KEY,
            {
                {sf::tanker::VALIDATOR_DESCRIPTION,
                    sf::DescriptionI18n(validatorKey.value(), {})
                }
            });
    }
    return std::nullopt;
}

std::string createAttrIdsFromRevisionIds(const std::vector<revision::RevisionID>& revisionIds)
{
    std::vector<revision::DBID> objectIds;

    for (auto i : revisionIds) {
        objectIds.push_back(i.objectId());
    }
    std::sort(objectIds.begin(), objectIds.end());
        return common::join(objectIds, ',');
}

json::Value createContextFromRevisionIds(const std::vector<revision::RevisionID>& revisionIds)
{
    std::vector<json::Value> ids;
    for (auto i : revisionIds) {
        ids.emplace_back(std::to_string(i.objectId()));
    }

    return json::Value(json::repr::ObjectRepr{
            {"type", json::Value("validation")},
            {"content", json::Value(json::repr::ObjectRepr{
                        {"objectIds", json::Value(ids)}})}});
}

} // namespace

std::optional<geolib3::Point2> getPoint2FromWkb(const std::string& geomWkb)
{
    if (geomWkb.empty()) {
        return std::nullopt;
    }
    try {
        return geolib3::WKB::read<geolib3::Point2>(geomWkb);
    } catch (const maps::RuntimeError&) {
        try {
            return geolib3::WKB::read<geolib3::Polygon2>(geomWkb).exteriorRing().findCentroid();
        } catch (const maps::RuntimeError&) {
            return std::nullopt;
        }
    }
}

validator::storage::StoredMessageData getNewValidationTasksData(
    uint64_t validationTaskId,
    const validator::storage::MessageAttributesFilter& filter,
    pqxx::transaction_base& validationTxn,
    pqxx::transaction_base& coreTxn)
{
    validator::storage::ResultsGateway tasksGateway(validationTxn, validationTaskId);
    revision::RevisionsGateway revisionsGateway(coreTxn);
    auto snapshot = revisionsGateway.snapshot(revisionsGateway.headCommitId());

    auto newTasks = tasksGateway.messages(
        filter,
        snapshot,
        0, // offset
        LIMIT_MESSAGES + 1 // limit, try get limit+1 messages to check the excess
    );

    REQUIRE(newTasks.size() <= LIMIT_MESSAGES,
        "found more than " << LIMIT_MESSAGES << " validation tasks");
    REQUIRE(!newTasks.empty(),
        "no validation task found");
    return newTasks;
}

std::string getValidatorKeyFromDescription(sf::Description description)
{
    /**
     * description example:
     * {
     *   "i18nKey":"feedback-descriptions:validation-simple",
     *   "i18nParams":{
     *     "validatorDescription":{"i18nKey":"feedback-descriptions:validator-message-parking-territory-is-not-connected-to-parking-lot",
     *     "i18nParams":{}}
     *   }
     * }
     */
    return description.asTranslatable().i18nParams().at(sf::tanker::VALIDATOR_DESCRIPTION).asTranslatable().i18nKey();
}

geolib3::Point2 fillRoadGeometry(revision::DBID id,
                                 pqxx::transaction_base& coreTxn)
{
    revision::RevisionsGateway gtw(coreTxn);
    const auto snapshot = gtw.snapshot(gtw.maxSnapshotId());

    auto slaveRelations = snapshot.relationsByFilter(
        rf::ObjRevAttr::masterObjectId() == id &&
        rf::ObjRevAttr::isNotDeleted()
    );

    std::vector<revision::DBID> slaveObjectIds;
    for (const auto& relation : slaveRelations) {
        ASSERT(relation.data().relationData);
        slaveObjectIds.push_back(relation.data().relationData->slaveObjectId());
    }

    auto elements = snapshot.objectRevisionsByFilter(
        rf::ObjRevAttr::objectId().in(slaveObjectIds) &&
        rf::ObjRevAttr::isNotRelation() &&
        rf::ObjRevAttr::isNotDeleted() &&
        rf::Attr("cat:rd_el").defined() &&
        rf::Geom::defined());

    REQUIRE(!elements.empty(), "Road without geometry");
    auto geom = geolib3::WKB::read<geolib3::Polyline2>(*elements.front().data().geometry);
    return geom.pointAt(0);
}

size_t createFeedbackTasks(
    const validator::storage::StoredMessageData& tasksData,
    pqxx::transaction_base& socialTxn,
    pqxx::transaction_base& coreTxn)
{
    sf::Tasks insertedTasks;

    const auto isTasksEqual = [&](const sf::TaskNew& newTask, const sf::Task& oldTask) {
        if (newTask.attrs.getCustom(ATTR_IDS) != oldTask.attrs().getCustom(ATTR_IDS)){
            return false;
        }
        if (getValidatorKeyFromDescription(newTask.description)
            != getValidatorKeyFromDescription(oldTask.description())) {
            return false;
        }
        return true;
    };

    for (const auto& storedMessageDatum : tasksData) {
        const auto feedbackType = sf::taskTypeFromValidationCheckId(
            storedMessageDatum.message().attributes().checkId);
        if (!feedbackType) {
            continue;
        }
        const auto description = descriptionFromMessage(
            storedMessageDatum.message().attributes().description);
        // some validation tasks with bad description cannot be added
        if (!description) {
            continue;
        }
        auto position =
            getPoint2FromWkb(storedMessageDatum.message().geomWkb());

        if (!position) {
            auto validatorKey = sf::tankerKeyFromValidationDescriptionRoadWithoutGeometry(
                storedMessageDatum.message().attributes().description);
            if (validatorKey) {
                position = fillRoadGeometry(
                    storedMessageDatum.message().revisionIds()[0].objectId(),
                    coreTxn);
            }
        }

        if (!position) {
            WARN() << "Find task with bad geometry: 'check_id' = "
            << storedMessageDatum.message().attributes().checkId
            << ", 'description' = "
            << storedMessageDatum.message().attributes().description;
            continue;
        }
        sf::TaskNew newTask(
            position.value(),
            feedbackType.value(),
            getSourceFromCheckId(storedMessageDatum.message().attributes().checkId),
            description.value()
        );
        newTask.hidden = true;
        if (storedMessageDatum.message().revisionIds().size() == 1) {
            newTask.objectId = storedMessageDatum.message().revisionIds()[0].objectId();
        } else if (storedMessageDatum.message().revisionIds().size() > 1) {
            newTask.attrs.add(
                sf::AttrType::SourceContext,
                createContextFromRevisionIds(storedMessageDatum.message().revisionIds())
                );
        }

        newTask.attrs.addCustom(
            ATTR_IDS,
            createAttrIdsFromRevisionIds(storedMessageDatum.message().revisionIds())
            );

        sf::GatewayRW gatewayRw(socialTxn);
        const auto insertedTask = sf::addTaskIfNotTwin(
            gatewayRw,
            common::ROBOT_UID,
            newTask,
            isTasksEqual);

        if (insertedTask) {
            insertedTasks.push_back(insertedTask.value());
        }
    }
    socialTxn.commit();

    REQUIRE(!insertedTasks.empty(), "all feedback tasks already exist");

    return insertedTasks.size();
}

Worker::Worker(std::unique_ptr<common::ExtendedXmlDoc> configXml)
    : configXml_(std::move(configXml))
{
}

void Worker::doTask(const worker::Task& task)
try {
    uint64_t taskId = task.args()["taskId"].as<uint64_t>();
    common::PoolHolder corePool(*configXml_, "core", "grinder");
    tasks::TaskPgLogger logger(corePool.pool(), taskId);
    logger.logInfo() << "Import validation task started; task id = " << taskId;
    logger.logInfo() << "grinder task id = " << task.id();

    auto dualInfo = [&] (const std::string& message) {
        INFO() << message;
        logger.logInfo() << message;
    };

    const auto validationTaskId = task.args()["validationTaskId"].as<std::uint64_t>();
    dualInfo("worker param 'validationTaskId': " + std::to_string(validationTaskId));

    auto dualFilterParamInfo = [&] (std::string_view name, const auto& value) {
        auto paramInfo = [&] {
            std::ostringstream os;
            os << "worker param '" << name << "'";
            if (value) {
                os << ": " << *value;
            } else {
                os << " not set";
            }
            return os.str();
        };
        dualInfo(paramInfo());
    };

    validator::storage::MessageAttributesFilter filter;
    if (task.args().hasField("severity")) {
        filter.severity = static_cast<validator::Severity>(
            task.args()["severity"].as<int>());
    }
    dualFilterParamInfo("severity", filter.severity);

    if (task.args().hasField("checkId")) {
        filter.checkId = task.args()["checkId"].as<std::string>();
    }
    dualFilterParamInfo("checkId", filter.checkId);

    if (task.args().hasField("description")) {
        filter.description = task.args()["description"].as<std::string>();
    }
    dualFilterParamInfo("description", filter.description);

    if (task.args().hasField("importantRegion")) {
        filter.regionType = task.args()["importantRegion"].as<bool>()
            ? validator::RegionType::Important
            : validator::RegionType::Unimportant;
    }
    dualFilterParamInfo("importantRegion", filter.regionType);

    try {
        common::PoolHolder validationPool(*configXml_, "validation", "grinder");
        common::PoolHolder socialPool(*configXml_, "social", "grinder");
        pgp3utils::PgAdvisoryXactMutex dbCoreLocker(
                corePool.pool(),
                static_cast<int64_t>(common::AdvisoryLockIds::VALIDATION_FEEDBACK_CONVERTER));
        while (!dbCoreLocker.try_lock()) {
            INFO() << "Database is locked. Wait " << POLLING_INTERVAL.count() << " seconds";
            std::this_thread::sleep_for(POLLING_INTERVAL);
        }

        auto validationTxn = validationPool.pool().masterReadOnlyTransaction();
        auto socialTxn = socialPool.pool().masterWriteableTransaction();
        pqxx::transaction_base& coreTxn = dbCoreLocker.writableTxn();

        const auto tasksData = getNewValidationTasksData(
            validationTaskId,
            filter,
            validationTxn.get(),
            coreTxn);

        validationTxn.releaseConnection();

        INFO() << "find " << tasksData.size() << " tasks";
        logger.logInfo() << "find " << tasksData.size() << " tasks";

        const auto countTasks = createFeedbackTasks(
            tasksData,
            socialTxn.get(),
            coreTxn);

        INFO() << "created " << countTasks << " tasks";
        logger.logInfo() << "created " << countTasks << " tasks";

        coreTxn.exec(
            "INSERT INTO service.validation_feedback_converter_result "
            " (task_id, created_tasks_count) "
            " VALUES (" + std::to_string(taskId) + ", "
            + std::to_string(countTasks) + ");");
        coreTxn.commit();
    } catch (const std::exception& ex) {
        logger.logError() << ex.what();
        throw;
    }

} catch (const maps::Exception& ex) {
    ERROR() << "Task ERROR: " << ex;
    throw;
} catch (const std::exception& ex) {
    ERROR() << "Task ERROR: " << ex.what();
    throw;
}

} // namespace maps::wiki::validation_feedback_converter
