#include <maps/wikimap/mapspro/services/tasks_feedback/src/validation_feedback_converter_worker/lib/worker.h>

#include <maps/libs/geolib/include/conversion.h>
#include <yandex/maps/wiki/validator/storage/messages_writer.h>
#include <yandex/maps/wiki/unittest/arcadia.h>
#include <yandex/maps/wiki/social/feedback/gateway_ro.h>
#include <yandex/maps/wiki/social/feedback/task_filter.h>
#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/testing/unittest/gtest.h>


namespace maps::wiki::validation_feedback_converter::tests {

using RandomDbFixture = unittest::ArcadiaDbFixture;

Y_UNIT_TEST_SUITE(validation_feedback_converter)
{

Y_UNIT_TEST_F(test_correct_output, RandomDbFixture)
{
    uint64_t validationTaskId = 10;
    validator::storage::MessagesWriter messageWriter(pool(), validationTaskId);
    std::vector<validator::Message> messages = {
        {
            validator::Severity::Warning,
            "urban_roadnet_parking",
            "parking-territory-is-not-connected-to-parking-lot",
            validator::RegionType::Unimportant,
            geolib3::WKB::toString(geolib3::Polygon2({{4, 10}, {8, 10}, {8, 4}, {4, 4}})),
            {{4858376, 33854798}},
        },
        {
            validator::Severity::Warning,
            "addr_uniqueness",
            "duplicate-address-point-names",
            validator::RegionType::Important,
            geolib3::WKB::toString(geolib3::Point2({4, 3})),
            {{1557448305, 4860310}, {3079407834, 86689258}, {3079408554, 86689375}},
        }
    };
    messageWriter.writeMessagesBatchWithRetries(messages);

    auto txn = pool().masterWriteableTransaction();
    validator::storage::MessageAttributesFilter messagesFilter;

    const auto tasksData = getNewValidationTasksData(validationTaskId, messagesFilter, txn.get(), txn.get());
    createFeedbackTasks(tasksData, txn.get(), txn.get());

    txn = pool().masterWriteableTransaction();
    const social::feedback::TaskFilter tasksFilter;
    auto tasks = social::feedback::GatewayRO(*txn).tasksByFilter(tasksFilter);
    UNIT_ASSERT_VALUES_EQUAL(tasks.size(), 2);
    UNIT_ASSERT_VALUES_EQUAL(tasks[0].position().x(), 6.);
    UNIT_ASSERT_VALUES_EQUAL(tasks[0].position().y(), 7.);
    UNIT_ASSERT_EQUAL(tasks[0].type(), social::feedback::Type::Parking);
    UNIT_ASSERT_EQUAL(tasks[0].workflow(), social::feedback::Workflow::Task);
    UNIT_ASSERT(tasks[0].description().isTranslatable());
    UNIT_ASSERT_STRINGS_EQUAL(
        tasks[0].description().asTranslatable().i18nParams().find(
            "validatorDescription")->second.asTranslatable().i18nKey(),
        "feedback-descriptions:validator-message-parking-territory-is-not-connected-to-parking-lot");
    UNIT_ASSERT_STRINGS_EQUAL(tasks[0].description().asTranslatable().i18nKey(),
        "feedback-descriptions:validation-simple");
    UNIT_ASSERT_STRINGS_EQUAL(tasks[0].source(), "validation-urban-roadnet-parking");
    UNIT_ASSERT(tasks[0].objectId());
    UNIT_ASSERT_VALUES_EQUAL(*tasks[0].objectId(), 4858376);
    UNIT_ASSERT_VALUES_EQUAL(tasks[0].hidden(), true);
}

Y_UNIT_TEST(test_get_point_from_wkb_function)
{
    UNIT_ASSERT(!getPoint2FromWkb(""));

    geolib3::Point2 point(33.33, 55.55);
    UNIT_ASSERT_VALUES_EQUAL(getPoint2FromWkb(
            geolib3::WKB::toString(point))->x(), point.x());
    UNIT_ASSERT_VALUES_EQUAL(getPoint2FromWkb(
            geolib3::WKB::toString(point))->y(), point.y());

    geolib3::Polygon2 polygon({{10., 20.}, {20., 20.}, {20., 10.}, {10., 10.}});
    UNIT_ASSERT_VALUES_EQUAL(getPoint2FromWkb(
            geolib3::WKB::toString(polygon))->x(), 15.);
    UNIT_ASSERT_VALUES_EQUAL(getPoint2FromWkb(
            geolib3::WKB::toString(polygon))->y(), 15.);
}

Y_UNIT_TEST_F(test_not_add_twins, RandomDbFixture)
{
    validator::storage::MessagesWriter messageWriter(pool(), 123);
    std::vector<validator::Message> messages = {
        {
            validator::Severity::Warning,
            "poi_naming",
            "official-name-missing",
            validator::RegionType::Unimportant,
            geolib3::WKB::toString(geolib3::Point2({2, 3})),
            {{4858376, 33854798}},
        },
        {
            validator::Severity::Warning,
            "addr_uniqueness",
            "duplicate-address-point-names",
            validator::RegionType::Important,
            geolib3::WKB::toString(geolib3::Point2({2, 3})),
            {{1557448305, 4560310}, {3079107834, 86689258}, {3079108554, 86689375}},
        },
        {
            validator::Severity::Warning,
            "addr_uniqueness",
            "duplicate-address-point-names",
            validator::RegionType::Important,
            geolib3::WKB::toString(geolib3::Point2({2, 3})),
            {{1557448305, 4560310}, {3079107834, 86689258}, {3079108554, 86689375}},
        }
    };
    messageWriter.writeMessagesBatchWithRetries(messages);
    validator::storage::MessageAttributesFilter messagesFilter;
    auto txn = pool().masterWriteableTransaction();
    auto tasksData = getNewValidationTasksData(123, messagesFilter, txn.get(), txn.get());
    auto countTasks = createFeedbackTasks(tasksData, txn.get(), txn.get());
    UNIT_ASSERT_VALUES_EQUAL(countTasks, 2);
}

Y_UNIT_TEST_F(test_big_input, RandomDbFixture)
{
    auto txn = pool().masterWriteableTransaction();
    validator::storage::MessagesWriter messageWriter(pool(), 123);
    std::vector<validator::Message> messages;
    for (int i = 0; i < 1001; ++i) {
        messages.push_back({
            validator::Severity::Warning,
            "poi_naming",
            "official-name-missing",
            validator::RegionType::Unimportant,
            geolib3::WKB::toString(geolib3::Point2({8, 10})),
            {{4858376, 33854798}}
        });
    }
    validator::storage::MessageAttributesFilter messagesFilter;
    UNIT_CHECK_GENERATED_EXCEPTION(
        getNewValidationTasksData(123, messagesFilter, txn.get(), txn.get()),
        RuntimeError);
}

} // Y_UNIT_TEST_SUITE

} // namespace maps::wiki::validation_feedback_converter::tests
