#include <yandex/maps/wiki/groupedit/session.h>
#include <yandex/maps/wiki/groupedit/object.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/filters.h>
#include <yandex/maps/wiki/common/default_config.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>
#include <yandex/maps/wiki/common/retry_duration.h>
#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/tasks/task_logger.h>
#include <yandex/maps/wiki/tasks/commit_sync.h>
#include <yandex/maps/wiki/tasks/tasks.h>
#include <yandex/maps/wiki/tasks/tool_commands.h>
#include <maps/libs/common/include/profiletimer.h>
#include <maps/libs/cmdline/include/cmdline.h>
#include <maps/tools/grinder/worker/include/api.h>
#include <maps/libs/log8/include/log8.h>
#include <maps/libs/mongo/include/init.h>
#include <maps/libs/xml/include/xml.h>

#include <filesystem>
#include <functional>
#include <vector>

namespace common = maps::wiki::common;
namespace tasks = maps::wiki::tasks;
namespace rf = maps::wiki::revision::filters;
namespace mwg = maps::wiki::groupedit;
namespace rev = maps::wiki::revision;
namespace worker = maps::grinder::worker;
namespace fs = std::filesystem;

using CommitIds = std::vector<maps::wiki::groupedit::TCommitId>;

namespace {

const std::string GROUP_MOD_ATTRS_ACTION = "group-modified-attributes";
const std::string COMMIT_TABLE_NAME = "service.apply_shadow_attributes_commit";

const std::string EDITOR_CONFIG_XPATH = "/config/services/editor/config";
const std::string ATTRIBUTE_XPATH = "/attributes/attribute";
const std::string ID_ATTR = "id";
const std::string TYPE_ATTR = "type";
const std::string BOOLEAN_TYPE = "boolean";
const std::string RD_EL_PREFIX = "rd_el:";
const std::string SHADOW_SUFFIX = "_shadow";
const std::string USE_SHADOW_SUFFIX = "_use_shadow";

constexpr size_t WORKER_CONCURRENCY = 1;

} // anonymous namespace

namespace {

/**
Indicates that the task is not really failed but frozen
*/
class FrozenException: public maps::Exception {
public:
    FrozenException() {}
    explicit FrozenException(std::string what): Exception(std::move(what)) {}
};

struct AttributeInfo {
    std::string originalAttributeId;
    std::string shadowAttributeId;
    std::string useShadowAttributeId;
    bool isBoolean;

    AttributeInfo(std::string originalAttributeId,
                  std::string shadowAttributeId,
                  std::string useShadowAttributeId,
                  bool isBoolean = false)
        : originalAttributeId(std::move(originalAttributeId))
        , shadowAttributeId(std::move(shadowAttributeId))
        , useShadowAttributeId(std::move(useShadowAttributeId))
        , isBoolean(isBoolean)
    {
    }
};

using ShadowAttributes = std::vector<AttributeInfo>;

/**
Searches for the shadow attributes in the attributes.xml file and fills in the map
*/
ShadowAttributes parseShadowAttributes(const std::string& filepath)
{
    ShadowAttributes shadowAttributes;

    auto doc = maps::xml3::Doc::fromFile(filepath);

    auto attributes = doc.nodes(ATTRIBUTE_XPATH, true);
    for (size_t i = 0; i < attributes.size(); ++i) {
        auto attribute = attributes[i];
        auto attributeId = attribute.attr<std::string>(ID_ATTR);

        if (attributeId.starts_with(RD_EL_PREFIX) && attributeId.ends_with(SHADOW_SUFFIX)) {
            auto originalAttributeId =
                    attributeId.substr(0, attributeId.length() - SHADOW_SUFFIX.length());
            auto useShadowAttributeId = originalAttributeId + USE_SHADOW_SUFFIX;
            auto type = attribute.attr<std::string>(TYPE_ATTR);

            INFO() << "Shadow attribute found: original " << originalAttributeId
                   << " shadow " << attributeId
                   << " use shadow " << useShadowAttributeId
                   << " type " << type;

            shadowAttributes.emplace_back(
                std::move(originalAttributeId), std::move(attributeId),
                std::move(useShadowAttributeId), type == BOOLEAN_TYPE);
        }
    }

    if (shadowAttributes.empty()) {
        WARN() << "No shadow attributes found";
    }

    return shadowAttributes;
}

CommitIds applyShadowAttributes(
    const ShadowAttributes& shadowAttributes,
    uint64_t taskId,
    mwg::TUserId uid,
    rev::DBID branchId,
    maps::pgpool3::Pool& pgPool,
    tasks::TaskPgLogger& logger)
{
    ProfileTimer editTimer;

    CommitIds commitIds;

    auto txn = pgPool.masterWriteableTransaction();

    try {
        rf::ProxyFilterExpr filter = rf::False();
        for (const auto& attrInfo : shadowAttributes) {
            filter |= rf::Attr(attrInfo.useShadowAttributeId).defined();
        }
        filter &= rf::ObjRevAttr::isNotRelation() && rf::Geom::defined();

        rev::BranchManager branchManager(*txn);
        auto branch = branchManager.load(branchId);
        REQUIRE(branch.type() == rev::BranchType::Stable, "Worker works only with stable branch");
        auto prevState = branch.state();
        REQUIRE(prevState != rev::BranchState::Progress, "Branch state should not be progress");
        auto needRevertState = branch.setState(*txn, rev::BranchState::Normal);

        mwg::Session session(*txn, branchId);
        commitIds = session.query(std::move(filter)).update(
            GROUP_MOD_ATTRS_ACTION, uid, [&](mwg::Object& obj)
            {
                for (const auto& attrInfo : shadowAttributes) {
                    auto useShadowAttr = obj.attribute(attrInfo.useShadowAttributeId);
                    if (useShadowAttr) {
                        auto shadowAttr = obj.attribute(attrInfo.shadowAttributeId);
                        if (shadowAttr) {
                            obj.setAttribute(attrInfo.originalAttributeId, *shadowAttr);
                        }
                        else if (attrInfo.isBoolean) {
                            obj.removeAttribute(attrInfo.originalAttributeId);
                        }
                    }
                }
            });

        if (needRevertState) {
            branch.setState(*txn, prevState);
        }
    }
    catch (...) {
        ERROR() << "Shadow attributes application failed";
        logger.logError() << "Task failed";
        txn->abort();
        throw;
    }

    INFO()
        << "Wrote " << commitIds.size() << " commits"
        << ", shadow attributes application completed successfully in "
        << editTimer.getElapsedTime() << " s.";

    if (commitIds.empty()) {
        INFO() << "Nothing to update. Task finished.";
        logger.logInfo() << "Task finished";
        return {};
    }

    tasks::CommitRecordGateway commitGtw(txn, COMMIT_TABLE_NAME);
    commitGtw.addCommits(taskId, commitIds, tasks::CommitStatus::NotSynced);
    txn->commit();

    INFO() << "Shadow attribute application commit done";
    logger.logInfo() << "Shadow attribute application commit done";

    return commitIds;
}

void refreshViews(
    const CommitIds& commitIds,
    uint64_t taskId,
    rev::DBID branchId,
    maps::pgpool3::Pool& pgPool,
    tasks::TaskPgLogger& logger,
    const std::string& configPath)
{
    tasks::refreshViews(commitIds, branchId, configPath);

    common::retryDuration([&] {
        auto txn = pgPool.masterWriteableTransaction();
        tasks::CommitRecordGateway commitGtw(txn, COMMIT_TABLE_NAME);
        commitGtw.updateStatus(taskId, commitIds, tasks::CommitStatus::Synced);
        txn->commit();
    });

    INFO() << "Refreshing view done";
    logger.logInfo() << "Refreshing view done";
}

void runShadowAttributesApplication(const common::ExtendedXmlDoc& configDoc,
                                    const worker::Task& task,
                                    const ShadowAttributes& shadowAttributes)
{
    const auto& args = task.args();
    auto taskId = args["taskId"].as<uint64_t>();
    auto uid = args["uid"].as<mwg::TUserId>();
    auto branchId = args["branchId"].as<rev::DBID>();

    INFO() << "Received task: " << taskId << ". "
           << "Grinder task id: " << task.id();

    common::PoolHolder dbHolder(configDoc, "core", "grinder");
    auto& pgPool = dbHolder.pool();

    tasks::TaskPgLogger logger(pgPool, taskId);
    logger.logInfo() << "Task started. Grinder task id: " << task.id();

    try {
        bool frozen = common::retryDuration([&] {
            auto txn = pgPool.masterReadOnlyTransaction();
            return tasks::isFrozen(*txn, taskId);
        });
        // If the task is 'frozen', it will resume at view refresh stage
        logger.logInfo() << "Task " << (frozen ? "resumed" : "started")
                << ". Grinder task id: " << task.id();

        CommitIds commitIds;

        if (frozen) {
            commitIds = common::retryDuration([&] {
                auto txn = pgPool.masterReadOnlyTransaction();
                tasks::CommitRecordGateway commitGtw(txn, COMMIT_TABLE_NAME);
                return commitGtw.getCommitIds(taskId);
            });
        } else {
            commitIds = applyShadowAttributes(shadowAttributes, taskId, uid, branchId, pgPool, logger);
        }

        if (!commitIds.empty()) {
            auto branch = common::retryDuration([&] {
                auto txn = pgPool.masterReadOnlyTransaction();
                rev::BranchManager branchManager(*txn);
                return branchManager.load(branchId);
            });
            if (branch.state() == rev::BranchState::Normal) {
                refreshViews(commitIds, taskId, branchId, pgPool, logger, configDoc.sourcePaths()[0]);
            } else {
                common::retryDuration([&] {
                    auto txn = pgPool.masterWriteableTransaction();
                    tasks::freezeTask(*txn, taskId);
                    txn->commit();
                });
                throw FrozenException(); //Stop task execution
            }
        }

        if (frozen) {
            common::retryDuration([&] {
                auto txn = pgPool.masterWriteableTransaction();
                tasks::unfreezeTask(*txn, taskId);
                txn->commit();
            });
        }
    }
    catch (const FrozenException&) {
        INFO() << "Task " << taskId << " frozen.";
        logger.logInfo() << "Task frozen.";
        throw;
    }
    catch (const maps::Exception& e) {
        ERROR() << "Task " << taskId << " failed: " << e;
        logger.logError() << "Task failed.";
        throw;
    }
    catch (const std::exception& e) {
        ERROR() << "Task " << taskId << " failed: " << e.what();
        logger.logError() << "Task failed.";
        throw;
    }
    logger.logInfo() << "Task finished";
}

} // anonymous namespace

int main(int argc, char* argv[]) try {
    maps::cmdline::Parser parser;
    auto workerConfig
        = parser.string("config").help("path to worker configuration");
    auto syslogTag = parser.string("syslog-tag").help(
        "redirect log output to syslog with given tag");
    auto grinderConfig = parser.string("grinder-config")
                             .help("path to grinder configuration file");

    parser.parse(argc, argv);

    if (syslogTag.defined()) {
        maps::log8::setBackend(maps::log8::toSyslog(syslogTag));
    }

    maps::mongo::init();

    std::unique_ptr<common::ExtendedXmlDoc> configDocPtr;
    if (workerConfig.defined()) {
        configDocPtr.reset(new common::ExtendedXmlDoc(workerConfig));
    } else {
        configDocPtr = common::loadDefaultConfig();
    }

    auto editorConfigPath = configDocPtr->get<std::string>(EDITOR_CONFIG_XPATH);
    auto editorAttributesConfigPath = fs::path(editorConfigPath).parent_path()  / fs::path("attributes.xml");

    ShadowAttributes shadowAttributes =
            parseShadowAttributes(editorAttributesConfigPath.string());

    worker::Options workerOpts;

    if (grinderConfig.defined()) {
        workerOpts.setConfigLocation(grinderConfig);
    }

    workerOpts.on("apply_shadow_attributes",
                [&](const worker::Task& task) {
                    runShadowAttributesApplication(*configDocPtr, task, shadowAttributes);
                })
            .setConcurrencyLevel(WORKER_CONCURRENCY);

    worker::run(std::move(workerOpts));

    return EXIT_SUCCESS;
} catch (const maps::Exception& e) {
    ERROR() << "Worker failed: " << e;
    return EXIT_FAILURE;
} catch (const std::exception& e) {
    ERROR() << "Worker failed: " << e.what();
    return EXIT_FAILURE;
}
