#include "add.h"
#include "json_helpers.h"

#include <yandex/maps/wiki/common/batch.h>
#include <yandex/maps/wiki/common/robot.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>
#include <yandex/maps/wiki/revisionapi/revisionapi.h>
#include <maps/libs/log8/include/log8.h>

#include <fstream>

namespace fs = std::filesystem;
namespace rev = maps::wiki::revision;

namespace maps {
namespace wiki {
namespace importer {

namespace {

const std::string JSON_FILE_NAME = "json";

CommitIds json2revision(
    const fs::path& jsonPath,
    const DbIdToFeatureIdMap& idMap,
    TaskParams& params,
    MessageReporter& messageReporter)
{
    std::ifstream input;
    input.open(jsonPath.string().c_str());
    REQUIRE(!input.fail(), "Can't open '" << jsonPath << "' for reading");

    const auto mode = revisionapi::IdMode::PreserveId;
    revisionapi::RevisionAPI revApi(params.corePool);

    CommitIds commitIds;
    std::map<std::string, std::string> jsonIdToError;

    try {
        commitIds =  revApi.importData(common::ROBOT_UID, mode, input, jsonIdToError, params.mainTxn, COMMIT_BATCH_SIZE, 0);
    } catch (const maps::Exception& e) {
        messageReporter.error() << e.what();
    }

    for (const auto& kv : jsonIdToError) {
        const auto& jsonId = kv.first;
        const auto& error = kv.second;
        const auto& id = idMap.at(jsonId);
        messageReporter.error(id) << error;
    }

    return commitIds;
}

} // namespace

CommitIds addObjects(
    const Objects& objects,
    const DbIdToFeatureIdMap& idMap,
    TaskParams& params,
    tasks::TaskPgLogger& logger,
    MessageReporter& messageReporter)
{
    if (objects.empty()) {
        return {};
    }

    auto jsonPath = params.workDir / JSON_FILE_NAME;

    runLoggable([&]
        {
            std::ofstream output(jsonPath.string().c_str());
            objects2json(objects, output, params.globalAttrs, messageReporter);
        },
        logger, "Generate json from objects");

    if (messageReporter.hasErrors()) {
        return {};
    }

    CommitIds commits;

    runLoggable([&]{ commits = json2revision(jsonPath, idMap, params, messageReporter); },
        logger, "Load json to revision");

    return commits;
}

CommitIds addRelations(
    const std::vector<RelationToAdd>& relations,
    TaskParams& params)
{
    if (relations.empty()) {
        return {};
    }

    rev::RevisionsGateway gateway(*params.mainTxn);

    CommitIds commitIds;

    common::applyBatchOp<std::vector<RelationToAdd>>(relations, COMMIT_BATCH_SIZE, [&](const std::vector<RelationToAdd>& batch)
    {
        std::vector<revision::RevisionsGateway::NewRevisionData> preparedRelations;
        for (const auto& relation : batch)
        {
            revision::ObjectRevision::Data data;
            data.attributes = relation.attributes;
            data.relationData = revision::RelationData(relation.masterId, relation.slaveId);
            preparedRelations.emplace_back(gateway.acquireObjectId(), std::move(data));
        }
        commitIds.push_back(gateway.createCommit(preparedRelations, common::ROBOT_UID, params.globalAttrs).id());
    });

    return commitIds;
}

} // importer
} // wiki
} // maps
