#include "common.h"

#include <maps/libs/log8/include/log8.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/distance.h>

#include <boost/algorithm/string/predicate.hpp>

namespace maps {
namespace wiki {
namespace importer {

void runLoggable(const LoggableAction& fun,
                 tasks::TaskPgLogger& logger,
                 const std::string& label)
{
    logger.logInfo() << label << " started";
    INFO() << label << " started";
    fun();
    logger.logInfo() << label << " finished";
    INFO() << label << " finished";
}

std::string extractCategory(const StringMap& attributes)
{
    for (const auto& pair : attributes) {
        if (boost::algorithm::starts_with(pair.first, "cat:")) {
            return pair.first.substr(4);
        }
    }
    return {};
}

std::string extractRole(const StringMap& attributes)
{
    for (const auto& pair : attributes) {
        if (pair.first == "rel:role") {
            return pair.second;
        }
    }
    return {};
}

std::string extractLang(const StringMap& attributes)
{
    for (const auto& pair : attributes) {
        if (boost::algorithm::ends_with(pair.first, ":lang")) {
            return pair.second;
        }
    }
    return {};
}

std::optional<LevelKind> extractLevelKind(const StringMap& attributes)
{
    for (const auto& pair : attributes) {
        if (pair.first == "ad:level_kind") {
            return boost::lexical_cast<LevelKind>(pair.second);
        }
    }
    return std::nullopt;
}

std::string getWkb(const OGRGeometry* geometry)
{
    ASSERT(geometry);
    std::string geomWkb;
    geomWkb.resize(geometry->WkbSize());
    auto res = geometry->exportToWkb(
        OGRwkbByteOrder::wkbNDR,
        reinterpret_cast<unsigned char*>(&geomWkb[0]));
    REQUIRE(res == OGRERR_NONE, "Failed to export wkb: result " << static_cast<int>(res));
    REQUIRE(!geomWkb.empty(), "Failed to export wkb: empty string");
    return geomWkb;
}

double getRealLength(const OGRGeometry* geometry)
{
    ASSERT(geometry);
    REQUIRE(geometry->getGeometryType() == wkbLineString,
        "Wrong geometry type " << OGRGeometryTypeToName(geometry->getGeometryType()));

    const auto* line = static_cast<const OGRLineString*>(geometry);
    REQUIRE(line->getNumPoints() >= 2, "Wrong points count " << line->getNumPoints());

    geolib3::Polyline2 polyline;
    OGRPoint point;
    for (int index = 0; index < line->getNumPoints(); index++) {
        line->getPoint(index, &point);
        polyline.add(geolib3::Point2(point.getX(), point.getY()));
    }
    return geolib3::geoLength(polyline);
}

} // namespace importer
} // namespace wiki
} // namespace maps
