#include "dbhelpers.h"

#include <maps/libs/log8/include/log8.h>

#include <boost/algorithm/string/predicate.hpp>

namespace rev = maps::wiki::revision;

namespace maps {
namespace wiki {
namespace importer {

namespace {

ObjectIds collectExistingObjectIds(const ObjectsCache& cache)
{
    ObjectIds objectIds;
    for (const auto& object : cache.objects()) {
        if (object->state() == ObjectState::Existing) {
            objectIds.insert(object->dbId());
        }
    }
    return objectIds;
}

} // namespace

DbObject::DbObject(const revision::ObjectRevision& revision)
    : revisionId_(revision.id())
{
    const auto& revData = revision.data();
    REQUIRE(!revData.deleted,
        "Object " << revisionId_.objectId() << " is deleted");
    REQUIRE(revData.attributes,
        "Object " << revisionId_.objectId() << " has no attributes");

    categoryId_ = extractCategory(*revData.attributes);

    for (const auto& pair : *revData.attributes) {
        if (boost::algorithm::starts_with(pair.first, "cat:")) {
            continue;
        }
        attributesMap_.emplace(pair.first, pair.second);
    }
}

DbRelation::DbRelation(
        RelationType relationType,
        const revision::ObjectRevision& revision)
    : revisionId_(revision.id())
{
    const auto& revData = revision.data();
    REQUIRE(revData.attributes, "Relation object " << revisionId_.objectId() << " has no attributes");
    REQUIRE(revData.relationData, "Relation object " << revisionId_.objectId() << " has no relationData");

    roleId_ = extractRole(*revData.attributes);

    const auto& relationData = *revData.relationData;

    mainObjectId_ = relationType == RelationType::Master
        ? relationData.slaveObjectId()
        : relationData.masterObjectId();

    relatedObjectId_ = relationType == RelationType::Master
        ? relationData.masterObjectId()
        : relationData.slaveObjectId();
}

RevisionsFacade::RevisionsFacade(TaskParams& params)
    : editorConfig_(params.editorConfig)
    , readTxn_(params.corePool.masterReadOnlyTransaction())
    , gateway_(*readTxn_)
    , snapshot_(gateway_.snapshot(gateway_.headCommitId()))
{
}

const DbObject& RevisionsFacade::object(TObjectId objectId) const
{
    auto it = objects_.find(objectId);
    REQUIRE(it != objects_.end(), "Object " << objectId << " not found");
    return it->second;
}

void RevisionsFacade::loadObjectFromDb(
    const ObjectsCache& cache,
    MessageReporter& messageReporter)
{
    if (cache.objects().empty()) {
        return;
    }

    auto revisions = snapshot_.objectRevisions(collectExistingObjectIds(cache));

    auto draftCommitIds = loadDraftCommits();

    auto processObject = [&](const ObjectPtr& objectInFile)
    {
        auto dbId = objectInFile->dbId();
        auto revisionIt = revisions.find(dbId);
        REQUIRE(revisionIt != revisions.end(), "Object " << dbId << " does not exist in the DB");

        DbObject object(revisionIt->second);

        if (objectInFile->category().id() != object.categoryId()) {
            ASSERT(editorConfig_.hasCategory(object.categoryId()));
            REQUIRE(!objectInFile->category().kind().empty() &&
                objectInFile->category().kind() == editorConfig_.category(object.categoryId()).kind(),
                "Object " << dbId << " has unexpected category "
                << objectInFile->category().id() << " of kind " << objectInFile->category().kind()
                << ", category in db "
                << object.categoryId() << " of kind " << editorConfig_.category(object.categoryId()).kind());
        }

        REQUIRE(!draftCommitIds.count(revisionIt->second.id().commitId()),
            "Object " << dbId << " is in a draft state");

        objectIds_.emplace(dbId);
        objects_.emplace(dbId, std::move(object));
    };

    for (const auto& objectInFile : cache.objects()) {
        if (objectInFile->state() != ObjectState::Existing) {
            continue;
        }
        try {
            processObject(objectInFile);
        } catch (const maps::Exception& e) {
            messageReporter.warning() << e.what();
            objectInFile->setSkipped(true);
        }
    }

    loadRelations(RelationType::Slave, messageReporter);
    loadRelations(RelationType::Master, messageReporter);
}

std::set<revision::DBID> RevisionsFacade::loadDraftCommits()
{
    std::set<revision::DBID> draftCommitIds;

    auto commits = revision::Commit::load(
        *readTxn_,
        revision::filters::CommitAttr::isDraft());
    for (const auto& commit : commits) {
        draftCommitIds.insert(commit.id());
    }

    return draftCommitIds;
}

void RevisionsFacade::loadRelations(
    RelationType relationType,
    MessageReporter& messageReporter)
{
    ObjectIds relatedObjectIds;
    std::map<TObjectId, std::vector<DbRelation>> relatedObjectIdToRelation;

    auto relationRevs = relationType == RelationType::Master
        ? snapshot_.loadMasterRelations(objectIds_)
        : snapshot_.loadSlaveRelations(objectIds_);
    for (const auto& rev : relationRevs) {
        try {
            DbRelation relation(relationType, rev);
            relatedObjectIds.emplace(relation.relatedObjectId());
            relatedObjectIdToRelation[relation.relatedObjectId()].emplace_back(std::move(relation));
        } catch (const maps::Exception& e) {
            messageReporter.error() << "Error loading relation " << rev.id() << " " << e.what();
        }
    }

    if (messageReporter.hasErrors()) {
        return;
    }

    auto revisions = snapshot_.objectRevisions(relatedObjectIds);
    for (const auto& pair : revisions) {
        auto relatedObjectId = pair.first;
        const auto& rev = pair.second;

        try {
            auto& relations = relatedObjectIdToRelation.at(relatedObjectId);
            for (auto& relation : relations) {
                relation.setRelatedObject(DbObject(rev));

                auto& mainObject = objects_.at(relation.mainObjectId());
                if (relationType == RelationType::Master) {
                    mainObject.addMasterRelation(std::move(relation));
                } else {
                    mainObject.addSlaveRelation(std::move(relation));
                }
            }
        } catch (const maps::Exception& e) {
            messageReporter.error() << e.what();
        }
    }
}

} // importer
} // wiki
} // maps
